/*
 * Java
 *
 * Copyright 2015-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util;

import java.io.IOException;
import java.io.InputStream;

import com.microej.kf.util.message.KfUtilMessage;

import ej.kf.Feature;
import ej.kf.Kernel;
import ej.kf.Proxy;
import ej.util.message.Level;

/**
 * A wrapper on an InputStream
 */
public class InputStreamProxy extends InputStream {

	private final Proxy<InputStream> inputStreamRef;

	public InputStreamProxy(Proxy<InputStream> inputStreamRef) {
		this.inputStreamRef = inputStreamRef;
	}

	@Override
	public int available() throws IOException {
		Kernel.enter();
		try {
			return this.inputStreamRef.getReference().available();
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.INPUT_STREAM_ERROR, t);
		}
		throw new IOException();
		// Kernel.exit();

	}

	@Override
	public void close() throws IOException {
		Kernel.enter();
		try {
			this.inputStreamRef.getReference().close();
			return;
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.INPUT_STREAM_ERROR, t);
		}
		throw new IOException();
		// Kernel.exit();
	}

	@Override
	public int read() throws IOException {
		Kernel.enter();
		try {
			return this.inputStreamRef.getReference().read();
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.INPUT_STREAM_ERROR, t);
		}
		throw new IOException();
		// Kernel.exit();
	}

	@Override
	public long skip(long n) throws IOException {
		Kernel.enter();
		try {
			return this.inputStreamRef.getReference().skip(n);
		} catch (Throwable t) {
			KfUtilMessage.LOGGER.log(Level.INFO, KfUtilMessage.CATEGORY, KfUtilMessage.INPUT_STREAM_ERROR, t);
		}
		throw new IOException();
		// Kernel.exit();
	}

	@Override
	public int read(byte[] b, final int off, final int len) throws IOException {
		Kernel.enter();
		final InputStream target = this.inputStreamRef.getReference();
		if (target == null) {
			throw new IOException();
		}
		Feature owner = (Feature) Kernel.getOwner(target);

		ReadResult r = new ReadResult() {

			@Override
			public byte[] runWithResult() {
				byte[] bytes = new byte[len];
				try {
					this.nbRead = target.read(bytes, 0, len);
				} catch (IOException e) {
					return null;
				}
				return bytes;
			}
		};
		Kernel.runUnderContext(owner, r);

		int bytesRead = r.nbRead;
		if (bytesRead == -1) {
			return -1;
		} else {
			byte[] bytes = r.getResult();
			if (bytes != null) {
				// transfer bytes
				System.arraycopy(bytes, 0, b, off, bytesRead);
				return bytesRead;
			} else {
				Kernel.exit();
				throw new IOException();
			}
		}
	}

}

abstract class ReadResult extends RunnableWithResult<byte[]> {

	public int nbRead;

	@Override
	protected abstract byte[] runWithResult();

}
