/*
 * Java
 *
 * Copyright 2026 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.MicroUIException;
import ej.microui.display.BufferedImage;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BufferedVectorImage;
import ej.microvg.ResourceVectorImage;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorImage;
import ej.sni.NativeResource;

/**
 * Tests the release of native resources.
 */
@SuppressWarnings("nls")
public class TestResource {

	private static final long UNSUPPORTED_RESOURCE = -1;
	private static final String MASCOT_INTERNAL = "/com/microej/microvg/test/test_resource_mascot.xml";
	private static final String MASCOT_EXTERNAL = "/com/microej/microvg/test/test_resource_mascot_external.xml";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests the automatic closing of {@link BufferedImage}.
	 */
	@Test
	public static void filterImageFromRaw() {

		final VectorImage raw = VectorImage.getImage(MASCOT_INTERNAL);
		final float[] colorMatrix = new float[] { 1, 0, 0, 0, 0, // red
				0, 1, 0, 0, 0, // green
				0, 0, 1, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};

		testResource(new ResourceBuilder() {
			@Override
			public Closeable create() {
				try {
					return raw.filterImage(colorMatrix);
				} catch (VectorGraphicsException e) {
					Assert.assertEquals("OOM expected", VectorGraphicsException.OUT_OF_MEMORY, e.getErrorCode());
				}
				throw new OutOfMemoryError();
			}

			@Override
			public long getCloseFunction() {
				return getSNICloseFunctionForFilteredImage();
			}

			@Override
			public String getDescription() {
				return "FilteredVectorImage";
			}
		});
	}

	/**
	 * @return the SNI close function for the filtered images.
	 */
	private static native long getSNICloseFunctionForFilteredImage();

	/**
	 * Tests the automatic closing of {@link BufferedVectorImage}.
	 */
	@Test
	public static void bufferedVectorImage() {
		testResource(new ResourceBuilder() {
			@Override
			public Closeable create() {
				BufferedVectorImage i = new BufferedVectorImage(100, 100);
				GraphicsContext gc = i.getGraphicsContext();
				int j = 0;
				try {
					// add elements in the BVI to allocate in the image heap
					for (; j < 100; j++) {
						Painter.fillRectangle(gc, 0, 0, 99, 99);
						gc.checkDrawingLogFlags();
					}
					return i;
				} catch (MicroUIException e) {
					switch (e.getErrorCode()) {
					case MicroUIException.IMAGE_OUT_OF_MEMORY:
						// OOM during image creation
						break;
					case MicroUIException.DRAWING_ERROR:
						// OOM during drawing: consider global OOM
						i.close();
						break;
					default:
						Assert.fail("OOM expected");
						break;
					}
				}
				throw new OutOfMemoryError();
			}

			@Override
			public long getCloseFunction() {
				return getSNICloseFunctionForBufferedVectorImage();
			}

			@Override
			public String getDescription() {
				return "BufferedVectorImage";
			}

			@Override
			public int getSNIResourcesNumber() {
				return 2; // bi and bvi
			}

			@Override
			public int getSNIResourcesIndex() {
				return 0; // first resource
			}
		});
	}

	/**
	 * @return the SNI close function for the {@link BufferedVectorImage}
	 */
	private static native long getSNICloseFunctionForBufferedVectorImage();

	/**
	 * Tests the automatic closing of an external image.
	 */
	@Test
	public static void externalImageFromRaw() {
		testResource(new ResourceBuilder() {
			@Override
			public Closeable create() {
				try {
					return ResourceVectorImage.loadImage(MASCOT_EXTERNAL);
				} catch (VectorGraphicsException e) {
					Assert.assertEquals("OOM expected", VectorGraphicsException.OUT_OF_MEMORY, e.getErrorCode());
				}
				throw new OutOfMemoryError();
			}

			@Override
			public long getCloseFunction() {
				return getSNICloseFunctionForExternalImage();
			}

			@Override
			public String getDescription() {
				return "ExternalVectorImage";
			}
		});
	}

	/**
	 * @return the SNI close function for the external images.
	 */
	private static native long getSNICloseFunctionForExternalImage();

	/**
	 * Tests the automatic closing of fonts.
	 */
	@Test
	public static void font() {
		testResource(new ResourceBuilder() {
			@Override
			public Closeable create() {
				try {
					return loadFont("firstfont");
				} catch (VectorGraphicsException e) {
					Assert.assertEquals("OOM expected", VectorGraphicsException.FONT_INVALID, e.getErrorCode());
				}
				throw new OutOfMemoryError();
			}

			@Override
			public long getCloseFunction() {
				return getSNICloseFunctionForFont();
			}

			@Override
			public String getDescription() {
				return "VectorFont";
			}
		});
	}

	private static VectorFont loadFont(String name) {
		return VectorFont.loadFont("/fonts/" + name + ".ttf");
	}

	/**
	 * @return the SNI close function for the fonts.
	 */
	private static native long getSNICloseFunctionForFont();

	/**
	 * Generic function to test the automatic closing of a resource.
	 */
	private static void testResource(ResourceBuilder builder) {

		if (UNSUPPORTED_RESOURCE == builder.getCloseFunction()) {
			// kind of resource not supported (buffered vector image, external image, etc.)
			return;
		}

		List<NativeResource> before = getNativeResources();

		int nbObjects = 0;
		nbObjects = oom(builder);
		Assert.assertEquals("[1] No more native resource", before.size(),
				NativeResource.getRegisteredNativeResourcesCount());

		// no assignment to not keep a reference on the object
		closeOnGC(before, builder);

		// only one more resource except when an object holds other objects (ex: a bi in a bvi)
		Assert.assertEquals("[2] One more native resource", before.size() + builder.getSNIResourcesNumber(),
				NativeResource.getRegisteredNativeResourcesCount());

		// cannot create one more object
		Assert.assertEquals("[3] One less object", nbObjects - 1, oom(builder));

		System.gc();

		// can only test the main resource associated to the object (and not the sub resources like bi in bvi)
		Assert.assertEquals("[4] No more native resource", before.size() + builder.getSNIResourcesNumber() - 1,
				NativeResource.getRegisteredNativeResourcesCount());

		// warning about resource of sub objects
		Assert.assertEquals("[5] One more object", nbObjects, oom(builder));
	}

	static abstract class ResourceBuilder {

		/**
		 * Creates a new element. Implementation must check the condition of its own out of memory error.
		 *
		 * @return the new element.
		 * @throws OutOfMemoryError
		 *             when no more element can be created
		 */
		abstract Closeable create();

		/**
		 * Gets the identifier of the native close function.
		 *
		 * @return a 64-bit identifier
		 */
		abstract long getCloseFunction();

		/**
		 * Gets the expected description of the object.
		 *
		 * @return a string
		 */
		abstract String getDescription();

		/**
		 * Gets the expected number of SNI resources registered for one application resource.
		 *
		 * @return 1 by default
		 */
		int getSNIResourcesNumber() {
			return 1;
		}

		/**
		 * Gets the index of the SNI resource to test (a value between 0 and {@link #getSNIResourcesNumber()} - 1).
		 *
		 * Beware about the order: see {@link #getNativeResources()}).
		 *
		 * @return 0 by default
		 */
		int getSNIResourcesIndex() {
			return 0;
		}
	}

	/**
	 * Creates a list on current iteration of native resources to be sure to not iterate on this mutable list later.
	 *
	 * Warning: The first element of {@link NativeResource#listRegisteredNativeResources()} (iterable) is the last entry
	 * added.
	 *
	 * @return a list of native resources
	 */
	private static List<NativeResource> getNativeResources() {
		List<NativeResource> resourceSet = new ArrayList<>();
		for (NativeResource resource : NativeResource.listRegisteredNativeResources()) {
			resourceSet.add(resource);
		}
		Assert.assertEquals("Invalid number of resouces", resourceSet.size(),
				NativeResource.getRegisteredNativeResourcesCount());
		return resourceSet;
	}

	/**
	 * Registers a new object to be closed on GC.
	 *
	 * @param before
	 *            the set of resource before creating the object
	 * @param builder
	 *            the builder that creates the object
	 */
	private static void closeOnGC(List<NativeResource> before, ResourceBuilder builder) {
		Object o = builder.create();
		NativeResource resource = getNewResource(before, builder);
		NativeResource.closeOnGC(resource.getId(), builder.getCloseFunction(), o);
	}

	/**
	 * Gets the latest resource added in the native resource list. Only one resource must be added since the creation of
	 * "before".
	 *
	 * @param before
	 *            the list of native resources before creating the new native resource
	 * @param builder
	 *            the builder that creates the object
	 * @return the last created native resource
	 */
	private static NativeResource getNewResource(List<NativeResource> before, ResourceBuilder builder) {
		List<NativeResource> current = getNativeResources();
		Assert.assertEquals("Only one more resource", before.size() + builder.getSNIResourcesNumber(), current.size());
		NativeResource ret = getNativeResources().get(builder.getSNIResourcesIndex());
		Assert.assertEquals("Invalid description", builder.getDescription(), ret.getDescription());
		return ret;
	}

	/**
	 * Counts the number of objects that can be created in the before an OOM.
	 */
	private static int oom(ResourceBuilder builder) {
		List<Closeable> list = new ArrayList<>();
		try {
			while (true) {
				list.add(builder.create());
			}
		} catch (OutOfMemoryError e) {
			// cannot add more elements
		}

		System.out.println("list before oom: " + list.size());
		for (Closeable object : list) {
			try {
				object.close();
			} catch (IOException e) {
				Assert.fail("Cannot close the resource");
			}
		}
		return list.size();
	}
}
