/*
 * Java
 *
 * Copyright 2022-2026 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests kerning.
 */
@SuppressWarnings("nls")
public class TestFontEmoji {

	private static final float DELTA = 0.01f;
	private static final int PADDING = 1;

	// "😀"
	// = ((0xd83d - MIN_HIGH_SURROGATE) << 10) + (0xde00 - MIN_LOW_SURROGATE) + MIN_SUPPLEMENTARY_CODE_POINT
	// = ((0xd83d - 0xd800) << 10) + (0xde00 - 0xdc00) + 0x10000
	// = 0x1f600
	private static final String STRING_EMOJI_GRINNING = new String(new char[] { 0xd83d, 0xde00 });

	private static final String STRING_SPUAB = "\uDBFF\uDFFD";

	private static VectorFont font;
	private static VectorFont emoji;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
		font = VectorFont.loadFont("/fonts/firstfont.ttf");
		emoji = VectorFont.loadFont("/fonts/seguiemj_reduced.ttf");
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		font.close();
		emoji.close();
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing a string with colored emoji.
	 */
	@Test
	public void testColoredEmojiDrawStringColored() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getEmojiFont();
		String string = STRING_EMOJI_GRINNING;
		int fontSize = 100;
		int baselinePosition = (int) font.getBaselinePosition(fontSize);
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		g.setColor(Colors.BLUE);
		float stringWidth = TestUtilities.measureCharWidth(string, font, fontSize);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		// Shift x to start of bbox
		x -= (int) (stringWidth * 0.11f);

		// outline
		float targetX = x + stringWidth * 0.22f;
		float targetY = y + baselinePosition;
		TestUtilities.check("outline", (int) targetX, (int) targetY, g, TestUtilities.BACKGROUND_COLOR);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
		}

		// mouth outline
		targetX = x + stringWidth / 2;
		targetY = y + baselinePosition + fontSize * 0.015f;
		TestUtilities.check("mouth outline", (int) targetX, (int) targetY, g, TestUtilities.BACKGROUND_COLOR);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
		}

		// tongue
		targetX = x + stringWidth / 2;
		targetY = y + baselinePosition - fontSize * 0.015f;
		TestUtilities.check("tongue", (int) targetX, (int) targetY, g, 0xf03a17);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
		}

		// teeth
		targetX = x + stringWidth / 2;
		targetY = y + baselinePosition - fontSize * 0.17f;
		TestUtilities.check("teeth", (int) targetX, (int) targetY, g, Colors.WHITE);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
		}

		// skin
		targetX = x + stringWidth / 2;
		targetY = y + baselinePosition + fontSize * 0.1f;
		TestUtilities.check("head", (int) targetX, (int) targetY, g, 0xffc83d);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
		}

		// eye
		targetX = x + stringWidth * 0.36f;
		targetY = y + baselinePosition - fontSize * 0.44f;
		TestUtilities.check("eye", (int) targetX, (int) targetY, g, TestUtilities.BACKGROUND_COLOR);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
		}

		// outside
		targetX = x + stringWidth * 0.1f;
		targetY = y + baselinePosition;
		TestUtilities.check("outside left", (int) targetX, (int) targetY, g, TestUtilities.BACKGROUND_COLOR);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
		}

		targetX = x + stringWidth * 0.9f;
		TestUtilities.check("outside right", (int) targetX, (int) targetY, g, TestUtilities.BACKGROUND_COLOR);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, (int) targetX, (int) targetY, 2, 2);
			display.flush();
		}
	}

	// /**
	// * Test loading a string with colored emoji.
	// *
	// */
	// @Test
	// public void testColoredEmojiCanLoadFont() {
	// boolean canLoadFont = VectorFont.canLoadFont("/fonts/seguiemj_reduced.ttf");
	// Assert.assertTrue("can load font with colored emoji", canLoadFont);
	// }

	/**
	 * Test loading a string with colored emoji.
	 *
	 */
	@Test
	public void testColoredEmojiLoadFont() {
		int errorCode = 0;
		try {
			VectorFont.loadFont("/fonts/seguiemj_reduced.ttf").close();
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("load font with colored emoji", 0, errorCode);
	}

	/**
	 * Test the glyph width of a colored emoji.
	 *
	 */
	@Test
	public void testColoredEmojiStringWidth() {
		VectorFont font = getEmojiFont();
		String string = STRING_EMOJI_GRINNING;
		int fontSize = 100;
		float stringWidth = TestUtilities.measureCharWidth(string, font, fontSize);
		// the full width is 2812 em (including side bearings) for this glyph when the height is 2048 em (ratio =
		// 1.3730)
		Assert.assertEquals("string width colored emoji", 1.3730f * fontSize, stringWidth, DELTA);
	}

	/**
	 * Test drawing a string with emoji.
	 */
	@Test
	public void testEmojiDrawString() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_EMOJI_GRINNING;
		int fontSize = 48;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		// Check character : full box filled
		int charWidth = (int) TestUtilities.measureCharWidth(string, font, fontSize);
		TestUtilities.checkArea("emoji", color, x, y, charWidth, fontSize, PADDING);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("emoji", TestUtilities.BACKGROUND_COLOR, x, y, charWidth, fontSize, 50, 3);
	}

	/**
	 * Test drawing a string with a unicode in the Supplementary Private Use Area-B (U+10FFFD).
	 * <p>
	 * Not strictly emoji but this is a area where the user can put private/custom glyphs.
	 */
	@Test
	public void testEmojiDrawStringPrivateUseAreaB() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();
		String string = STRING_SPUAB;
		int fontSize = 48;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		// Check character : full box filled
		int charWidth = (int) TestUtilities.measureCharWidth(string, font, fontSize);
		TestUtilities.checkArea("Supplementary Private Area-B", color, x, y, charWidth, fontSize, PADDING);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("Supplementary Private Area-B", TestUtilities.BACKGROUND_COLOR, x, y,
				charWidth, fontSize, 50, 3);
	}

	/**
	 * Test the measurement of the width of an emoji.
	 */
	@Test
	public void testEmojiStringWidthEmoji() {
		VectorFont font = getTestFont();
		float fontSize = 30;

		float stringWidth = TestUtilities.measureCharWidth(STRING_EMOJI_GRINNING, font, fontSize);
		Assert.assertEquals("emoji string width", fontSize, stringWidth, DELTA);
	}

	/**
	 * Test the measurement of the width of a character in the Supplementary Private Use Area-B (U+10FFFD).
	 */
	@Test
	public void testEmojiStringWidthSupplementaryCharacter() {
		VectorFont font = getTestFont();
		float fontSize = 30;

		float stringWidth = TestUtilities.measureCharWidth(STRING_SPUAB, font, fontSize);
		Assert.assertEquals("Supplementary Private Use Area-B string width", fontSize, stringWidth, DELTA);
	}

	private static VectorFont getTestFont() {
		return font;
	}

	private static VectorFont getEmojiFont() {
		return emoji;
	}

}
