/*
 * Copyright (c) 1996, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2017-2019 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * Portions Copyright IBM Corporation, 2001. All Rights Reserved.
 */

package java.math;

import static java.math.BigInteger.LONG_MASK;

/**
 * Immutable, arbitrary-precision signed decimal numbers. A {@code BigDecimal} consists of an arbitrary precision
 * integer <i>unscaled value</i> and a 32-bit integer <i>scale</i>. If zero or positive, the scale is the number of
 * digits to the right of the decimal point. If negative, the unscaled value of the number is multiplied by ten to the
 * power of the negation of the scale. The value of the number represented by the {@code BigDecimal} is therefore
 * <tt>(unscaledValue &times; 10<sup>-scale</sup>)</tt>.
 *
 * <p>
 * The {@code BigDecimal} class provides operations for arithmetic, scale manipulation, rounding, comparison, hashing,
 * and format conversion. The {@link #toString} method provides a canonical representation of a {@code BigDecimal}.
 *
 * <p>
 * The {@code BigDecimal} class gives its user complete control over rounding behavior. If no rounding mode is specified
 * and the exact result cannot be represented, an exception is thrown; otherwise, calculations can be carried out to a
 * chosen precision and rounding mode by supplying an appropriate {@link MathContext} object to the operation. In either
 * case, eight <em>rounding
 * modes</em> are provided for the control of rounding. Using the integer fields in this class (such as
 * {@link #ROUND_HALF_UP}) to represent rounding mode is largely obsolete; the enumeration values of the
 * {@code RoundingMode} {@code enum}, (such as {@link RoundingMode#HALF_UP}) should be used instead.
 *
 * <p>
 * When a {@code MathContext} object is supplied with a precision setting of 0 (for example,
 * {@link MathContext#UNLIMITED}), arithmetic operations are exact, as are the arithmetic methods which take no
 * {@code MathContext} object. (This is the only behavior that was supported in releases prior to 5.) As a corollary of
 * computing the exact result, the rounding mode setting of a {@code MathContext} object with a precision setting of 0
 * is not used and thus irrelevant. In the case of divide, the exact quotient could have an infinitely long decimal
 * expansion; for example, 1 divided by 3. If the quotient has a nonterminating decimal expansion and the operation is
 * specified to return an exact result, an {@code ArithmeticException} is thrown. Otherwise, the exact result of the
 * division is returned, as done for other operations.
 *
 * <p>
 * When the precision setting is not 0, the rules of {@code BigDecimal} arithmetic are broadly compatible with selected
 * modes of operation of the arithmetic defined in ANSI X3.274-1996 and ANSI X3.274-1996/AM 1-2000 (section 7.4). Unlike
 * those standards, {@code BigDecimal} includes many rounding modes, which were mandatory for division in
 * {@code BigDecimal} releases prior to 5. Any conflicts between these ANSI standards and the {@code BigDecimal}
 * specification are resolved in favor of {@code BigDecimal}.
 *
 * <p>
 * Since the same numerical value can have different representations (with different scales), the rules of arithmetic
 * and rounding must specify both the numerical result and the scale used in the result's representation.
 *
 *
 * <p>
 * In general the rounding modes and precision setting determine how operations return results with a limited number of
 * digits when the exact result has more digits (perhaps infinitely many in the case of division) than the number of
 * digits returned.
 *
 * First, the total number of digits to return is specified by the {@code MathContext}'s {@code precision} setting; this
 * determines the result's <i>precision</i>. The digit count starts from the leftmost nonzero digit of the exact result.
 * The rounding mode determines how any discarded trailing digits affect the returned result.
 *
 * <p>
 * For all arithmetic operators , the operation is carried out as though an exact intermediate result were first
 * calculated and then rounded to the number of digits specified by the precision setting (if necessary), using the
 * selected rounding mode. If the exact result is not returned, some digit positions of the exact result are discarded.
 * When rounding increases the magnitude of the returned result, it is possible for a new digit position to be created
 * by a carry propagating to a leading {@literal "9"} digit. For example, rounding the value 999.9 to three digits
 * rounding up would be numerically equal to one thousand, represented as 100&times;10<sup>1</sup>. In such cases, the
 * new {@literal "1"} is the leading digit position of the returned result.
 *
 * <p>
 * Besides a logical exact result, each arithmetic operation has a preferred scale for representing a result. The
 * preferred scale for each operation is listed in the table below.
 *
 * <table border> <caption><b>Preferred Scales for Results of Arithmetic Operations </b></caption>
 * <tr>
 * <th>Operation</th>
 * <th>Preferred Scale of Result</th>
 * </tr>
 * <tr>
 * <td>Add</td>
 * <td>max(addend.scale(), augend.scale())</td>
 * <tr>
 * <td>Subtract</td>
 * <td>max(minuend.scale(), subtrahend.scale())</td>
 * <tr>
 * <td>Multiply</td>
 * <td>multiplier.scale() + multiplicand.scale()</td>
 * <tr>
 * <td>Divide</td>
 * <td>dividend.scale() - divisor.scale()</td>
 * </table>
 *
 * These scales are the ones used by the methods which return exact arithmetic results; except that an exact divide may
 * have to use a larger scale since the exact result may have more digits. For example, {@code 1/32} is {@code 0.03125}.
 *
 * <p>
 * Before rounding, the scale of the logical exact intermediate result is the preferred scale for that operation. If the
 * exact numerical result cannot be represented in {@code precision} digits, rounding selects the set of digits to
 * return and the scale of the result is reduced from the scale of the intermediate result to the least scale which can
 * represent the {@code precision} digits actually returned. If the exact result can be represented with at most
 * {@code precision} digits, the representation of the result with the scale closest to the preferred scale is returned.
 * In particular, an exactly representable quotient may be represented in fewer than {@code precision} digits by
 * removing trailing zeros and decreasing the scale. For example, rounding to three digits using the
 * {@linkplain RoundingMode#FLOOR floor} rounding mode, <br>
 *
 * {@code 19/100 = 0.19   // integer=19,  scale=2} <br>
 *
 * but<br>
 *
 * {@code 21/110 = 0.190  // integer=190, scale=3} <br>
 *
 * <p>
 * Note that for add, subtract, and multiply, the reduction in scale will equal the number of digit positions of the
 * exact result which are discarded. If the rounding causes a carry propagation to create a new high-order digit
 * position, an additional digit of the result is discarded than when no new digit position is created.
 *
 * <p>
 * Other methods may have slightly different rounding semantics. For example, the result of the {@code pow} method using
 * the {@linkplain #pow(int, MathContext) specified algorithm} can occasionally differ from the rounded mathematical
 * result by more than one unit in the last place, one <i>{@linkplain #ulp() ulp}</i>.
 *
 * <p>
 * Two types of operations are provided for manipulating the scale of a {@code BigDecimal}: scaling/rounding operations
 * and decimal point motion operations. Scaling/rounding operations ({@link #setScale setScale} and {@link #round round}
 * ) return a {@code BigDecimal} whose value is approximately (or exactly) equal to that of the operand, but whose scale
 * or precision is the specified value; that is, they increase or decrease the precision of the stored number with
 * minimal effect on its value. Decimal point motion operations ({@link #movePointLeft movePointLeft} and
 * {@link #movePointRight movePointRight}) return a {@code BigDecimal} created from the operand by moving the decimal
 * point a specified distance in the specified direction.
 *
 * <p>
 * For the sake of brevity and clarity, pseudo-code is used throughout the descriptions of {@code BigDecimal} methods.
 * The pseudo-code expression {@code (i + j)} is shorthand for "a {@code BigDecimal} whose value is that of the
 * {@code BigDecimal} {@code i} added to that of the {@code BigDecimal} {@code j}." The pseudo-code expression
 * {@code (i == j)} is shorthand for "{@code true} if and only if the {@code BigDecimal} {@code i} represents the same
 * value as the {@code BigDecimal} {@code j}." Other pseudo-code expressions are interpreted similarly. Square brackets
 * are used to represent the particular {@code BigInteger} and scale pair defining a {@code BigDecimal} value; for
 * example [19, 2] is the {@code BigDecimal} numerically equal to 0.19 having a scale of 2.
 *
 * <p>
 * Note: care should be exercised if {@code BigDecimal} objects are used as keys in a {@link java.util.SortedMap
 * SortedMap} or elements in a {@link java.util.SortedSet SortedSet} since {@code BigDecimal}'s <i>natural ordering</i>
 * is <i>inconsistent with equals</i>. See {@link Comparable}, {@link java.util.SortedMap} or
 * {@link java.util.SortedSet} for more information.
 *
 * <p>
 * All methods and constructors for this class throw {@code NullPointerException} when passed a {@code null} object
 * reference for any input parameter.
 *
 * @see BigInteger
 * @see MathContext
 * @see RoundingMode
 * @see java.util.SortedMap
 * @see java.util.SortedSet
 * @author Josh Bloch
 * @author Mike Cowlishaw
 * @author Joseph D. Darcy
 * @author Sergey V. Kuksenko
 */
public class BigDecimal extends Number implements Comparable<BigDecimal> {
	/**
	 * The unscaled value of this BigDecimal, as returned by {@link #unscaledValue}.
	 *
	 * @serial
	 * @see #unscaledValue
	 */
	private final BigInteger intVal;

	/**
	 * The scale of this BigDecimal, as returned by {@link #scale}.
	 *
	 * @serial
	 * @see #scale
	 */
	private final int scale; // Note: this may have any value, so
								// calculations must be done in longs

	/**
	 * The number of decimal digits in this BigDecimal, or 0 if the number of digits are not known (lookaside
	 * information). If nonzero, the value is guaranteed correct. Use the precision() method to obtain and set the value
	 * if it might be 0. This field is mutable until set nonzero.
	 *
	 * @since 1.5
	 */
	private transient int precision;

	/**
	 * Used to store the canonical string representation, if computed.
	 */
	private transient String stringCache;

	/**
	 * Sentinel value for {@link #intCompact} indicating the significand information is only available from
	 * {@code intVal}.
	 */
	static final long INFLATED = Long.MIN_VALUE;

	private static final BigInteger INFLATED_BIGINT = BigInteger.valueOf(INFLATED);

	/**
	 * If the absolute value of the significand of this BigDecimal is less than or equal to {@code Long.MAX_VALUE}, the
	 * value can be compactly stored in this field and used in computations.
	 */
	private final transient long intCompact;

	// All 18-digit base ten strings fit into a long; not all 19-digit
	// strings will
	private static final int MAX_COMPACT_DIGITS = 18;

	/* Appease the serialization gods */
	private static final long serialVersionUID = 6108874887143696463L;

	// Cache of common small BigDecimal values.
	private static final BigDecimal zeroThroughTen[] = { new BigDecimal(BigInteger.ZERO, 0, 0, 1),
			new BigDecimal(BigInteger.ONE, 1, 0, 1), new BigDecimal(BigInteger.valueOf(2), 2, 0, 1),
			new BigDecimal(BigInteger.valueOf(3), 3, 0, 1), new BigDecimal(BigInteger.valueOf(4), 4, 0, 1),
			new BigDecimal(BigInteger.valueOf(5), 5, 0, 1), new BigDecimal(BigInteger.valueOf(6), 6, 0, 1),
			new BigDecimal(BigInteger.valueOf(7), 7, 0, 1), new BigDecimal(BigInteger.valueOf(8), 8, 0, 1),
			new BigDecimal(BigInteger.valueOf(9), 9, 0, 1), new BigDecimal(BigInteger.TEN, 10, 0, 2), };

	// Cache of zero scaled by 0 - 15
	private static final BigDecimal[] ZERO_SCALED_BY = { zeroThroughTen[0], new BigDecimal(BigInteger.ZERO, 0, 1, 1),
			new BigDecimal(BigInteger.ZERO, 0, 2, 1), new BigDecimal(BigInteger.ZERO, 0, 3, 1),
			new BigDecimal(BigInteger.ZERO, 0, 4, 1), new BigDecimal(BigInteger.ZERO, 0, 5, 1),
			new BigDecimal(BigInteger.ZERO, 0, 6, 1), new BigDecimal(BigInteger.ZERO, 0, 7, 1),
			new BigDecimal(BigInteger.ZERO, 0, 8, 1), new BigDecimal(BigInteger.ZERO, 0, 9, 1),
			new BigDecimal(BigInteger.ZERO, 0, 10, 1), new BigDecimal(BigInteger.ZERO, 0, 11, 1),
			new BigDecimal(BigInteger.ZERO, 0, 12, 1), new BigDecimal(BigInteger.ZERO, 0, 13, 1),
			new BigDecimal(BigInteger.ZERO, 0, 14, 1), new BigDecimal(BigInteger.ZERO, 0, 15, 1), };

	// Half of Long.MIN_VALUE & Long.MAX_VALUE.
	private static final long HALF_LONG_MAX_VALUE = Long.MAX_VALUE / 2;
	private static final long HALF_LONG_MIN_VALUE = Long.MIN_VALUE / 2;

	// Constants
	/**
	 * The value 0, with a scale of 0.
	 *
	 * @since 1.5
	 */
	public static final BigDecimal ZERO = zeroThroughTen[0];

	/**
	 * The value 1, with a scale of 0.
	 *
	 * @since 1.5
	 */
	public static final BigDecimal ONE = zeroThroughTen[1];

	/**
	 * The value 10, with a scale of 0.
	 *
	 * @since 1.5
	 */
	public static final BigDecimal TEN = zeroThroughTen[10];

	// Constructors

	/**
	 * Trusted package private constructor. Trusted simply means if val is INFLATED, intVal could not be null and if
	 * intVal is null, val could not be INFLATED.
	 */
	BigDecimal(BigInteger intVal, long val, int scale, int prec) {
		this.scale = scale;
		this.precision = prec;
		this.intCompact = val;
		this.intVal = intVal;
	}

	/**
	 * Translates a character array representation of a {@code BigDecimal} into a {@code BigDecimal}, accepting the same
	 * sequence of characters as the {@link #BigDecimal(String)} constructor, while allowing a sub-array to be
	 * specified.
	 *
	 * <p>
	 * Note that if the sequence of characters is already available within a character array, using this constructor is
	 * faster than converting the {@code char} array to string and using the {@code BigDecimal(String)} constructor .
	 *
	 * @param in
	 *            {@code char} array that is the source of characters.
	 * @param offset
	 *            first character in the array to inspect.
	 * @param len
	 *            number of characters to consider.
	 * @throws NumberFormatException
	 *             if {@code in} is not a valid representation of a {@code BigDecimal} or the defined subarray is not
	 *             wholly within {@code in}.
	 * @since 1.5
	 */
	public BigDecimal(char[] in, int offset, int len) {
		this(in, offset, len, MathContext.UNLIMITED);
	}

	/**
	 * Translates a character array representation of a {@code BigDecimal} into a {@code BigDecimal}, accepting the same
	 * sequence of characters as the {@link #BigDecimal(String)} constructor, while allowing a sub-array to be specified
	 * and with rounding according to the context settings.
	 *
	 * <p>
	 * Note that if the sequence of characters is already available within a character array, using this constructor is
	 * faster than converting the {@code char} array to string and using the {@code BigDecimal(String)} constructor .
	 *
	 * @param in
	 *            {@code char} array that is the source of characters.
	 * @param offset
	 *            first character in the array to inspect.
	 * @param len
	 *            number of characters to consider..
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @throws NumberFormatException
	 *             if {@code in} is not a valid representation of a {@code BigDecimal} or the defined subarray is not
	 *             wholly within {@code in}.
	 * @since 1.5
	 */
	public BigDecimal(char[] in, int offset, int len, MathContext mc) {
		// protect against huge length.
		if (offset + len > in.length || offset < 0) {
			throw new NumberFormatException("Bad offset or len arguments for char[] input.");
			// This is the primary string to BigDecimal constructor; all
			// incoming strings end up here; it uses explicit (inline)
			// parsing for speed and generates at most one intermediate
			// (temporary) object (a char[] array) for non-compact case.
		}

		// Use locals for all fields values until completion
		int prec = 0; // record precision value
		int scl = 0; // record scale value
		long rs = 0; // the compact value in long
		BigInteger rb = null; // the inflated value in BigInteger
		// use array bounds checking to handle too-long, len == 0,
		// bad offset, etc.
		try {
			// handle the sign
			boolean isneg = false; // assume positive
			if (in[offset] == '-') {
				isneg = true; // leading minus means negative
				offset++;
				len--;
			} else if (in[offset] == '+') { // leading + allowed
				offset++;
				len--;
			}

			// should now be at numeric part of the significand
			boolean dot = false; // true when there is a '.'
			long exp = 0; // exponent
			char c; // current character
			boolean isCompact = (len <= MAX_COMPACT_DIGITS);
			// integer significand array & idx is the index to it. The array
			// is ONLY used when we can't use a compact representation.
			int idx = 0;
			if (isCompact) {
				// First compact case, we need not to preserve the character
				// and we can just compute the value in place.
				for (; len > 0; offset++, len--) {
					c = in[offset];
					if ((c == '0')) { // have zero
						if (prec == 0) {
							prec = 1;
						} else if (rs != 0) {
							rs *= 10;
							++prec;
						} // else digit is a redundant leading zero
						if (dot) {
							++scl;
						}
					} else if ((c >= '1' && c <= '9')) { // have digit
						int digit = c - '0';
						if (prec != 1 || rs != 0) {
							++prec; // prec unchanged if preceded by 0s
						}
						rs = rs * 10 + digit;
						if (dot) {
							++scl;
						}
					} else if (c == '.') { // have dot
						// have dot
						if (dot) {
							throw new NumberFormatException();
						}
						dot = true;
					} else if (Character.isDigit(c)) { // slow path
						int digit = Character.digit(c, 10);
						if (digit == 0) {
							if (prec == 0) {
								prec = 1;
							} else if (rs != 0) {
								rs *= 10;
								++prec;
							} // else digit is a redundant leading zero
						} else {
							if (prec != 1 || rs != 0) {
								++prec; // prec unchanged if preceded by 0s
							}
							rs = rs * 10 + digit;
						}
						if (dot) {
							++scl;
						}
					} else if ((c == 'e') || (c == 'E')) {
						exp = parseExp(in, offset, len);
						// Next test is required for backwards compatibility
						if ((int) exp != exp) {
							throw new NumberFormatException();
						}
						break; // [saves a test]
					} else {
						throw new NumberFormatException();
					}
				}
				if (prec == 0) {
					throw new NumberFormatException();
				}
				// Adjust scale if exp is not zero.
				if (exp != 0) { // had significant exponent
					scl = adjustScale(scl, exp);
				}
				rs = isneg ? -rs : rs;
				int mcp = mc.precision;
				int drop = prec - mcp; // prec has range [1, MAX_INT], mcp has range [0, MAX_INT];
										// therefore, this subtract cannot overflow
				if (mcp > 0 && drop > 0) { // do rounding
					while (drop > 0) {
						scl = checkScaleNonZero((long) scl - drop);
						rs = divideAndRound(rs, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
						prec = longDigitLength(rs);
						drop = prec - mcp;
					}
				}
			} else {
				char coeff[] = new char[len];
				for (; len > 0; offset++, len--) {
					c = in[offset];
					// have digit
					if ((c >= '0' && c <= '9') || Character.isDigit(c)) {
						// First compact case, we need not to preserve the character
						// and we can just compute the value in place.
						if (c == '0' || Character.digit(c, 10) == 0) {
							if (prec == 0) {
								coeff[idx] = c;
								prec = 1;
							} else if (idx != 0) {
								coeff[idx++] = c;
								++prec;
							} // else c must be a redundant leading zero
						} else {
							if (prec != 1 || idx != 0) {
								++prec; // prec unchanged if preceded by 0s
							}
							coeff[idx++] = c;
						}
						if (dot) {
							++scl;
						}
						continue;
					}
					// have dot
					if (c == '.') {
						// have dot
						if (dot) {
							throw new NumberFormatException();
						}
						dot = true;
						continue;
					}
					// exponent expected
					if ((c != 'e') && (c != 'E')) {
						throw new NumberFormatException();
					}
					exp = parseExp(in, offset, len);
					// Next test is required for backwards compatibility
					if ((int) exp != exp) {
						throw new NumberFormatException();
					}
					break; // [saves a test]
				}
				// here when no characters left
				if (prec == 0) {
					throw new NumberFormatException();
				}
				// Adjust scale if exp is not zero.
				if (exp != 0) { // had significant exponent
					scl = adjustScale(scl, exp);
				}
				// Remove leading zeros from precision (digits count)
				rb = new BigInteger(coeff, isneg ? -1 : 1, prec);
				rs = compactValFor(rb);
				int mcp = mc.precision;
				if (mcp > 0 && (prec > mcp)) {
					if (rs == INFLATED) {
						int drop = prec - mcp;
						while (drop > 0) {
							scl = checkScaleNonZero((long) scl - drop);
							rb = divideAndRoundByTenPow(rb, drop, mc.roundingMode.oldMode);
							rs = compactValFor(rb);
							if (rs != INFLATED) {
								prec = longDigitLength(rs);
								break;
							}
							prec = bigDigitLength(rb);
							drop = prec - mcp;
						}
					}
					if (rs != INFLATED) {
						int drop = prec - mcp;
						while (drop > 0) {
							scl = checkScaleNonZero((long) scl - drop);
							rs = divideAndRound(rs, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
							prec = longDigitLength(rs);
							drop = prec - mcp;
						}
						rb = null;
					}
				}
			}
		} catch (ArrayIndexOutOfBoundsException e) {
			throw new NumberFormatException();
		} catch (NegativeArraySizeException e) {
			throw new NumberFormatException();
		}
		this.scale = scl;
		this.precision = prec;
		this.intCompact = rs;
		this.intVal = rb;
	}

	private int adjustScale(int scl, long exp) {
		long adjustedScale = scl - exp;
		if (adjustedScale > Integer.MAX_VALUE || adjustedScale < Integer.MIN_VALUE) {
			throw new NumberFormatException("Scale out of range.");
		}
		scl = (int) adjustedScale;
		return scl;
	}

	/*
	 * parse exponent
	 */
	private static long parseExp(char[] in, int offset, int len) {
		long exp = 0;
		offset++;
		char c = in[offset];
		len--;
		boolean negexp = (c == '-');
		// optional sign
		if (negexp || c == '+') {
			offset++;
			c = in[offset];
			len--;
		}
		if (len <= 0) {
			throw new NumberFormatException();
		}
		// skip leading zeros in the exponent
		while (len > 10 && (c == '0' || (Character.digit(c, 10) == 0))) {
			offset++;
			c = in[offset];
			len--;
		}
		if (len > 10) {
			throw new NumberFormatException();
		}
		// c now holds first digit of exponent
		for (;; len--) {
			int v;
			if (c >= '0' && c <= '9') {
				v = c - '0';
			} else {
				v = Character.digit(c, 10);
				if (v < 0) {
					throw new NumberFormatException();
				}
			}
			exp = exp * 10 + v;
			if (len == 1) {
				break; // that was final character
			}
			offset++;
			c = in[offset];
		}
		if (negexp) {
			exp = -exp;
		}
		return exp;
	}

	/**
	 * Translates a character array representation of a {@code BigDecimal} into a {@code BigDecimal}, accepting the same
	 * sequence of characters as the {@link #BigDecimal(String)} constructor.
	 *
	 * <p>
	 * Note that if the sequence of characters is already available as a character array, using this constructor is
	 * faster than converting the {@code char} array to string and using the {@code BigDecimal(String)} constructor .
	 *
	 * @param in
	 *            {@code char} array that is the source of characters.
	 * @throws NumberFormatException
	 *             if {@code in} is not a valid representation of a {@code BigDecimal}.
	 * @since 1.5
	 */
	public BigDecimal(char[] in) {
		this(in, 0, in.length);
	}

	/**
	 * Translates a character array representation of a {@code BigDecimal} into a {@code BigDecimal}, accepting the same
	 * sequence of characters as the {@link #BigDecimal(String)} constructor and with rounding according to the context
	 * settings.
	 *
	 * <p>
	 * Note that if the sequence of characters is already available as a character array, using this constructor is
	 * faster than converting the {@code char} array to string and using the {@code BigDecimal(String)} constructor .
	 *
	 * @param in
	 *            {@code char} array that is the source of characters.
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @throws NumberFormatException
	 *             if {@code in} is not a valid representation of a {@code BigDecimal}.
	 * @since 1.5
	 */
	public BigDecimal(char[] in, MathContext mc) {
		this(in, 0, in.length, mc);
	}

	/**
	 * Translates the string representation of a {@code BigDecimal} into a {@code BigDecimal}. The string representation
	 * consists of an optional sign, {@code '+'} (<tt> '&#92;u002B'</tt>) or {@code '-'} (<tt>'&#92;u002D'</tt>),
	 * followed by a sequence of zero or more decimal digits ("the integer"), optionally followed by a fraction,
	 * optionally followed by an exponent.
	 *
	 * <p>
	 * The fraction consists of a decimal point followed by zero or more decimal digits. The string must contain at
	 * least one digit in either the integer or the fraction. The number formed by the sign, the integer and the
	 * fraction is referred to as the <i>significand</i>.
	 *
	 * <p>
	 * The exponent consists of the character {@code 'e'} (<tt>'&#92;u0065'</tt>) or {@code 'E'} (<tt>'&#92;u0045'</tt>)
	 * followed by one or more decimal digits. The value of the exponent must lie between -{@link Integer#MAX_VALUE} (
	 * {@link Integer#MIN_VALUE}+1) and {@link Integer#MAX_VALUE}, inclusive.
	 *
	 * <p>
	 * More formally, the strings this constructor accepts are described by the following grammar: <blockquote>
	 * <dl>
	 * <dt><i>BigDecimalString:</i>
	 * <dd><i>Sign<sub>opt</sub> Significand Exponent<sub>opt</sub></i>
	 * <dt><i>Sign:</i>
	 * <dd>{@code +}
	 * <dd>{@code -}
	 * <dt><i>Significand:</i>
	 * <dd><i>IntegerPart</i> {@code .} <i>FractionPart<sub>opt</sub></i>
	 * <dd>{@code .} <i>FractionPart</i>
	 * <dd><i>IntegerPart</i>
	 * <dt><i>IntegerPart:</i>
	 * <dd><i>Digits</i>
	 * <dt><i>FractionPart:</i>
	 * <dd><i>Digits</i>
	 * <dt><i>Exponent:</i>
	 * <dd><i>ExponentIndicator SignedInteger</i>
	 * <dt><i>ExponentIndicator:</i>
	 * <dd>{@code e}
	 * <dd>{@code E}
	 * <dt><i>SignedInteger:</i>
	 * <dd><i>Sign<sub>opt</sub> Digits</i>
	 * <dt><i>Digits:</i>
	 * <dd><i>Digit</i>
	 * <dd><i>Digits Digit</i>
	 * <dt><i>Digit:</i>
	 * <dd>any character for which {@link Character#isDigit} returns {@code true}, including 0, 1, 2 ...
	 * </dl>
	 * </blockquote>
	 *
	 * <p>
	 * The scale of the returned {@code BigDecimal} will be the number of digits in the fraction, or zero if the string
	 * contains no decimal point, subject to adjustment for any exponent; if the string contains an exponent, the
	 * exponent is subtracted from the scale. The value of the resulting scale must lie between
	 * {@code Integer.MIN_VALUE} and {@code Integer.MAX_VALUE}, inclusive.
	 *
	 * <p>
	 * The character-to-digit mapping is provided by {@link java.lang.Character#digit} set to convert to radix 10. The
	 * String may not contain any extraneous characters (whitespace, for example).
	 *
	 * <p>
	 * <b>Examples:</b><br>
	 * The value of the returned {@code BigDecimal} is equal to <i>significand</i> &times; 10<sup>&nbsp;
	 * <i>exponent</i></sup>. For each string on the left, the resulting representation [{@code BigInteger},
	 * {@code scale}] is shown on the right.
	 *
	 * <pre>
	 * "0"            [0,0]
	 * "0.00"         [0,2]
	 * "123"          [123,0]
	 * "-123"         [-123,0]
	 * "1.23E3"       [123,-1]
	 * "1.23E+3"      [123,-1]
	 * "12.3E+7"      [123,-6]
	 * "12.0"         [120,1]
	 * "12.3"         [123,1]
	 * "0.00123"      [123,5]
	 * "-1.23E-12"    [-123,14]
	 * "1234.5E-4"    [12345,5]
	 * "0E+7"         [0,-7]
	 * "-0"           [0,0]
	 * </pre>
	 *
	 * <p>
	 * Note: For values other than {@code float} and {@code double} NaN and &plusmn;Infinity, this constructor is
	 * compatible with the values returned by {@link Float#toString} and {@link Double#toString}. This is generally the
	 * preferred way to convert a {@code float} or {@code double} into a BigDecimal, as it doesn't suffer from the
	 * unpredictability of the {@link #BigDecimal(double)} constructor.
	 *
	 * @param val
	 *            String representation of {@code BigDecimal}.
	 *
	 * @throws NumberFormatException
	 *             if {@code val} is not a valid representation of a {@code BigDecimal}.
	 */
	public BigDecimal(String val) {
		this(val.toCharArray(), 0, val.length());
	}

	/**
	 * Translates the string representation of a {@code BigDecimal} into a {@code BigDecimal}, accepting the same
	 * strings as the {@link #BigDecimal(String)} constructor, with rounding according to the context settings.
	 *
	 * @param val
	 *            string representation of a {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @throws NumberFormatException
	 *             if {@code val} is not a valid representation of a BigDecimal.
	 * @since 1.5
	 */
	public BigDecimal(String val, MathContext mc) {
		this(val.toCharArray(), 0, val.length(), mc);
	}

	/**
	 * Translates a {@code double} into a {@code BigDecimal} which is the exact decimal representation of the
	 * {@code double}'s binary floating-point value. The scale of the returned {@code BigDecimal} is the smallest value
	 * such that <tt>(10<sup>scale</sup> &times; val)</tt> is an integer.
	 * <p>
	 * <b>Notes:</b>
	 * <ol>
	 * <li>The results of this constructor can be somewhat unpredictable. One might assume that writing
	 * {@code new BigDecimal(0.1)} in Java creates a {@code BigDecimal} which is exactly equal to 0.1 (an unscaled value
	 * of 1, with a scale of 1), but it is actually equal to 0.1000000000000000055511151231257827021181583404541015625.
	 * This is because 0.1 cannot be represented exactly as a {@code double} (or, for that matter, as a binary fraction
	 * of any finite length). Thus, the value that is being passed <i>in</i> to the constructor is not exactly equal to
	 * 0.1, appearances notwithstanding.
	 *
	 * <li>The {@code String} constructor, on the other hand, is perfectly predictable: writing
	 * {@code new BigDecimal("0.1")} creates a {@code BigDecimal} which is <i>exactly</i> equal to 0.1, as one would
	 * expect. Therefore, it is generally recommended that the {@linkplain #BigDecimal(String) <tt>String</tt>
	 * constructor} be used in preference to this one.
	 *
	 * <li>When a {@code double} must be used as a source for a {@code BigDecimal}, note that this constructor provides
	 * an exact conversion; it does not give the same result as converting the {@code double} to a {@code String} using
	 * the {@link Double#toString(double)} method and then using the {@link #BigDecimal(String)} constructor. To get
	 * that result, use the {@code static} {@link #valueOf(double)} method.
	 * </ol>
	 *
	 * @param val
	 *            {@code double} value to be converted to {@code BigDecimal}.
	 * @throws NumberFormatException
	 *             if {@code val} is infinite or NaN.
	 */
	public BigDecimal(double val) {
		this(val, MathContext.UNLIMITED);
	}

	/**
	 * Translates a {@code double} into a {@code BigDecimal}, with rounding according to the context settings. The scale
	 * of the {@code BigDecimal} is the smallest value such that <tt>(10<sup>scale</sup> &times; val)</tt> is an
	 * integer.
	 *
	 * <p>
	 * The results of this constructor can be somewhat unpredictable and its use is generally not recommended; see the
	 * notes under the {@link #BigDecimal(double)} constructor.
	 *
	 * @param val
	 *            {@code double} value to be converted to {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the RoundingMode is UNNECESSARY.
	 * @throws NumberFormatException
	 *             if {@code val} is infinite or NaN.
	 * @since 1.5
	 */
	public BigDecimal(double val, MathContext mc) {
		if (Double.isInfinite(val) || Double.isNaN(val)) {
			throw new NumberFormatException("Infinite or NaN");
		}
		// Translate the double into sign, exponent and significand, according
		// to the formulae in JLS, Section 20.10.22.
		long valBits = Double.doubleToLongBits(val);
		int sign = ((valBits >> 63) == 0 ? 1 : -1);
		int exponent = (int) ((valBits >> 52) & 0x7ffL);
		long significand = (exponent == 0 ? (valBits & ((1L << 52) - 1)) << 1
				: (valBits & ((1L << 52) - 1)) | (1L << 52));
		exponent -= 1075;
		// At this point, val == sign * significand * 2**exponent.

		/*
		 * Special case zero to supress nonterminating normalization and bogus scale calculation.
		 */
		if (significand == 0) {
			this.intVal = BigInteger.ZERO;
			this.scale = 0;
			this.intCompact = 0;
			this.precision = 1;
			return;
		}
		// Normalize
		while ((significand & 1) == 0) { // i.e., significand is even
			significand >>= 1;
			exponent++;
		}
		int scale = 0;
		// Calculate intVal and scale
		BigInteger intVal;
		long compactVal = sign * significand;
		if (exponent == 0) {
			intVal = (compactVal == INFLATED) ? INFLATED_BIGINT : null;
		} else {
			if (exponent < 0) {
				intVal = BigInteger.valueOf(5).pow(-exponent).multiply(compactVal);
				scale = -exponent;
			} else { // (exponent > 0)
				intVal = BigInteger.valueOf(2).pow(exponent).multiply(compactVal);
			}
			compactVal = compactValFor(intVal);
		}
		int prec = 0;
		int mcp = mc.precision;
		if (mcp > 0) { // do rounding
			int mode = mc.roundingMode.oldMode;
			int drop;
			if (compactVal == INFLATED) {
				prec = bigDigitLength(intVal);
				drop = prec - mcp;
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					intVal = divideAndRoundByTenPow(intVal, drop, mode);
					compactVal = compactValFor(intVal);
					if (compactVal != INFLATED) {
						break;
					}
					prec = bigDigitLength(intVal);
					drop = prec - mcp;
				}
			}
			if (compactVal != INFLATED) {
				prec = longDigitLength(compactVal);
				drop = prec - mcp;
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					compactVal = divideAndRound(compactVal, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
					prec = longDigitLength(compactVal);
					drop = prec - mcp;
				}
				intVal = null;
			}
		}
		this.intVal = intVal;
		this.intCompact = compactVal;
		this.scale = scale;
		this.precision = prec;
	}

	/**
	 * Translates a {@code BigInteger} into a {@code BigDecimal}. The scale of the {@code BigDecimal} is zero.
	 *
	 * @param val
	 *            {@code BigInteger} value to be converted to {@code BigDecimal}.
	 */
	public BigDecimal(BigInteger val) {
		this.scale = 0;
		this.intVal = val;
		this.intCompact = compactValFor(val);
	}

	/**
	 * Translates a {@code BigInteger} into a {@code BigDecimal} rounding according to the context settings. The scale
	 * of the {@code BigDecimal} is zero.
	 *
	 * @param val
	 *            {@code BigInteger} value to be converted to {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal(BigInteger val, MathContext mc) {
		this(val, 0, mc);
	}

	/**
	 * Translates a {@code BigInteger} unscaled value and an {@code int} scale into a {@code BigDecimal}. The value of
	 * the {@code BigDecimal} is <tt>(unscaledVal &times; 10<sup>-scale</sup>)</tt>.
	 *
	 * @param unscaledVal
	 *            unscaled value of the {@code BigDecimal}.
	 * @param scale
	 *            scale of the {@code BigDecimal}.
	 */
	public BigDecimal(BigInteger unscaledVal, int scale) {
		// Negative scales are now allowed
		this.intVal = unscaledVal;
		this.intCompact = compactValFor(unscaledVal);
		this.scale = scale;
	}

	/**
	 * Translates a {@code BigInteger} unscaled value and an {@code int} scale into a {@code BigDecimal}, with rounding
	 * according to the context settings. The value of the {@code BigDecimal} is <tt>(unscaledVal &times;
	 * 10<sup>-scale</sup>)</tt>, rounded according to the {@code precision} and rounding mode settings.
	 *
	 * @param unscaledVal
	 *            unscaled value of the {@code BigDecimal}.
	 * @param scale
	 *            scale of the {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal(BigInteger unscaledVal, int scale, MathContext mc) {
		long compactVal = compactValFor(unscaledVal);
		int mcp = mc.precision;
		int prec = 0;
		if (mcp > 0) { // do rounding
			int mode = mc.roundingMode.oldMode;
			if (compactVal == INFLATED) {
				prec = bigDigitLength(unscaledVal);
				int drop = prec - mcp;
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					unscaledVal = divideAndRoundByTenPow(unscaledVal, drop, mode);
					compactVal = compactValFor(unscaledVal);
					if (compactVal != INFLATED) {
						break;
					}
					prec = bigDigitLength(unscaledVal);
					drop = prec - mcp;
				}
			}
			if (compactVal != INFLATED) {
				prec = longDigitLength(compactVal);
				int drop = prec - mcp; // drop can't be more than 18
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					compactVal = divideAndRound(compactVal, LONG_TEN_POWERS_TABLE[drop], mode);
					prec = longDigitLength(compactVal);
					drop = prec - mcp;
				}
				unscaledVal = null;
			}
		}
		this.intVal = unscaledVal;
		this.intCompact = compactVal;
		this.scale = scale;
		this.precision = prec;
	}

	/**
	 * Translates an {@code int} into a {@code BigDecimal}. The scale of the {@code BigDecimal} is zero.
	 *
	 * @param val
	 *            {@code int} value to be converted to {@code BigDecimal}.
	 * @since 1.5
	 */
	public BigDecimal(int val) {
		this.intCompact = val;
		this.scale = 0;
		this.intVal = null;
	}

	/**
	 * Translates an {@code int} into a {@code BigDecimal}, with rounding according to the context settings. The scale
	 * of the {@code BigDecimal}, before any rounding, is zero.
	 *
	 * @param val
	 *            {@code int} value to be converted to {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal(int val, MathContext mc) {
		int mcp = mc.precision;
		long compactVal = val;
		int scale = 0;
		int prec = 0;
		if (mcp > 0) { // do rounding
			prec = longDigitLength(compactVal);
			int drop = prec - mcp; // drop can't be more than 18
			while (drop > 0) {
				scale = checkScaleNonZero((long) scale - drop);
				compactVal = divideAndRound(compactVal, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
				prec = longDigitLength(compactVal);
				drop = prec - mcp;
			}
		}
		this.intVal = null;
		this.intCompact = compactVal;
		this.scale = scale;
		this.precision = prec;
	}

	/**
	 * Translates a {@code long} into a {@code BigDecimal}. The scale of the {@code BigDecimal} is zero.
	 *
	 * @param val
	 *            {@code long} value to be converted to {@code BigDecimal}.
	 * @since 1.5
	 */
	public BigDecimal(long val) {
		this.intCompact = val;
		this.intVal = (val == INFLATED) ? INFLATED_BIGINT : null;
		this.scale = 0;
	}

	/**
	 * Translates a {@code long} into a {@code BigDecimal}, with rounding according to the context settings. The scale
	 * of the {@code BigDecimal}, before any rounding, is zero.
	 *
	 * @param val
	 *            {@code long} value to be converted to {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal(long val, MathContext mc) {
		int mcp = mc.precision;
		int mode = mc.roundingMode.oldMode;
		int prec = 0;
		int scale = 0;
		BigInteger intVal = (val == INFLATED) ? INFLATED_BIGINT : null;
		if (mcp > 0) { // do rounding
			if (val == INFLATED) {
				prec = 19;
				int drop = prec - mcp;
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					intVal = divideAndRoundByTenPow(intVal, drop, mode);
					val = compactValFor(intVal);
					if (val != INFLATED) {
						break;
					}
					prec = bigDigitLength(intVal);
					drop = prec - mcp;
				}
			}
			if (val != INFLATED) {
				prec = longDigitLength(val);
				int drop = prec - mcp;
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					val = divideAndRound(val, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
					prec = longDigitLength(val);
					drop = prec - mcp;
				}
				intVal = null;
			}
		}
		this.intVal = intVal;
		this.intCompact = val;
		this.scale = scale;
		this.precision = prec;
	}

	// Static Factory Methods

	/**
	 * Translates a {@code long} unscaled value and an {@code int} scale into a {@code BigDecimal}. This
	 * {@literal "static factory method"} is provided in preference to a ({@code long}, {@code int}) constructor because
	 * it allows for reuse of frequently used {@code BigDecimal} values..
	 *
	 * @param unscaledVal
	 *            unscaled value of the {@code BigDecimal}.
	 * @param scale
	 *            scale of the {@code BigDecimal}.
	 * @return a {@code BigDecimal} whose value is <tt>(unscaledVal &times; 10<sup>-scale</sup>)</tt>.
	 */
	public static BigDecimal valueOf(long unscaledVal, int scale) {
		if (scale == 0) {
			return valueOf(unscaledVal);
		} else if (unscaledVal == 0) {
			return zeroValueOf(scale);
		}
		return new BigDecimal(unscaledVal == INFLATED ? INFLATED_BIGINT : null, unscaledVal, scale, 0);
	}

	/**
	 * Translates a {@code long} value into a {@code BigDecimal} with a scale of zero. This
	 * {@literal "static factory method"} is provided in preference to a ({@code long}) constructor because it allows
	 * for reuse of frequently used {@code BigDecimal} values.
	 *
	 * @param val
	 *            value of the {@code BigDecimal}.
	 * @return a {@code BigDecimal} whose value is {@code val}.
	 */
	public static BigDecimal valueOf(long val) {
		if (val >= 0 && val < zeroThroughTen.length) {
			return zeroThroughTen[(int) val];
		} else if (val != INFLATED) {
			return new BigDecimal(null, val, 0, 0);
		}
		return new BigDecimal(INFLATED_BIGINT, val, 0, 0);
	}

	static BigDecimal valueOf(long unscaledVal, int scale, int prec) {
		if (scale == 0 && unscaledVal >= 0 && unscaledVal < zeroThroughTen.length) {
			return zeroThroughTen[(int) unscaledVal];
		} else if (unscaledVal == 0) {
			return zeroValueOf(scale);
		}
		return new BigDecimal(unscaledVal == INFLATED ? INFLATED_BIGINT : null, unscaledVal, scale, prec);
	}

	static BigDecimal valueOf(BigInteger intVal, int scale, int prec) {
		long val = compactValFor(intVal);
		if (val == 0) {
			return zeroValueOf(scale);
		} else if (scale == 0 && val >= 0 && val < zeroThroughTen.length) {
			return zeroThroughTen[(int) val];
		}
		return new BigDecimal(intVal, val, scale, prec);
	}

	static BigDecimal zeroValueOf(int scale) {
		if (scale >= 0 && scale < ZERO_SCALED_BY.length) {
			return ZERO_SCALED_BY[scale];
		} else {
			return new BigDecimal(BigInteger.ZERO, 0, scale, 1);
		}
	}

	/**
	 * Translates a {@code double} into a {@code BigDecimal}, using the {@code double}'s canonical string representation
	 * provided by the {@link Double#toString(double)} method.
	 *
	 * <p>
	 * <b>Note:</b> This is generally the preferred way to convert a {@code double} (or {@code float}) into a
	 * {@code BigDecimal}, as the value returned is equal to that resulting from constructing a {@code BigDecimal} from
	 * the result of using {@link Double#toString(double)}.
	 *
	 * @param val
	 *            {@code double} to convert to a {@code BigDecimal}.
	 * @return a {@code BigDecimal} whose value is equal to or approximately equal to the value of {@code val}.
	 * @throws NumberFormatException
	 *             if {@code val} is infinite or NaN.
	 * @since 1.5
	 */
	public static BigDecimal valueOf(double val) {
		// Reminder: a zero double returns '0.0', so we cannot fastpath
		// to use the constant ZERO. This might be important enough to
		// justify a factory approach, a cache, or a few private
		// constants, later.
		return new BigDecimal(Double.toString(val));
	}

	// Arithmetic Operations
	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this +
	 * augend)}, and whose scale is {@code max(this.scale(),
	 * augend.scale())}.
	 *
	 * @param augend
	 *            value to be added to this {@code BigDecimal}.
	 * @return {@code this + augend}
	 */
	public BigDecimal add(BigDecimal augend) {
		if (this.intCompact != INFLATED) {
			if ((augend.intCompact != INFLATED)) {
				return add(this.intCompact, this.scale, augend.intCompact, augend.scale);
			} else {
				return add(this.intCompact, this.scale, augend.intVal, augend.scale);
			}
		} else {
			if ((augend.intCompact != INFLATED)) {
				return add(augend.intCompact, augend.scale, this.intVal, this.scale);
			} else {
				return add(this.intVal, this.scale, augend.intVal, augend.scale);
			}
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this + augend)}, with rounding according to the context
	 * settings.
	 *
	 * If either number is zero and the precision setting is nonzero then the other number, rounded if necessary, is
	 * used as the result.
	 *
	 * @param augend
	 *            value to be added to this {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @return {@code this + augend}, rounded as necessary.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal add(BigDecimal augend, MathContext mc) {
		if (mc.precision == 0) {
			return add(augend);
		}
		BigDecimal lhs = this;

		// If either number is zero then the other number, rounded and
		// scaled if necessary, is used as the result.
		{
			boolean lhsIsZero = lhs.signum() == 0;
			boolean augendIsZero = augend.signum() == 0;

			if (lhsIsZero || augendIsZero) {
				int preferredScale = Math.max(lhs.scale(), augend.scale());
				BigDecimal result;

				if (lhsIsZero && augendIsZero) {
					return zeroValueOf(preferredScale);
				}
				result = lhsIsZero ? doRound(augend, mc) : doRound(lhs, mc);

				if (result.scale() == preferredScale) {
					return result;
				} else if (result.scale() > preferredScale) {
					return stripZerosToMatchScale(result.intVal, result.intCompact, result.scale, preferredScale);
				} else { // result.scale < preferredScale
					int precisionDiff = mc.precision - result.precision();
					int scaleDiff = preferredScale - result.scale();

					if (precisionDiff >= scaleDiff) {
						return result.setScale(preferredScale); // can achieve target scale
					} else {
						return result.setScale(result.scale() + precisionDiff);
					}
				}
			}
		}

		long padding = (long) lhs.scale - augend.scale;
		if (padding != 0) { // scales differ; alignment needed
			BigDecimal arg[] = preAlign(lhs, augend, padding, mc);
			matchScale(arg);
			lhs = arg[0];
			augend = arg[1];
		}
		return doRound(lhs.inflated().add(augend.inflated()), lhs.scale, mc);
	}

	/**
	 * Returns an array of length two, the sum of whose entries is equal to the rounded sum of the {@code BigDecimal}
	 * arguments.
	 *
	 * <p>
	 * If the digit positions of the arguments have a sufficient gap between them, the value smaller in magnitude can be
	 * condensed into a {@literal "sticky bit"} and the end result will round the same way <em>if</em> the precision of
	 * the final result does not include the high order digit of the small magnitude operand.
	 *
	 * <p>
	 * Note that while strictly speaking this is an optimization, it makes a much wider range of additions practical.
	 *
	 * <p>
	 * This corresponds to a pre-shift operation in a fixed precision floating-point adder; this method is complicated
	 * by variable precision of the result as determined by the MathContext. A more nuanced operation could implement a
	 * {@literal "right shift"} on the smaller magnitude operand so that the number of digits of the smaller operand
	 * could be reduced even though the significands partially overlapped.
	 */
	private BigDecimal[] preAlign(BigDecimal lhs, BigDecimal augend, long padding, MathContext mc) {
		assert padding != 0;
		BigDecimal big;
		BigDecimal small;

		if (padding < 0) { // lhs is big; augend is small
			big = lhs;
			small = augend;
		} else { // lhs is small; augend is big
			big = augend;
			small = lhs;
		}

		/*
		 * This is the estimated scale of an ulp of the result; it assumes that the result doesn't have a carry-out on a
		 * true add (e.g. 999 + 1 => 1000) or any subtractive cancellation on borrowing (e.g. 100 - 1.2 => 98.8)
		 */
		long estResultUlpScale = (long) big.scale - big.precision() + mc.precision;

		/*
		 * The low-order digit position of big is big.scale(). This is true regardless of whether big has a positive or
		 * negative scale. The high-order digit position of small is small.scale - (small.precision() - 1). To do the
		 * full condensation, the digit positions of big and small must be disjoint *and* the digit positions of small
		 * should not be directly visible in the result.
		 */
		long smallHighDigitPos = (long) small.scale - small.precision() + 1;
		if (smallHighDigitPos > big.scale + 2 && // big and small disjoint
				smallHighDigitPos > estResultUlpScale + 2) { // small digits not visible
			small = BigDecimal.valueOf(small.signum(), this.checkScale(Math.max(big.scale, estResultUlpScale) + 3));
		}

		// Since addition is symmetric, preserving input order in
		// returned operands doesn't matter
		BigDecimal[] result = { big, small };
		return result;
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this -
	 * subtrahend)}, and whose scale is {@code max(this.scale(),
	 * subtrahend.scale())}.
	 *
	 * @param subtrahend
	 *            value to be subtracted from this {@code BigDecimal}.
	 * @return {@code this - subtrahend}
	 */
	public BigDecimal subtract(BigDecimal subtrahend) {
		if (this.intCompact != INFLATED) {
			if ((subtrahend.intCompact != INFLATED)) {
				return add(this.intCompact, this.scale, -subtrahend.intCompact, subtrahend.scale);
			} else {
				return add(this.intCompact, this.scale, subtrahend.intVal.negate(), subtrahend.scale);
			}
		} else {
			if ((subtrahend.intCompact != INFLATED)) {
				// Pair of subtrahend values given before pair of
				// values from this BigDecimal to avoid need for
				// method overloading on the specialized add method
				return add(-subtrahend.intCompact, subtrahend.scale, this.intVal, this.scale);
			} else {
				return add(this.intVal, this.scale, subtrahend.intVal.negate(), subtrahend.scale);
			}
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this - subtrahend)}, with rounding according to the context
	 * settings.
	 *
	 * If {@code subtrahend} is zero then this, rounded if necessary, is used as the result. If this is zero then the
	 * result is {@code subtrahend.negate(mc)}.
	 *
	 * @param subtrahend
	 *            value to be subtracted from this {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @return {@code this - subtrahend}, rounded as necessary.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal subtract(BigDecimal subtrahend, MathContext mc) {
		if (mc.precision == 0) {
			return subtract(subtrahend);
		}
		// share the special rounding code in add()
		return add(subtrahend.negate(), mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is <tt>(this &times;
	 * multiplicand)</tt>, and whose scale is {@code (this.scale() +
	 * multiplicand.scale())}.
	 *
	 * @param multiplicand
	 *            value to be multiplied by this {@code BigDecimal}.
	 * @return {@code this * multiplicand}
	 */
	public BigDecimal multiply(BigDecimal multiplicand) {
		int productScale = checkScale((long) this.scale + multiplicand.scale);
		if (this.intCompact != INFLATED) {
			if ((multiplicand.intCompact != INFLATED)) {
				return multiply(this.intCompact, multiplicand.intCompact, productScale);
			} else {
				return multiply(this.intCompact, multiplicand.intVal, productScale);
			}
		} else {
			if ((multiplicand.intCompact != INFLATED)) {
				return multiply(multiplicand.intCompact, this.intVal, productScale);
			} else {
				return multiply(this.intVal, multiplicand.intVal, productScale);
			}
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is <tt>(this &times;
	 * multiplicand)</tt>, with rounding according to the context settings.
	 *
	 * @param multiplicand
	 *            value to be multiplied by this {@code BigDecimal}.
	 * @param mc
	 *            the context to use.
	 * @return {@code this * multiplicand}, rounded as necessary.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal multiply(BigDecimal multiplicand, MathContext mc) {
		if (mc.precision == 0) {
			return multiply(multiplicand);
		}
		int productScale = checkScale((long) this.scale + multiplicand.scale);
		if (this.intCompact != INFLATED) {
			if ((multiplicand.intCompact != INFLATED)) {
				return multiplyAndRound(this.intCompact, multiplicand.intCompact, productScale, mc);
			} else {
				return multiplyAndRound(this.intCompact, multiplicand.intVal, productScale, mc);
			}
		} else {
			if ((multiplicand.intCompact != INFLATED)) {
				return multiplyAndRound(multiplicand.intCompact, this.intVal, productScale, mc);
			} else {
				return multiplyAndRound(this.intVal, multiplicand.intVal, productScale, mc);
			}
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this /
	 * divisor)}, and whose scale is as specified. If rounding must be performed to generate a result with the specified
	 * scale, the specified rounding mode is applied.
	 *
	 * <p>
	 * The new {@link #divide(BigDecimal, int, RoundingMode)} method should be used in preference to this legacy method.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @param scale
	 *            scale of the {@code BigDecimal} quotient to be returned.
	 * @param roundingMode
	 *            rounding mode to apply.
	 * @return {@code this / divisor}
	 * @throws ArithmeticException
	 *             if {@code divisor} is zero, {@code roundingMode==ROUND_UNNECESSARY} and the specified scale is
	 *             insufficient to represent the result of the division exactly.
	 * @throws IllegalArgumentException
	 *             if {@code roundingMode} does not represent a valid rounding mode.
	 * @see #ROUND_UP
	 * @see #ROUND_DOWN
	 * @see #ROUND_CEILING
	 * @see #ROUND_FLOOR
	 * @see #ROUND_HALF_UP
	 * @see #ROUND_HALF_DOWN
	 * @see #ROUND_HALF_EVEN
	 * @see #ROUND_UNNECESSARY
	 */
	public BigDecimal divide(BigDecimal divisor, int scale, int roundingMode) {
		if (roundingMode < ROUND_UP || roundingMode > ROUND_UNNECESSARY) {
			throw new IllegalArgumentException("Invalid rounding mode");
		}
		if (this.intCompact != INFLATED) {
			if ((divisor.intCompact != INFLATED)) {
				return divide(this.intCompact, this.scale, divisor.intCompact, divisor.scale, scale, roundingMode);
			} else {
				return divide(this.intCompact, this.scale, divisor.intVal, divisor.scale, scale, roundingMode);
			}
		} else {
			if ((divisor.intCompact != INFLATED)) {
				return divide(this.intVal, this.scale, divisor.intCompact, divisor.scale, scale, roundingMode);
			} else {
				return divide(this.intVal, this.scale, divisor.intVal, divisor.scale, scale, roundingMode);
			}
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this /
	 * divisor)}, and whose scale is as specified. If rounding must be performed to generate a result with the specified
	 * scale, the specified rounding mode is applied.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @param scale
	 *            scale of the {@code BigDecimal} quotient to be returned.
	 * @param roundingMode
	 *            rounding mode to apply.
	 * @return {@code this / divisor}
	 * @throws ArithmeticException
	 *             if {@code divisor} is zero, {@code roundingMode==RoundingMode.UNNECESSARY} and the specified scale is
	 *             insufficient to represent the result of the division exactly.
	 * @since 1.5
	 */
	public BigDecimal divide(BigDecimal divisor, int scale, RoundingMode roundingMode) {
		return divide(divisor, scale, roundingMode.oldMode);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this /
	 * divisor)}, and whose scale is {@code this.scale()}. If rounding must be performed to generate a result with the
	 * given scale, the specified rounding mode is applied.
	 *
	 * <p>
	 * The new {@link #divide(BigDecimal, RoundingMode)} method should be used in preference to this legacy method.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @param roundingMode
	 *            rounding mode to apply.
	 * @return {@code this / divisor}
	 * @throws ArithmeticException
	 *             if {@code divisor==0}, or {@code roundingMode==ROUND_UNNECESSARY} and {@code this.scale()} is
	 *             insufficient to represent the result of the division exactly.
	 * @throws IllegalArgumentException
	 *             if {@code roundingMode} does not represent a valid rounding mode.
	 * @see #ROUND_UP
	 * @see #ROUND_DOWN
	 * @see #ROUND_CEILING
	 * @see #ROUND_FLOOR
	 * @see #ROUND_HALF_UP
	 * @see #ROUND_HALF_DOWN
	 * @see #ROUND_HALF_EVEN
	 * @see #ROUND_UNNECESSARY
	 */
	public BigDecimal divide(BigDecimal divisor, int roundingMode) {
		return this.divide(divisor, this.scale, roundingMode);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this /
	 * divisor)}, and whose scale is {@code this.scale()}. If rounding must be performed to generate a result with the
	 * given scale, the specified rounding mode is applied.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @param roundingMode
	 *            rounding mode to apply.
	 * @return {@code this / divisor}
	 * @throws ArithmeticException
	 *             if {@code divisor==0}, or {@code roundingMode==RoundingMode.UNNECESSARY} and {@code this.scale()} is
	 *             insufficient to represent the result of the division exactly.
	 * @since 1.5
	 */
	public BigDecimal divide(BigDecimal divisor, RoundingMode roundingMode) {
		return this.divide(divisor, this.scale, roundingMode.oldMode);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this /
	 * divisor)}, and whose preferred scale is {@code (this.scale() -
	 * divisor.scale())}; if the exact quotient cannot be represented (because it has a non-terminating decimal
	 * expansion) an {@code ArithmeticException} is thrown.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @throws ArithmeticException
	 *             if the exact quotient does not have a terminating decimal expansion
	 * @return {@code this / divisor}
	 * @since 1.5
	 * @author Joseph D. Darcy
	 */
	public BigDecimal divide(BigDecimal divisor) {
		/*
		 * Handle zero cases first.
		 */
		if (divisor.signum() == 0) { // x/0
			if (this.signum() == 0) {
				throw new ArithmeticException("Division undefined"); // NaN
			}
			throw new ArithmeticException("Division by zero");
		}

		// Calculate preferred scale
		int preferredScale = saturateLong((long) this.scale - divisor.scale);

		if (this.signum() == 0) {
			return zeroValueOf(preferredScale);
		} else {
			/*
			 * If the quotient this/divisor has a terminating decimal expansion, the expansion can have no more than
			 * (a.precision() + ceil(10*b.precision)/3) digits. Therefore, create a MathContext object with this
			 * precision and do a divide with the UNNECESSARY rounding mode.
			 */
			MathContext mc = new MathContext((int) Math
					.min(this.precision() + (long) Math.ceil(10.0 * divisor.precision() / 3.0), Integer.MAX_VALUE),
					RoundingMode.UNNECESSARY);
			BigDecimal quotient;
			try {
				quotient = this.divide(divisor, mc);
			} catch (ArithmeticException e) {
				throw new ArithmeticException(
						"Non-terminating decimal expansion; " + "no exact representable decimal result.");
			}

			int quotientScale = quotient.scale();

			// divide(BigDecimal, mc) tries to adjust the quotient to
			// the desired one by removing trailing zeros; since the
			// exact divide method does not have an explicit digit
			// limit, we can add zeros too.
			if (preferredScale > quotientScale) {
				return quotient.setScale(preferredScale, ROUND_UNNECESSARY);
			}

			return quotient;
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this /
	 * divisor)}, with rounding according to the context settings.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @param mc
	 *            the context to use.
	 * @return {@code this / divisor}, rounded as necessary.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY} or {@code mc.precision == 0}
	 *             and the quotient has a non-terminating decimal expansion.
	 * @since 1.5
	 */
	public BigDecimal divide(BigDecimal divisor, MathContext mc) {
		int mcp = mc.precision;
		if (mcp == 0) {
			return divide(divisor);
		}

		BigDecimal dividend = this;
		long preferredScale = (long) dividend.scale - divisor.scale;
		// Now calculate the answer. We use the existing
		// divide-and-round method, but as this rounds to scale we have
		// to normalize the values here to achieve the desired result.
		// For x/y we first handle y=0 and x=0, and then normalize x and
		// y to give x' and y' with the following constraints:
		// (a) 0.1 <= x' < 1
		// (b) x' <= y' < 10*x'
		// Dividing x'/y' with the required scale set to mc.precision then
		// will give a result in the range 0.1 to 1 rounded to exactly
		// the right number of digits (except in the case of a result of
		// 1.000... which can arise when x=y, or when rounding overflows
		// The 1.000... case will reduce properly to 1.
		if (divisor.signum() == 0) { // x/0
			if (dividend.signum() == 0) {
				throw new ArithmeticException("Division undefined"); // NaN
			}
			throw new ArithmeticException("Division by zero");
		}
		if (dividend.signum() == 0) {
			return zeroValueOf(saturateLong(preferredScale));
		}
		int xscale = dividend.precision();
		int yscale = divisor.precision();
		if (dividend.intCompact != INFLATED) {
			if (divisor.intCompact != INFLATED) {
				return divide(dividend.intCompact, xscale, divisor.intCompact, yscale, preferredScale, mc);
			} else {
				return divide(dividend.intCompact, xscale, divisor.intVal, yscale, preferredScale, mc);
			}
		} else {
			if (divisor.intCompact != INFLATED) {
				return divide(dividend.intVal, xscale, divisor.intCompact, yscale, preferredScale, mc);
			} else {
				return divide(dividend.intVal, xscale, divisor.intVal, yscale, preferredScale, mc);
			}
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is the integer part of the quotient {@code (this / divisor)} rounded
	 * down. The preferred scale of the result is {@code (this.scale() -
	 * divisor.scale())}.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @return The integer part of {@code this / divisor}.
	 * @throws ArithmeticException
	 *             if {@code divisor==0}
	 * @since 1.5
	 */
	public BigDecimal divideToIntegralValue(BigDecimal divisor) {
		// Calculate preferred scale
		int preferredScale = saturateLong((long) this.scale - divisor.scale);
		if (this.compareMagnitude(divisor) < 0) {
			// much faster when this << divisor
			return zeroValueOf(preferredScale);
		}

		if (this.signum() == 0 && divisor.signum() != 0) {
			return this.setScale(preferredScale, ROUND_UNNECESSARY);
		}

		// Perform a divide with enough digits to round to a correct
		// integer value; then remove any fractional digits

		int maxDigits = (int) Math.min(this.precision() + (long) Math.ceil(10.0 * divisor.precision() / 3.0)
				+ Math.abs((long) this.scale() - divisor.scale()) + 2, Integer.MAX_VALUE);
		BigDecimal quotient = this.divide(divisor, new MathContext(maxDigits, RoundingMode.DOWN));
		if (quotient.scale > 0) {
			quotient = quotient.setScale(0, RoundingMode.DOWN);
			quotient = stripZerosToMatchScale(quotient.intVal, quotient.intCompact, quotient.scale, preferredScale);
		}

		if (quotient.scale < preferredScale) {
			// pad with zeros if necessary
			quotient = quotient.setScale(preferredScale, ROUND_UNNECESSARY);
		}

		return quotient;
	}

	/**
	 * Returns a {@code BigDecimal} whose value is the integer part of {@code (this / divisor)}. Since the integer part
	 * of the exact quotient does not depend on the rounding mode, the rounding mode does not affect the values returned
	 * by this method. The preferred scale of the result is {@code (this.scale() - divisor.scale())}. An
	 * {@code ArithmeticException} is thrown if the integer part of the exact quotient needs more than
	 * {@code mc.precision} digits.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @param mc
	 *            the context to use.
	 * @return The integer part of {@code this / divisor}.
	 * @throws ArithmeticException
	 *             if {@code divisor==0}
	 * @throws ArithmeticException
	 *             if {@code mc.precision} {@literal >} 0 and the result requires a precision of more than
	 *             {@code mc.precision} digits.
	 * @since 1.5
	 * @author Joseph D. Darcy
	 */
	public BigDecimal divideToIntegralValue(BigDecimal divisor, MathContext mc) {
		if (mc.precision == 0 || // exact result
				(this.compareMagnitude(divisor) < 0)) {
			return divideToIntegralValue(divisor);
		}

		// Calculate preferred scale
		int preferredScale = saturateLong((long) this.scale - divisor.scale);

		/*
		 * Perform a normal divide to mc.precision digits. If the remainder has absolute value less than the divisor,
		 * the integer portion of the quotient fits into mc.precision digits. Next, remove any fractional digits from
		 * the quotient and adjust the scale to the preferred value.
		 */
		BigDecimal result = this.divide(divisor, new MathContext(mc.precision, RoundingMode.DOWN));

		if (result.scale() < 0) {
			/*
			 * Result is an integer. See if quotient represents the full integer portion of the exact quotient; if it
			 * does, the computed remainder will be less than the divisor.
			 */
			BigDecimal product = result.multiply(divisor);
			// If the quotient is the full integer value,
			// |dividend-product| < |divisor|.
			if (this.subtract(product).compareMagnitude(divisor) >= 0) {
				throw new ArithmeticException("Division impossible");
			}
		} else if (result.scale() > 0) {
			/*
			 * Integer portion of quotient will fit into precision digits; recompute quotient to scale 0 to avoid double
			 * rounding and then try to adjust, if necessary.
			 */
			result = result.setScale(0, RoundingMode.DOWN);
		}
		// else result.scale() == 0;

		int precisionDiff;
		if ((preferredScale > result.scale()) && (precisionDiff = mc.precision - result.precision()) > 0) {
			return result.setScale(result.scale() + Math.min(precisionDiff, preferredScale - result.scale));
		} else {
			return stripZerosToMatchScale(result.intVal, result.intCompact, result.scale, preferredScale);
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this % divisor)}.
	 *
	 * <p>
	 * The remainder is given by {@code this.subtract(this.divideToIntegralValue(divisor).multiply(divisor))}. Note that
	 * this is not the modulo operation (the result can be negative).
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @return {@code this % divisor}.
	 * @throws ArithmeticException
	 *             if {@code divisor==0}
	 * @since 1.5
	 */
	public BigDecimal remainder(BigDecimal divisor) {
		BigDecimal divrem[] = this.divideAndRemainder(divisor);
		return divrem[1];
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (this %
	 * divisor)}, with rounding according to the context settings. The {@code MathContext} settings affect the implicit
	 * divide used to compute the remainder. The remainder computation itself is by definition exact. Therefore, the
	 * remainder may contain more than {@code mc.getPrecision()} digits.
	 *
	 * <p>
	 * The remainder is given by {@code this.subtract(this.divideToIntegralValue(divisor,
	 * mc).multiply(divisor))}. Note that this is not the modulo operation (the result can be negative).
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided.
	 * @param mc
	 *            the context to use.
	 * @return {@code this % divisor}, rounded as necessary.
	 * @throws ArithmeticException
	 *             if {@code divisor==0}
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}, or {@code mc.precision}
	 *             {@literal >} 0 and the result of {@code this.divideToIntgralValue(divisor)} would require a precision
	 *             of more than {@code mc.precision} digits.
	 * @see #divideToIntegralValue(java.math.BigDecimal, java.math.MathContext)
	 * @since 1.5
	 */
	public BigDecimal remainder(BigDecimal divisor, MathContext mc) {
		BigDecimal divrem[] = this.divideAndRemainder(divisor, mc);
		return divrem[1];
	}

	/**
	 * Returns a two-element {@code BigDecimal} array containing the result of {@code divideToIntegralValue} followed by
	 * the result of {@code remainder} on the two operands.
	 *
	 * <p>
	 * Note that if both the integer quotient and remainder are needed, this method is faster than using the
	 * {@code divideToIntegralValue} and {@code remainder} methods separately because the division need only be carried
	 * out once.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided, and the remainder computed.
	 * @return a two element {@code BigDecimal} array: the quotient (the result of {@code divideToIntegralValue}) is the
	 *         initial element and the remainder is the final element.
	 * @throws ArithmeticException
	 *             if {@code divisor==0}
	 * @see #divideToIntegralValue(java.math.BigDecimal, java.math.MathContext)
	 * @see #remainder(java.math.BigDecimal, java.math.MathContext)
	 * @since 1.5
	 */
	public BigDecimal[] divideAndRemainder(BigDecimal divisor) {
		// we use the identity x = i * y + r to determine r
		BigDecimal[] result = new BigDecimal[2];

		result[0] = this.divideToIntegralValue(divisor);
		result[1] = this.subtract(result[0].multiply(divisor));
		return result;
	}

	/**
	 * Returns a two-element {@code BigDecimal} array containing the result of {@code divideToIntegralValue} followed by
	 * the result of {@code remainder} on the two operands calculated with rounding according to the context settings.
	 *
	 * <p>
	 * Note that if both the integer quotient and remainder are needed, this method is faster than using the
	 * {@code divideToIntegralValue} and {@code remainder} methods separately because the division need only be carried
	 * out once.
	 *
	 * @param divisor
	 *            value by which this {@code BigDecimal} is to be divided, and the remainder computed.
	 * @param mc
	 *            the context to use.
	 * @return a two element {@code BigDecimal} array: the quotient (the result of {@code divideToIntegralValue}) is the
	 *         initial element and the remainder is the final element.
	 * @throws ArithmeticException
	 *             if {@code divisor==0}
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}, or {@code mc.precision}
	 *             {@literal >} 0 and the result of {@code this.divideToIntgralValue(divisor)} would require a precision
	 *             of more than {@code mc.precision} digits.
	 * @see #divideToIntegralValue(java.math.BigDecimal, java.math.MathContext)
	 * @see #remainder(java.math.BigDecimal, java.math.MathContext)
	 * @since 1.5
	 */
	public BigDecimal[] divideAndRemainder(BigDecimal divisor, MathContext mc) {
		if (mc.precision == 0) {
			return divideAndRemainder(divisor);
		}

		BigDecimal[] result = new BigDecimal[2];
		BigDecimal lhs = this;

		result[0] = lhs.divideToIntegralValue(divisor, mc);
		result[1] = lhs.subtract(result[0].multiply(divisor));
		return result;
	}

	/**
	 * Returns a {@code BigDecimal} whose value is <tt>(this<sup>n</sup>)</tt>, The power is computed exactly, to
	 * unlimited precision.
	 *
	 * <p>
	 * The parameter {@code n} must be in the range 0 through 999999999, inclusive. {@code ZERO.pow(0)} returns
	 * {@link #ONE}.
	 *
	 * Note that future releases may expand the allowable exponent range of this method.
	 *
	 * @param n
	 *            power to raise this {@code BigDecimal} to.
	 * @return <tt>this<sup>n</sup></tt>
	 * @throws ArithmeticException
	 *             if {@code n} is out of range.
	 * @since 1.5
	 */
	public BigDecimal pow(int n) {
		if (n < 0 || n > 999999999) {
			throw new ArithmeticException("Invalid operation");
		}
		// No need to calculate pow(n) if result will over/underflow.
		// Don't attempt to support "supernormal" numbers.
		int newScale = checkScale((long) this.scale * n);
		return new BigDecimal(this.inflated().pow(n), newScale);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is <tt>(this<sup>n</sup>)</tt>. The current implementation uses the core
	 * algorithm defined in ANSI standard X3.274-1996 with rounding according to the context settings. In general, the
	 * returned numerical value is within two ulps of the exact numerical value for the chosen precision. Note that
	 * future releases may use a different algorithm with a decreased allowable error bound and increased allowable
	 * exponent range.
	 *
	 * <p>
	 * The X3.274-1996 algorithm is:
	 *
	 * <ul>
	 * <li>An {@code ArithmeticException} exception is thrown if
	 * <ul>
	 * <li>{@code abs(n) > 999999999}
	 * <li>{@code mc.precision == 0} and {@code n < 0}
	 * <li>{@code mc.precision > 0} and {@code n} has more than {@code mc.precision} decimal digits
	 * </ul>
	 *
	 * <li>if {@code n} is zero, {@link #ONE} is returned even if {@code this} is zero, otherwise
	 * <ul>
	 * <li>if {@code n} is positive, the result is calculated via the repeated squaring technique into a single
	 * accumulator. The individual multiplications with the accumulator use the same math context settings as in
	 * {@code mc} except for a precision increased to {@code mc.precision + elength + 1} where {@code elength} is the
	 * number of decimal digits in {@code n}.
	 *
	 * <li>if {@code n} is negative, the result is calculated as if {@code n} were positive; this value is then divided
	 * into one using the working precision specified above.
	 *
	 * <li>The final value from either the positive or negative case is then rounded to the destination precision.
	 * </ul>
	 * </ul>
	 *
	 * @param n
	 *            power to raise this {@code BigDecimal} to.
	 * @param mc
	 *            the context to use.
	 * @return <tt>this<sup>n</sup></tt> using the ANSI standard X3.274-1996 algorithm
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}, or {@code n} is out of range.
	 * @since 1.5
	 */
	public BigDecimal pow(int n, MathContext mc) {
		if (mc.precision == 0) {
			return pow(n);
		}
		if (n < -999999999 || n > 999999999) {
			throw new ArithmeticException("Invalid operation");
		}
		if (n == 0) {
			return ONE; // x**0 == 1 in X3.274
		}
		BigDecimal lhs = this;
		MathContext workmc = mc; // working settings
		int mag = Math.abs(n); // magnitude of n
		if (mc.precision > 0) {
			int elength = longDigitLength(mag); // length of n in digits
			if (elength > mc.precision) {
				throw new ArithmeticException("Invalid operation");
			}
			workmc = new MathContext(mc.precision + elength + 1, mc.roundingMode);
		}
		// ready to carry out power calculation...
		BigDecimal acc = ONE; // accumulator
		boolean seenbit = false; // set once we've seen a 1-bit
		for (int i = 1;; i++) { // for each bit [top bit ignored]
			mag += mag; // shift left 1 bit
			if (mag < 0) { // top bit is set
				seenbit = true; // OK, we're off
				acc = acc.multiply(lhs, workmc); // acc=acc*x
			}
			if (i == 31) {
				break; // that was the last bit
			}
			if (seenbit) {
				acc = acc.multiply(acc, workmc); // acc=acc*acc [square]
				// else (!seenbit) no point in squaring ONE
			}
		}
		// if negative n, calculate the reciprocal using working precision
		if (n < 0) {
			acc = ONE.divide(acc, workmc);
		}
		// round to final precision and strip zeros
		return doRound(acc, mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is the absolute value of this {@code BigDecimal}, and whose scale is
	 * {@code this.scale()}.
	 *
	 * @return {@code abs(this)}
	 */
	public BigDecimal abs() {
		return (signum() < 0 ? negate() : this);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is the absolute value of this {@code BigDecimal}, with rounding
	 * according to the context settings.
	 *
	 * @param mc
	 *            the context to use.
	 * @return {@code abs(this)}, rounded as necessary.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal abs(MathContext mc) {
		return (signum() < 0 ? negate(mc) : plus(mc));
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (-this)}, and whose scale is {@code this.scale()}.
	 *
	 * @return {@code -this}.
	 */
	public BigDecimal negate() {
		if (this.intCompact == INFLATED) {
			return new BigDecimal(this.intVal.negate(), INFLATED, this.scale, this.precision);
		} else {
			return valueOf(-this.intCompact, this.scale, this.precision);
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (-this)}, with rounding according to the context settings.
	 *
	 * @param mc
	 *            the context to use.
	 * @return {@code -this}, rounded as necessary.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @since 1.5
	 */
	public BigDecimal negate(MathContext mc) {
		return negate().plus(mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (+this)}, and whose scale is {@code this.scale()}.
	 *
	 * <p>
	 * This method, which simply returns this {@code BigDecimal} is included for symmetry with the unary minus method
	 * {@link #negate()}.
	 *
	 * @return {@code this}.
	 * @see #negate()
	 * @since 1.5
	 */
	public BigDecimal plus() {
		return this;
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (+this)}, with rounding according to the context settings.
	 *
	 * <p>
	 * The effect of this method is identical to that of the {@link #round(MathContext)} method.
	 *
	 * @param mc
	 *            the context to use.
	 * @return {@code this}, rounded as necessary. A zero result will have a scale of 0.
	 * @throws ArithmeticException
	 *             if the result is inexact but the rounding mode is {@code UNNECESSARY}.
	 * @see #round(MathContext)
	 * @since 1.5
	 */
	public BigDecimal plus(MathContext mc) {
		if (mc.precision == 0) {
			return this;
		}
		return doRound(this, mc);
	}

	/**
	 * Returns the signum function of this {@code BigDecimal}.
	 *
	 * @return -1, 0, or 1 as the value of this {@code BigDecimal} is negative, zero, or positive.
	 */
	public int signum() {
		return (this.intCompact != INFLATED) ? Long.signum(this.intCompact) : this.intVal.signum();
	}

	/**
	 * Returns the <i>scale</i> of this {@code BigDecimal}. If zero or positive, the scale is the number of digits to
	 * the right of the decimal point. If negative, the unscaled value of the number is multiplied by ten to the power
	 * of the negation of the scale. For example, a scale of {@code -3} means the unscaled value is multiplied by 1000.
	 *
	 * @return the scale of this {@code BigDecimal}.
	 */
	public int scale() {
		return this.scale;
	}

	/**
	 * Returns the <i>precision</i> of this {@code BigDecimal}. (The precision is the number of digits in the unscaled
	 * value.)
	 *
	 * <p>
	 * The precision of a zero value is 1.
	 *
	 * @return the precision of this {@code BigDecimal}.
	 * @since 1.5
	 */
	public int precision() {
		int result = this.precision;
		if (result == 0) {
			long s = this.intCompact;
			if (s != INFLATED) {
				result = longDigitLength(s);
			} else {
				result = bigDigitLength(this.intVal);
			}
			this.precision = result;
		}
		return result;
	}

	/**
	 * Returns a {@code BigInteger} whose value is the <i>unscaled value</i> of this {@code BigDecimal}. (Computes
	 * <tt>(this *
	 * 10<sup>this.scale()</sup>)</tt>.)
	 *
	 * @return the unscaled value of this {@code BigDecimal}.
	 * @since 1.2
	 */
	public BigInteger unscaledValue() {
		return this.inflated();
	}

	// Rounding Modes

	/**
	 * Rounding mode to round away from zero. Always increments the digit prior to a nonzero discarded fraction. Note
	 * that this rounding mode never decreases the magnitude of the calculated value.
	 */
	public final static int ROUND_UP = 0;

	/**
	 * Rounding mode to round towards zero. Never increments the digit prior to a discarded fraction (i.e., truncates).
	 * Note that this rounding mode never increases the magnitude of the calculated value.
	 */
	public final static int ROUND_DOWN = 1;

	/**
	 * Rounding mode to round towards positive infinity. If the {@code BigDecimal} is positive, behaves as for
	 * {@code ROUND_UP}; if negative, behaves as for {@code ROUND_DOWN}. Note that this rounding mode never decreases
	 * the calculated value.
	 */
	public final static int ROUND_CEILING = 2;

	/**
	 * Rounding mode to round towards negative infinity. If the {@code BigDecimal} is positive, behave as for
	 * {@code ROUND_DOWN}; if negative, behave as for {@code ROUND_UP}. Note that this rounding mode never increases the
	 * calculated value.
	 */
	public final static int ROUND_FLOOR = 3;

	/**
	 * Rounding mode to round towards {@literal "nearest neighbor"} unless both neighbors are equidistant, in which case
	 * round up. Behaves as for {@code ROUND_UP} if the discarded fraction is &ge; 0.5; otherwise, behaves as for
	 * {@code ROUND_DOWN}. Note that this is the rounding mode that most of us were taught in grade school.
	 */
	public final static int ROUND_HALF_UP = 4;

	/**
	 * Rounding mode to round towards {@literal "nearest neighbor"} unless both neighbors are equidistant, in which case
	 * round down. Behaves as for {@code ROUND_UP} if the discarded fraction is {@literal >} 0.5; otherwise, behaves as
	 * for {@code ROUND_DOWN}.
	 */
	public final static int ROUND_HALF_DOWN = 5;

	/**
	 * Rounding mode to round towards the {@literal "nearest neighbor"} unless both neighbors are equidistant, in which
	 * case, round towards the even neighbor. Behaves as for {@code ROUND_HALF_UP} if the digit to the left of the
	 * discarded fraction is odd; behaves as for {@code ROUND_HALF_DOWN} if it's even. Note that this is the rounding
	 * mode that minimizes cumulative error when applied repeatedly over a sequence of calculations.
	 */
	public final static int ROUND_HALF_EVEN = 6;

	/**
	 * Rounding mode to assert that the requested operation has an exact result, hence no rounding is necessary. If this
	 * rounding mode is specified on an operation that yields an inexact result, an {@code ArithmeticException} is
	 * thrown.
	 */
	public final static int ROUND_UNNECESSARY = 7;

	// Scaling/Rounding Operations

	/**
	 * Returns a {@code BigDecimal} rounded according to the {@code MathContext} settings. If the precision setting is 0
	 * then no rounding takes place.
	 *
	 * <p>
	 * The effect of this method is identical to that of the {@link #plus(MathContext)} method.
	 *
	 * @param mc
	 *            the context to use.
	 * @return a {@code BigDecimal} rounded according to the {@code MathContext} settings.
	 * @throws ArithmeticException
	 *             if the rounding mode is {@code UNNECESSARY} and the {@code BigDecimal} operation would require
	 *             rounding.
	 * @see #plus(MathContext)
	 * @since 1.5
	 */
	public BigDecimal round(MathContext mc) {
		return plus(mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose scale is the specified value, and whose unscaled value is determined by
	 * multiplying or dividing this {@code BigDecimal}'s unscaled value by the appropriate power of ten to maintain its
	 * overall value. If the scale is reduced by the operation, the unscaled value must be divided (rather than
	 * multiplied), and the value may be changed; in this case, the specified rounding mode is applied to the division.
	 *
	 * <p>
	 * Note that since BigDecimal objects are immutable, calls of this method do <i>not</i> result in the original
	 * object being modified, contrary to the usual convention of having methods named <tt>set<i>X</i></tt> mutate field
	 * <i>{@code X}</i>. Instead, {@code setScale} returns an object with the proper scale; the returned object may or
	 * may not be newly allocated.
	 *
	 * @param newScale
	 *            scale of the {@code BigDecimal} value to be returned.
	 * @param roundingMode
	 *            The rounding mode to apply.
	 * @return a {@code BigDecimal} whose scale is the specified value, and whose unscaled value is determined by
	 *         multiplying or dividing this {@code BigDecimal}'s unscaled value by the appropriate power of ten to
	 *         maintain its overall value.
	 * @throws ArithmeticException
	 *             if {@code roundingMode==UNNECESSARY} and the specified scaling operation would require rounding.
	 * @see RoundingMode
	 * @since 1.5
	 */
	public BigDecimal setScale(int newScale, RoundingMode roundingMode) {
		return setScale(newScale, roundingMode.oldMode);
	}

	/**
	 * Returns a {@code BigDecimal} whose scale is the specified value, and whose unscaled value is determined by
	 * multiplying or dividing this {@code BigDecimal}'s unscaled value by the appropriate power of ten to maintain its
	 * overall value. If the scale is reduced by the operation, the unscaled value must be divided (rather than
	 * multiplied), and the value may be changed; in this case, the specified rounding mode is applied to the division.
	 *
	 * <p>
	 * Note that since BigDecimal objects are immutable, calls of this method do <i>not</i> result in the original
	 * object being modified, contrary to the usual convention of having methods named <tt>set<i>X</i></tt> mutate field
	 * <i>{@code X}</i>. Instead, {@code setScale} returns an object with the proper scale; the returned object may or
	 * may not be newly allocated.
	 *
	 * <p>
	 * The new {@link #setScale(int, RoundingMode)} method should be used in preference to this legacy method.
	 *
	 * @param newScale
	 *            scale of the {@code BigDecimal} value to be returned.
	 * @param roundingMode
	 *            The rounding mode to apply.
	 * @return a {@code BigDecimal} whose scale is the specified value, and whose unscaled value is determined by
	 *         multiplying or dividing this {@code BigDecimal}'s unscaled value by the appropriate power of ten to
	 *         maintain its overall value.
	 * @throws ArithmeticException
	 *             if {@code roundingMode==ROUND_UNNECESSARY} and the specified scaling operation would require
	 *             rounding.
	 * @throws IllegalArgumentException
	 *             if {@code roundingMode} does not represent a valid rounding mode.
	 * @see #ROUND_UP
	 * @see #ROUND_DOWN
	 * @see #ROUND_CEILING
	 * @see #ROUND_FLOOR
	 * @see #ROUND_HALF_UP
	 * @see #ROUND_HALF_DOWN
	 * @see #ROUND_HALF_EVEN
	 * @see #ROUND_UNNECESSARY
	 */
	public BigDecimal setScale(int newScale, int roundingMode) {
		if (roundingMode < ROUND_UP || roundingMode > ROUND_UNNECESSARY) {
			throw new IllegalArgumentException("Invalid rounding mode");
		}

		int oldScale = this.scale;
		if (newScale == oldScale) {
			return this;
		}
		if (this.signum() == 0) {
			return zeroValueOf(newScale);
		}
		if (this.intCompact != INFLATED) {
			long rs = this.intCompact;
			if (newScale > oldScale) {
				int raise = checkScale((long) newScale - oldScale);
				if ((rs = longMultiplyPowerTen(rs, raise)) != INFLATED) {
					return valueOf(rs, newScale);
				}
				BigInteger rb = bigMultiplyPowerTen(raise);
				return new BigDecimal(rb, INFLATED, newScale, (this.precision > 0) ? this.precision + raise : 0);
			} else {
				// newScale < oldScale -- drop some digits
				// Can't predict the precision due to the effect of rounding.
				int drop = checkScale((long) oldScale - newScale);
				if (drop < LONG_TEN_POWERS_TABLE.length) {
					return divideAndRound(rs, LONG_TEN_POWERS_TABLE[drop], newScale, roundingMode, newScale);
				} else {
					return divideAndRound(this.inflated(), bigTenToThe(drop), newScale, roundingMode, newScale);
				}
			}
		} else {
			if (newScale > oldScale) {
				int raise = checkScale((long) newScale - oldScale);
				BigInteger rb = bigMultiplyPowerTen(this.intVal, raise);
				return new BigDecimal(rb, INFLATED, newScale, (this.precision > 0) ? this.precision + raise : 0);
			} else {
				// newScale < oldScale -- drop some digits
				// Can't predict the precision due to the effect of rounding.
				int drop = checkScale((long) oldScale - newScale);
				if (drop < LONG_TEN_POWERS_TABLE.length) {
					return divideAndRound(this.intVal, LONG_TEN_POWERS_TABLE[drop], newScale, roundingMode, newScale);
				} else {
					return divideAndRound(this.intVal, bigTenToThe(drop), newScale, roundingMode, newScale);
				}
			}
		}
	}

	/**
	 * Returns a {@code BigDecimal} whose scale is the specified value, and whose value is numerically equal to this
	 * {@code BigDecimal}'s. Throws an {@code ArithmeticException} if this is not possible.
	 *
	 * <p>
	 * This call is typically used to increase the scale, in which case it is guaranteed that there exists a
	 * {@code BigDecimal} of the specified scale and the correct value. The call can also be used to reduce the scale if
	 * the caller knows that the {@code BigDecimal} has sufficiently many zeros at the end of its fractional part (i.e.,
	 * factors of ten in its integer value) to allow for the rescaling without changing its value.
	 *
	 * <p>
	 * This method returns the same result as the two-argument versions of {@code setScale}, but saves the caller the
	 * trouble of specifying a rounding mode in cases where it is irrelevant.
	 *
	 * <p>
	 * Note that since {@code BigDecimal} objects are immutable, calls of this method do <i>not</i> result in the
	 * original object being modified, contrary to the usual convention of having methods named <tt>set<i>X</i></tt>
	 * mutate field <i>{@code X}</i>. Instead, {@code setScale} returns an object with the proper scale; the returned
	 * object may or may not be newly allocated.
	 *
	 * @param newScale
	 *            scale of the {@code BigDecimal} value to be returned.
	 * @return a {@code BigDecimal} whose scale is the specified value, and whose unscaled value is determined by
	 *         multiplying or dividing this {@code BigDecimal}'s unscaled value by the appropriate power of ten to
	 *         maintain its overall value.
	 * @throws ArithmeticException
	 *             if the specified scaling operation would require rounding.
	 * @see #setScale(int, int)
	 * @see #setScale(int, RoundingMode)
	 */
	public BigDecimal setScale(int newScale) {
		return setScale(newScale, ROUND_UNNECESSARY);
	}

	// Decimal Point Motion Operations

	/**
	 * Returns a {@code BigDecimal} which is equivalent to this one with the decimal point moved {@code n} places to the
	 * left. If {@code n} is non-negative, the call merely adds {@code n} to the scale. If {@code n} is negative, the
	 * call is equivalent to {@code movePointRight(-n)}. The {@code BigDecimal} returned by this call has value
	 * <tt>(this &times;
	 * 10<sup>-n</sup>)</tt> and scale {@code max(this.scale()+n,
	 * 0)}.
	 *
	 * @param n
	 *            number of places to move the decimal point to the left.
	 * @return a {@code BigDecimal} which is equivalent to this one with the decimal point moved {@code n} places to the
	 *         left.
	 * @throws ArithmeticException
	 *             if scale overflows.
	 */
	public BigDecimal movePointLeft(int n) {
		// Cannot use movePointRight(-n) in case of n==Integer.MIN_VALUE
		int newScale = checkScale((long) this.scale + n);
		BigDecimal num = new BigDecimal(this.intVal, this.intCompact, newScale, 0);
		return num.scale < 0 ? num.setScale(0, ROUND_UNNECESSARY) : num;
	}

	/**
	 * Returns a {@code BigDecimal} which is equivalent to this one with the decimal point moved {@code n} places to the
	 * right. If {@code n} is non-negative, the call merely subtracts {@code n} from the scale. If {@code n} is
	 * negative, the call is equivalent to {@code movePointLeft(-n)}. The {@code BigDecimal} returned by this call has
	 * value <tt>(this
	 * &times; 10<sup>n</sup>)</tt> and scale {@code max(this.scale()-n,
	 * 0)}.
	 *
	 * @param n
	 *            number of places to move the decimal point to the right.
	 * @return a {@code BigDecimal} which is equivalent to this one with the decimal point moved {@code n} places to the
	 *         right.
	 * @throws ArithmeticException
	 *             if scale overflows.
	 */
	public BigDecimal movePointRight(int n) {
		// Cannot use movePointLeft(-n) in case of n==Integer.MIN_VALUE
		int newScale = checkScale((long) this.scale - n);
		BigDecimal num = new BigDecimal(this.intVal, this.intCompact, newScale, 0);
		return num.scale < 0 ? num.setScale(0, ROUND_UNNECESSARY) : num;
	}

	/**
	 * Returns a BigDecimal whose numerical value is equal to ({@code this} * 10<sup>n</sup>). The scale of the result
	 * is {@code (this.scale() - n)}.
	 *
	 * @param n
	 *            the exponent power of ten to scale by
	 * @return a BigDecimal whose numerical value is equal to ({@code this} * 10<sup>n</sup>)
	 * @throws ArithmeticException
	 *             if the scale would be outside the range of a 32-bit integer.
	 *
	 * @since 1.5
	 */
	public BigDecimal scaleByPowerOfTen(int n) {
		return new BigDecimal(this.intVal, this.intCompact, checkScale((long) this.scale - n), this.precision);
	}

	/**
	 * Returns a {@code BigDecimal} which is numerically equal to this one but with any trailing zeros removed from the
	 * representation. For example, stripping the trailing zeros from the {@code BigDecimal} value {@code 600.0}, which
	 * has [{@code BigInteger}, {@code scale}] components equals to [6000, 1], yields {@code 6E2} with [
	 * {@code BigInteger}, {@code scale}] components equals to [6, -2]. If this BigDecimal is numerically equal to zero,
	 * then {@code BigDecimal.ZERO} is returned.
	 *
	 * @return a numerically equal {@code BigDecimal} with any trailing zeros removed.
	 * @since 1.5
	 */
	public BigDecimal stripTrailingZeros() {
		if (this.intCompact == 0 || (this.intVal != null && this.intVal.signum() == 0)) {
			return BigDecimal.ZERO;
		} else if (this.intCompact != INFLATED) {
			return createAndStripZerosToMatchScale(this.intCompact, this.scale, Long.MIN_VALUE);
		} else {
			return createAndStripZerosToMatchScale(this.intVal, this.scale, Long.MIN_VALUE);
		}
	}

	// Comparison Operations

	/**
	 * Compares this {@code BigDecimal} with the specified {@code BigDecimal}. Two {@code BigDecimal} objects that are
	 * equal in value but have a different scale (like 2.0 and 2.00) are considered equal by this method. This method is
	 * provided in preference to individual methods for each of the six boolean comparison operators ({@literal <}, ==,
	 * {@literal >}, {@literal >=}, !=, {@literal <=}). The suggested idiom for performing these comparisons is:
	 * {@code (x.compareTo(y)} &lt;<i>op</i>&gt; {@code 0)}, where &lt;<i>op</i>&gt; is one of the six comparison
	 * operators.
	 *
	 * @param val
	 *            {@code BigDecimal} to which this {@code BigDecimal} is to be compared.
	 * @return -1, 0, or 1 as this {@code BigDecimal} is numerically less than, equal to, or greater than {@code val}.
	 */
	@Override
	public int compareTo(BigDecimal val) {
		// Quick path for equal scale and non-inflated case.
		if (this.scale == val.scale) {
			long xs = this.intCompact;
			long ys = val.intCompact;
			if (xs != INFLATED && ys != INFLATED) {
				return xs != ys ? ((xs > ys) ? 1 : -1) : 0;
			}
		}
		int xsign = this.signum();
		int ysign = val.signum();
		if (xsign != ysign) {
			return (xsign > ysign) ? 1 : -1;
		}
		if (xsign == 0) {
			return 0;
		}
		int cmp = compareMagnitude(val);
		return (xsign > 0) ? cmp : -cmp;
	}

	/**
	 * Version of compareTo that ignores sign.
	 */
	private int compareMagnitude(BigDecimal val) {
		// Match scales, avoid unnecessary inflation
		long ys = val.intCompact;
		long xs = this.intCompact;
		if (xs == 0) {
			return (ys == 0) ? 0 : -1;
		}
		if (ys == 0) {
			return 1;
		}

		long sdiff = (long) this.scale - val.scale;
		if (sdiff != 0) {
			// Avoid matching scales if the (adjusted) exponents differ
			long xae = (long) this.precision() - this.scale; // [-1]
			long yae = (long) val.precision() - val.scale; // [-1]
			if (xae < yae) {
				return -1;
			}
			if (xae > yae) {
				return 1;
			}
			BigInteger rb = null;
			if (sdiff < 0) {
				// The cases sdiff <= Integer.MIN_VALUE intentionally fall through.
				if (sdiff > Integer.MIN_VALUE
						&& (xs == INFLATED || (xs = longMultiplyPowerTen(xs, (int) -sdiff)) == INFLATED)
						&& ys == INFLATED) {
					rb = bigMultiplyPowerTen((int) -sdiff);
					return rb.compareMagnitude(val.intVal);
				}
			} else { // sdiff > 0
				// The cases sdiff > Integer.MAX_VALUE intentionally fall through.
				if (sdiff <= Integer.MAX_VALUE
						&& (ys == INFLATED || (ys = longMultiplyPowerTen(ys, (int) sdiff)) == INFLATED)
						&& xs == INFLATED) {
					rb = val.bigMultiplyPowerTen((int) sdiff);
					return this.intVal.compareMagnitude(rb);
				}
			}
		}
		if (xs != INFLATED) {
			return (ys != INFLATED) ? longCompareMagnitude(xs, ys) : -1;
		} else if (ys != INFLATED) {
			return 1;
		} else {
			return this.intVal.compareMagnitude(val.intVal);
		}
	}

	/**
	 * Compares this {@code BigDecimal} with the specified {@code Object} for equality. Unlike
	 * {@link #compareTo(BigDecimal) compareTo}, this method considers two {@code BigDecimal} objects equal only if they
	 * are equal in value and scale (thus 2.0 is not equal to 2.00 when compared by this method).
	 *
	 * @param x
	 *            {@code Object} to which this {@code BigDecimal} is to be compared.
	 * @return {@code true} if and only if the specified {@code Object} is a {@code BigDecimal} whose value and scale
	 *         are equal to this {@code BigDecimal}'s.
	 * @see #compareTo(java.math.BigDecimal)
	 * @see #hashCode
	 */
	@Override
	public boolean equals(Object x) {
		if (!(x instanceof BigDecimal)) {
			return false;
		}
		BigDecimal xDec = (BigDecimal) x;
		if (x == this) {
			return true;
		}
		if (this.scale != xDec.scale) {
			return false;
		}
		long s = this.intCompact;
		long xs = xDec.intCompact;
		if (s != INFLATED) {
			if (xs == INFLATED) {
				xs = compactValFor(xDec.intVal);
			}
			return xs == s;
		} else if (xs != INFLATED) {
			return xs == compactValFor(this.intVal);
		}

		return this.inflated().equals(xDec.inflated());
	}

	/**
	 * Returns the minimum of this {@code BigDecimal} and {@code val}.
	 *
	 * @param val
	 *            value with which the minimum is to be computed.
	 * @return the {@code BigDecimal} whose value is the lesser of this {@code BigDecimal} and {@code val}. If they are
	 *         equal, as defined by the {@link #compareTo(BigDecimal) compareTo} method, {@code this} is returned.
	 * @see #compareTo(java.math.BigDecimal)
	 */
	public BigDecimal min(BigDecimal val) {
		return (compareTo(val) <= 0 ? this : val);
	}

	/**
	 * Returns the maximum of this {@code BigDecimal} and {@code val}.
	 *
	 * @param val
	 *            value with which the maximum is to be computed.
	 * @return the {@code BigDecimal} whose value is the greater of this {@code BigDecimal} and {@code val}. If they are
	 *         equal, as defined by the {@link #compareTo(BigDecimal) compareTo} method, {@code this} is returned.
	 * @see #compareTo(java.math.BigDecimal)
	 */
	public BigDecimal max(BigDecimal val) {
		return (compareTo(val) >= 0 ? this : val);
	}

	// Hash Function

	/**
	 * Returns the hash code for this {@code BigDecimal}. Note that two {@code BigDecimal} objects that are numerically
	 * equal but differ in scale (like 2.0 and 2.00) will generally <i>not</i> have the same hash code.
	 *
	 * @return hash code for this {@code BigDecimal}.
	 * @see #equals(Object)
	 */
	@Override
	public int hashCode() {
		if (this.intCompact != INFLATED) {
			long val2 = (this.intCompact < 0) ? -this.intCompact : this.intCompact;
			int temp = (int) (((int) (val2 >>> 32)) * 31 + (val2 & LONG_MASK));
			return 31 * ((this.intCompact < 0) ? -temp : temp) + this.scale;
		} else {
			return 31 * this.intVal.hashCode() + this.scale;
		}
	}

	// Format Converters

	/**
	 * Returns the string representation of this {@code BigDecimal}, using scientific notation if an exponent is needed.
	 *
	 * <p>
	 * A standard canonical string form of the {@code BigDecimal} is created as though by the following steps: first,
	 * the absolute value of the unscaled value of the {@code BigDecimal} is converted to a string in base ten using the
	 * characters {@code '0'} through {@code '9'} with no leading zeros (except if its value is zero, in which case a
	 * single {@code '0'} character is used).
	 *
	 * <p>
	 * Next, an <i>adjusted exponent</i> is calculated; this is the negated scale, plus the number of characters in the
	 * converted unscaled value, less one. That is, {@code -scale+(ulength-1)}, where {@code ulength} is the length of
	 * the absolute value of the unscaled value in decimal digits (its <i>precision</i>).
	 *
	 * <p>
	 * If the scale is greater than or equal to zero and the adjusted exponent is greater than or equal to {@code -6},
	 * the number will be converted to a character form without using exponential notation. In this case, if the scale
	 * is zero then no decimal point is added and if the scale is positive a decimal point will be inserted with the
	 * scale specifying the number of characters to the right of the decimal point. {@code '0'} characters are added to
	 * the left of the converted unscaled value as necessary. If no character precedes the decimal point after this
	 * insertion then a conventional {@code '0'} character is prefixed.
	 *
	 * <p>
	 * Otherwise (that is, if the scale is negative, or the adjusted exponent is less than {@code -6}), the number will
	 * be converted to a character form using exponential notation. In this case, if the converted {@code BigInteger}
	 * has more than one digit a decimal point is inserted after the first digit. An exponent in character form is then
	 * suffixed to the converted unscaled value (perhaps with inserted decimal point); this comprises the letter
	 * {@code 'E'} followed immediately by the adjusted exponent converted to a character form. The latter is in base
	 * ten, using the characters {@code '0'} through {@code '9'} with no leading zeros, and is always prefixed by a sign
	 * character {@code '-'} (<tt>'&#92;u002D'</tt>) if the adjusted exponent is negative, {@code '+'} (
	 * <tt>'&#92;u002B'</tt>) otherwise).
	 *
	 * <p>
	 * Finally, the entire string is prefixed by a minus sign character {@code '-'} (<tt>'&#92;u002D'</tt>) if the
	 * unscaled value is less than zero. No sign character is prefixed if the unscaled value is zero or positive.
	 *
	 * <p>
	 * <b>Examples:</b>
	 * <p>
	 * For each representation [<i>unscaled value</i>, <i>scale</i>] on the left, the resulting string is shown on the
	 * right.
	 *
	 * <pre>
	 * [123,0]      "123"
	 * [-123,0]     "-123"
	 * [123,-1]     "1.23E+3"
	 * [123,-3]     "1.23E+5"
	 * [123,1]      "12.3"
	 * [123,5]      "0.00123"
	 * [123,10]     "1.23E-8"
	 * [-123,12]    "-1.23E-10"
	 * </pre>
	 *
	 * <b>Notes:</b>
	 * <ol>
	 *
	 * <li>There is a one-to-one mapping between the distinguishable {@code BigDecimal} values and the result of this
	 * conversion. That is, every distinguishable {@code BigDecimal} value (unscaled value and scale) has a unique
	 * string representation as a result of using {@code toString}. If that string representation is converted back to a
	 * {@code BigDecimal} using the {@link #BigDecimal(String)} constructor, then the original value will be recovered.
	 *
	 * <li>The string produced for a given number is always the same; it is not affected by locale. This means that it
	 * can be used as a canonical string representation for exchanging decimal data, or as a key for a Hashtable, etc.
	 * Locale-sensitive number formatting and parsing is handled by the {java.text.NumberFormat} class and its
	 * subclasses.
	 *
	 * <li>The {@link #toEngineeringString} method may be used for presenting numbers with exponents in engineering
	 * notation, and the {@link #setScale(int,RoundingMode) setScale} method may be used for rounding a
	 * {@code BigDecimal} so it has a known number of digits after the decimal point.
	 *
	 * <li>The digit-to-character mapping provided by {@code Character.forDigit} is used.
	 *
	 * </ol>
	 *
	 * @return string representation of this {@code BigDecimal}.
	 * @see Character#forDigit
	 * @see #BigDecimal(java.lang.String)
	 */
	@Override
	public String toString() {
		String sc = this.stringCache;
		if (sc == null) {
			this.stringCache = sc = layoutChars(true);
		}
		return sc;
	}

	/**
	 * Returns a string representation of this {@code BigDecimal}, using engineering notation if an exponent is needed.
	 *
	 * <p>
	 * Returns a string that represents the {@code BigDecimal} as described in the {@link #toString()} method, except
	 * that if exponential notation is used, the power of ten is adjusted to be a multiple of three (engineering
	 * notation) such that the integer part of nonzero values will be in the range 1 through 999. If exponential
	 * notation is used for zero values, a decimal point and one or two fractional zero digits are used so that the
	 * scale of the zero value is preserved. Note that unlike the output of {@link #toString()}, the output of this
	 * method is <em>not</em> guaranteed to recover the same [integer, scale] pair of this {@code BigDecimal} if the
	 * output string is converting back to a {@code BigDecimal} using the {@linkplain #BigDecimal(String) string
	 * constructor}. The result of this method meets the weaker constraint of always producing a numerically equal
	 * result from applying the string constructor to the method's output.
	 *
	 * @return string representation of this {@code BigDecimal}, using engineering notation if an exponent is needed.
	 * @since 1.5
	 */
	public String toEngineeringString() {
		return layoutChars(false);
	}

	/**
	 * Returns a string representation of this {@code BigDecimal} without an exponent field. For values with a positive
	 * scale, the number of digits to the right of the decimal point is used to indicate scale. For values with a zero
	 * or negative scale, the resulting string is generated as if the value were converted to a numerically equal value
	 * with zero scale and as if all the trailing zeros of the zero scale value were present in the result.
	 *
	 * The entire string is prefixed by a minus sign character '-' (<tt>'&#92;u002D'</tt>) if the unscaled value is less
	 * than zero. No sign character is prefixed if the unscaled value is zero or positive.
	 *
	 * Note that if the result of this method is passed to the {@linkplain #BigDecimal(String) string constructor}, only
	 * the numerical value of this {@code BigDecimal} will necessarily be recovered; the representation of the new
	 * {@code BigDecimal} may have a different scale. In particular, if this {@code BigDecimal} has a negative scale,
	 * the string resulting from this method will have a scale of zero when processed by the string constructor.
	 *
	 * (This method behaves analogously to the {@code toString} method in 1.4 and earlier releases.)
	 *
	 * @return a string representation of this {@code BigDecimal} without an exponent field.
	 * @since 1.5
	 * @see #toString()
	 * @see #toEngineeringString()
	 */
	public String toPlainString() {
		if (this.scale == 0) {
			if (this.intCompact != INFLATED) {
				return Long.toString(this.intCompact);
			} else {
				return this.intVal.toString();
			}
		}
		if (this.scale < 0) { // No decimal point
			if (signum() == 0) {
				return "0";
			}
			int tailingZeros = checkScaleNonZero((-(long) this.scale));
			StringBuilder buf;
			if (this.intCompact != INFLATED) {
				buf = new StringBuilder(20 + tailingZeros);
				buf.append(this.intCompact);
			} else {
				String str = this.intVal.toString();
				buf = new StringBuilder(str.length() + tailingZeros);
				buf.append(str);
			}
			for (int i = 0; i < tailingZeros; i++) {
				buf.append('0');
			}
			return buf.toString();
		}
		String str;
		if (this.intCompact != INFLATED) {
			str = Long.toString(Math.abs(this.intCompact));
		} else {
			str = this.intVal.abs().toString();
		}
		return getValueString(signum(), str, this.scale);
	}

	/* Returns a digit.digit string */
	private String getValueString(int signum, String intString, int scale) {
		/* Insert decimal point */
		StringBuilder buf;
		int insertionPoint = intString.length() - scale;
		if (insertionPoint == 0) { /* Point goes right before intVal */
			return (signum < 0 ? "-0." : "0.") + intString;
		} else if (insertionPoint > 0) { /* Point goes inside intVal */
			buf = new StringBuilder(intString);
			buf.insert(insertionPoint, '.');
			if (signum < 0) {
				buf.insert(0, '-');
			}
		} else { /* We must insert zeros between point and intVal */
			buf = new StringBuilder(3 - insertionPoint + intString.length());
			buf.append(signum < 0 ? "-0." : "0.");
			for (int i = 0; i < -insertionPoint; i++) {
				buf.append('0');
			}
			buf.append(intString);
		}
		return buf.toString();
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code BigInteger}. This conversion is analogous to the <i>narrowing
	 * primitive conversion</i> from {@code double} to {@code long} as defined in section 5.1.3 of <cite>The Java&trade;
	 * Language Specification</cite>: any fractional part of this {@code BigDecimal} will be discarded. Note that this
	 * conversion can lose information about the precision of the {@code BigDecimal} value.
	 * <p>
	 * To have an exception thrown if the conversion is inexact (in other words if a nonzero fractional part is
	 * discarded), use the {@link #toBigIntegerExact()} method.
	 *
	 * @return this {@code BigDecimal} converted to a {@code BigInteger}.
	 */
	public BigInteger toBigInteger() {
		// force to an integer, quietly
		return this.setScale(0, ROUND_DOWN).inflated();
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code BigInteger}, checking for lost information. An exception is thrown
	 * if this {@code BigDecimal} has a nonzero fractional part.
	 *
	 * @return this {@code BigDecimal} converted to a {@code BigInteger}.
	 * @throws ArithmeticException
	 *             if {@code this} has a nonzero fractional part.
	 * @since 1.5
	 */
	public BigInteger toBigIntegerExact() {
		// round to an integer, with Exception if decimal part non-0
		return this.setScale(0, ROUND_UNNECESSARY).inflated();
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code long}. This conversion is analogous to the <i>narrowing primitive
	 * conversion</i> from {@code double} to {@code short} as defined in section 5.1.3 of <cite>The Java&trade; Language
	 * Specification</cite>: any fractional part of this {@code BigDecimal} will be discarded, and if the resulting
	 * "{@code BigInteger}" is too big to fit in a {@code long}, only the low-order 64 bits are returned. Note that this
	 * conversion can lose information about the overall magnitude and precision of this {@code BigDecimal} value as
	 * well as return a result with the opposite sign.
	 *
	 * @return this {@code BigDecimal} converted to a {@code long}.
	 */
	@Override
	public long longValue() {
		return (this.intCompact != INFLATED && this.scale == 0) ? this.intCompact : toBigInteger().longValue();
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code long}, checking for lost information. If this {@code BigDecimal} has
	 * a nonzero fractional part or is out of the possible range for a {@code long} result then an
	 * {@code ArithmeticException} is thrown.
	 *
	 * @return this {@code BigDecimal} converted to a {@code long}.
	 * @throws ArithmeticException
	 *             if {@code this} has a nonzero fractional part, or will not fit in a {@code long}.
	 * @since 1.5
	 */
	public long longValueExact() {
		if (this.intCompact != INFLATED && this.scale == 0) {
			return this.intCompact;
		}
		// If more than 19 digits in integer part it cannot possibly fit
		if ((precision() - this.scale) > 19) {
			throw new java.lang.ArithmeticException("Overflow");
		}
		// Fastpath zero and < 1.0 numbers (the latter can be very slow
		// to round if very small)
		if (this.signum() == 0) {
			return 0;
		}
		if ((this.precision() - this.scale) <= 0) {
			throw new ArithmeticException("Rounding necessary");
		}
		// round to an integer, with Exception if decimal part non-0
		BigDecimal num = this.setScale(0, ROUND_UNNECESSARY);
		if (num.precision() >= 19) {
			LongOverflow.check(num);
		}
		return num.inflated().longValue();
	}

	private static class LongOverflow {
		/** BigInteger equal to Long.MIN_VALUE. */
		private static final BigInteger LONGMIN = BigInteger.valueOf(Long.MIN_VALUE);

		/** BigInteger equal to Long.MAX_VALUE. */
		private static final BigInteger LONGMAX = BigInteger.valueOf(Long.MAX_VALUE);

		public static void check(BigDecimal num) {
			BigInteger intVal = num.inflated();
			if (intVal.compareTo(LONGMIN) < 0 || intVal.compareTo(LONGMAX) > 0) {
				throw new java.lang.ArithmeticException("Overflow");
			}
		}
	}

	/**
	 * Converts this {@code BigDecimal} to an {@code int}. This conversion is analogous to the <i>narrowing primitive
	 * conversion</i> from {@code double} to {@code short} as defined in section 5.1.3 of <cite>The Java&trade; Language
	 * Specification</cite>: any fractional part of this {@code BigDecimal} will be discarded, and if the resulting
	 * "{@code BigInteger}" is too big to fit in an {@code int}, only the low-order 32 bits are returned. Note that this
	 * conversion can lose information about the overall magnitude and precision of this {@code BigDecimal} value as
	 * well as return a result with the opposite sign.
	 *
	 * @return this {@code BigDecimal} converted to an {@code int}.
	 */
	@Override
	public int intValue() {
		return (this.intCompact != INFLATED && this.scale == 0) ? (int) this.intCompact : toBigInteger().intValue();
	}

	/**
	 * Converts this {@code BigDecimal} to an {@code int}, checking for lost information. If this {@code BigDecimal} has
	 * a nonzero fractional part or is out of the possible range for an {@code int} result then an
	 * {@code ArithmeticException} is thrown.
	 *
	 * @return this {@code BigDecimal} converted to an {@code int}.
	 * @throws ArithmeticException
	 *             if {@code this} has a nonzero fractional part, or will not fit in an {@code int}.
	 * @since 1.5
	 */
	public int intValueExact() {
		long num;
		num = this.longValueExact(); // will check decimal part
		if ((int) num != num) {
			throw new java.lang.ArithmeticException("Overflow");
		}
		return (int) num;
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code short}, checking for lost information. If this {@code BigDecimal}
	 * has a nonzero fractional part or is out of the possible range for a {@code short} result then an
	 * {@code ArithmeticException} is thrown.
	 *
	 * @return this {@code BigDecimal} converted to a {@code short}.
	 * @throws ArithmeticException
	 *             if {@code this} has a nonzero fractional part, or will not fit in a {@code short}.
	 * @since 1.5
	 */
	public short shortValueExact() {
		long num;
		num = this.longValueExact(); // will check decimal part
		if ((short) num != num) {
			throw new java.lang.ArithmeticException("Overflow");
		}
		return (short) num;
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code byte}, checking for lost information. If this {@code BigDecimal} has
	 * a nonzero fractional part or is out of the possible range for a {@code byte} result then an
	 * {@code ArithmeticException} is thrown.
	 *
	 * @return this {@code BigDecimal} converted to a {@code byte}.
	 * @throws ArithmeticException
	 *             if {@code this} has a nonzero fractional part, or will not fit in a {@code byte}.
	 * @since 1.5
	 */
	public byte byteValueExact() {
		long num;
		num = this.longValueExact(); // will check decimal part
		if ((byte) num != num) {
			throw new java.lang.ArithmeticException("Overflow");
		}
		return (byte) num;
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code float}. This conversion is similar to the <i>narrowing primitive
	 * conversion</i> from {@code double} to {@code float} as defined in section 5.1.3 of <cite>The Java&trade; Language
	 * Specification</cite>: if this {@code BigDecimal} has too great a magnitude to represent as a {@code float}, it
	 * will be converted to {@link Float#NEGATIVE_INFINITY} or {@link Float#POSITIVE_INFINITY} as appropriate. Note that
	 * even when the return value is finite, this conversion can lose information about the precision of the
	 * {@code BigDecimal} value.
	 *
	 * @return this {@code BigDecimal} converted to a {@code float}.
	 */
	@Override
	public float floatValue() {
		if (this.intCompact != INFLATED) {
			if (this.scale == 0) {
				return this.intCompact;
			} else {
				/*
				 * If both intCompact and the scale can be exactly represented as float values, perform a single float
				 * multiply or divide to compute the (properly rounded) result.
				 */
				if (Math.abs(this.intCompact) < 1L << 22) {
					// Don't have too guard against
					// Math.abs(MIN_VALUE) because of outer check
					// against INFLATED.
					if (this.scale > 0 && this.scale < float10pow.length) {
						return this.intCompact / float10pow[this.scale];
					} else if (this.scale < 0 && this.scale > -float10pow.length) {
						return this.intCompact * float10pow[-this.scale];
					}
				}
			}
		}
		// Somewhat inefficient, but guaranteed to work.
		return Float.parseFloat(this.toString());
	}

	/**
	 * Converts this {@code BigDecimal} to a {@code double}. This conversion is similar to the <i>narrowing primitive
	 * conversion</i> from {@code double} to {@code float} as defined in section 5.1.3 of <cite>The Java&trade; Language
	 * Specification</cite>: if this {@code BigDecimal} has too great a magnitude represent as a {@code double}, it will
	 * be converted to {@link Double#NEGATIVE_INFINITY} or {@link Double#POSITIVE_INFINITY} as appropriate. Note that
	 * even when the return value is finite, this conversion can lose information about the precision of the
	 * {@code BigDecimal} value.
	 *
	 * @return this {@code BigDecimal} converted to a {@code double}.
	 */
	@Override
	public double doubleValue() {
		if (this.intCompact != INFLATED) {
			if (this.scale == 0) {
				return this.intCompact;
			} else {
				/*
				 * If both intCompact and the scale can be exactly represented as double values, perform a single double
				 * multiply or divide to compute the (properly rounded) result.
				 */
				if (Math.abs(this.intCompact) < 1L << 52) {
					// Don't have too guard against
					// Math.abs(MIN_VALUE) because of outer check
					// against INFLATED.
					if (this.scale > 0 && this.scale < double10pow.length) {
						return this.intCompact / double10pow[this.scale];
					} else if (this.scale < 0 && this.scale > -double10pow.length) {
						return this.intCompact * double10pow[-this.scale];
					}
				}
			}
		}
		// Somewhat inefficient, but guaranteed to work.
		return Double.parseDouble(this.toString());
	}

	/**
	 * Powers of 10 which can be represented exactly in {@code
	 * double}.
	 */
	private static final double double10pow[] = { 1.0e0, 1.0e1, 1.0e2, 1.0e3, 1.0e4, 1.0e5, 1.0e6, 1.0e7, 1.0e8, 1.0e9,
			1.0e10, 1.0e11, 1.0e12, 1.0e13, 1.0e14, 1.0e15, 1.0e16, 1.0e17, 1.0e18, 1.0e19, 1.0e20, 1.0e21, 1.0e22 };

	/**
	 * Powers of 10 which can be represented exactly in {@code
	 * float}.
	 */
	private static final float float10pow[] = { 1.0e0f, 1.0e1f, 1.0e2f, 1.0e3f, 1.0e4f, 1.0e5f, 1.0e6f, 1.0e7f, 1.0e8f,
			1.0e9f, 1.0e10f };

	/**
	 * Returns the size of an ulp, a unit in the last place, of this {@code BigDecimal}. An ulp of a nonzero
	 * {@code BigDecimal} value is the positive distance between this value and the {@code BigDecimal} value next larger
	 * in magnitude with the same number of digits. An ulp of a zero value is numerically equal to 1 with the scale of
	 * {@code this}. The result is stored with the same scale as {@code this} so the result for zero and nonzero values
	 * is equal to {@code [1,
	 * this.scale()]}.
	 *
	 * @return the size of an ulp of {@code this}
	 * @since 1.5
	 */
	public BigDecimal ulp() {
		return BigDecimal.valueOf(1, this.scale(), 1);
	}

	// Private class to build a string representation for BigDecimal object.
	// "StringBuilderHelper" is constructed as a thread local variable so it is
	// thread safe. The StringBuilder field acts as a buffer to hold the temporary
	// representation of BigDecimal. The cmpCharArray holds all the characters for
	// the compact representation of BigDecimal (except for '-' sign' if it is
	// negative) if its intCompact field is not INFLATED. It is shared by all
	// calls to toString() and its variants in that particular thread.
	static class StringBuilderHelper {
		final StringBuilder sb; // Placeholder for BigDecimal string
		final char[] cmpCharArray; // character array to place the intCompact

		StringBuilderHelper() {
			this.sb = new StringBuilder();
			// All non negative longs can be made to fit into 19 character array.
			this.cmpCharArray = new char[19];
		}

		// Accessors.
		StringBuilder getStringBuilder() {
			this.sb.setLength(0);
			return this.sb;
		}

		char[] getCompactCharArray() {
			return this.cmpCharArray;
		}

		/**
		 * Places characters representing the intCompact in {@code long} into cmpCharArray and returns the offset to the
		 * array where the representation starts.
		 *
		 * @param intCompact
		 *            the number to put into the cmpCharArray.
		 * @return offset to the array where the representation starts. Note: intCompact must be greater or equal to
		 *         zero.
		 */
		int putIntCompact(long intCompact) {
			assert intCompact >= 0;

			long q;
			int r;
			// since we start from the least significant digit, charPos points to
			// the last character in cmpCharArray.
			int charPos = this.cmpCharArray.length;

			// Get 2 digits/iteration using longs until quotient fits into an int
			while (intCompact > Integer.MAX_VALUE) {
				q = intCompact / 100;
				r = (int) (intCompact - q * 100);
				intCompact = q;
				this.cmpCharArray[--charPos] = DIGIT_ONES[r];
				this.cmpCharArray[--charPos] = DIGIT_TENS[r];
			}

			// Get 2 digits/iteration using ints when i2 >= 100
			int q2;
			int i2 = (int) intCompact;
			while (i2 >= 100) {
				q2 = i2 / 100;
				r = i2 - q2 * 100;
				i2 = q2;
				this.cmpCharArray[--charPos] = DIGIT_ONES[r];
				this.cmpCharArray[--charPos] = DIGIT_TENS[r];
			}

			this.cmpCharArray[--charPos] = DIGIT_ONES[i2];
			if (i2 >= 10) {
				this.cmpCharArray[--charPos] = DIGIT_TENS[i2];
			}

			return charPos;
		}

		final static char[] DIGIT_TENS = { '0', '0', '0', '0', '0', '0', '0', '0', '0', '0', '1', '1', '1', '1', '1',
				'1', '1', '1', '1', '1', '2', '2', '2', '2', '2', '2', '2', '2', '2', '2', '3', '3', '3', '3', '3', '3',
				'3', '3', '3', '3', '4', '4', '4', '4', '4', '4', '4', '4', '4', '4', '5', '5', '5', '5', '5', '5', '5',
				'5', '5', '5', '6', '6', '6', '6', '6', '6', '6', '6', '6', '6', '7', '7', '7', '7', '7', '7', '7', '7',
				'7', '7', '8', '8', '8', '8', '8', '8', '8', '8', '8', '8', '9', '9', '9', '9', '9', '9', '9', '9', '9',
				'9', };

		final static char[] DIGIT_ONES = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4',
				'5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5',
				'6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6',
				'7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7',
				'8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '1', '2', '3', '4', '5', '6', '7', '8',
				'9', };
	}

	/**
	 * Lay out this {@code BigDecimal} into a {@code char[]} array. The Java 1.2 equivalent to this was called
	 * {@code getValueString}.
	 *
	 * @param sci
	 *            {@code true} for Scientific exponential notation; {@code false} for Engineering
	 * @return string with canonical string representation of this {@code BigDecimal}
	 */
	private String layoutChars(boolean sci) {
		if (this.scale == 0) {
			return (this.intCompact != INFLATED) ? Long.toString(this.intCompact) : this.intVal.toString();
		}
		if (this.scale == 2 && this.intCompact >= 0 && this.intCompact < Integer.MAX_VALUE) {
			// currency fast path
			int lowInt = (int) this.intCompact % 100;
			int highInt = (int) this.intCompact / 100;
			return (Integer.toString(highInt) + '.' + StringBuilderHelper.DIGIT_TENS[lowInt]
					+ StringBuilderHelper.DIGIT_ONES[lowInt]);
		}

		StringBuilderHelper sbHelper = new StringBuilderHelper();
		char[] coeff;
		int offset; // offset is the starting index for coeff array
		// Get the significand as an absolute value
		if (this.intCompact != INFLATED) {
			offset = sbHelper.putIntCompact(Math.abs(this.intCompact));
			coeff = sbHelper.getCompactCharArray();
		} else {
			offset = 0;
			coeff = this.intVal.abs().toString().toCharArray();
		}

		// Construct a buffer, with sufficient capacity for all cases.
		// If E-notation is needed, length will be: +1 if negative, +1
		// if '.' needed, +2 for "E+", + up to 10 for adjusted exponent.
		// Otherwise it could have +1 if negative, plus leading "0.00000"
		StringBuilder buf = sbHelper.getStringBuilder();
		if (signum() < 0) {
			buf.append('-');
		}
		int coeffLen = coeff.length - offset;
		long adjusted = -(long) this.scale + (coeffLen - 1);
		if ((this.scale >= 0) && (adjusted >= -6)) { // plain number
			int pad = this.scale - coeffLen; // count of padding zeros
			if (pad >= 0) { // 0.xxx form
				buf.append('0');
				buf.append('.');
				for (; pad > 0; pad--) {
					buf.append('0');
				}
				buf.append(coeff, offset, coeffLen);
			} else { // xx.xx form
				buf.append(coeff, offset, -pad);
				buf.append('.');
				buf.append(coeff, -pad + offset, this.scale);
			}
		} else { // E-notation is needed
			if (sci) { // Scientific notation
				buf.append(coeff[offset]); // first character
				if (coeffLen > 1) { // more to come
					buf.append('.');
					buf.append(coeff, offset + 1, coeffLen - 1);
				}
			} else { // Engineering notation
				int sig = (int) (adjusted % 3);
				if (sig < 0) {
					sig += 3; // [adjusted was negative]
				}
				adjusted -= sig; // now a multiple of 3
				sig++;
				if (signum() == 0) {
					switch (sig) {
					case 1:
						buf.append('0'); // exponent is a multiple of three
						break;
					case 2:
						buf.append("0.00");
						adjusted += 3;
						break;
					case 3:
						buf.append("0.0");
						adjusted += 3;
						break;
					default:
						throw new AssertionError("Unexpected sig value " + sig);
					}
				} else if (sig >= coeffLen) { // significand all in integer
					buf.append(coeff, offset, coeffLen);
					// may need some zeros, too
					for (int i = sig - coeffLen; i > 0; i--) {
						buf.append('0');
					}
				} else { // xx.xxE form
					buf.append(coeff, offset, sig);
					buf.append('.');
					buf.append(coeff, offset + sig, coeffLen - sig);
				}
			}
			if (adjusted != 0) { // [!sci could have made 0]
				buf.append('E');
				if (adjusted > 0) {
					buf.append('+');
				}
				buf.append(adjusted);
			}
		}
		return buf.toString();
	}

	/**
	 * Return 10 to the power n, as a {@code BigInteger}.
	 *
	 * @param n
	 *            the power of ten to be returned (>=0)
	 * @return a {@code BigInteger} with the value (10<sup>n</sup>)
	 */
	private static BigInteger bigTenToThe(int n) {
		if (n < 0) {
			return BigInteger.ZERO;
		}

		if (n < BIG_TEN_POWERS_TABLE_MAX) {
			BigInteger[] pows = BIG_TEN_POWERS_TABLE;
			if (n < pows.length) {
				return pows[n];
			} else {
				return expandBigIntegerTenPowers(n);
			}
		}

		return BigInteger.TEN.pow(n);
	}

	/**
	 * Expand the BIG_TEN_POWERS_TABLE array to contain at least 10**n.
	 *
	 * @param n
	 *            the power of ten to be returned (>=0)
	 * @return a {@code BigDecimal} with the value (10<sup>n</sup>) and in the meantime, the BIG_TEN_POWERS_TABLE array
	 *         gets expanded to the size greater than n.
	 */
	private static BigInteger expandBigIntegerTenPowers(int n) {
		synchronized (BigDecimal.class) {
			BigInteger[] pows = BIG_TEN_POWERS_TABLE;
			int curLen = pows.length;
			// The following comparison and the above synchronized statement is
			// to prevent multiple threads from expanding the same array.
			if (curLen <= n) {
				int newLen = curLen << 1;
				while (newLen <= n) {
					newLen <<= 1;
				}
				pows = BigInteger.Arrays_copyOfBigInteger(pows, newLen);
				for (int i = curLen; i < newLen; i++) {
					pows[i] = pows[i - 1].multiply(BigInteger.TEN);
				}
				// Based on the following facts:
				// 1. pows is a private local varible;
				// 2. the following store is a volatile store.
				// the newly created array elements can be safely published.
				BIG_TEN_POWERS_TABLE = pows;
			}
			return pows[n];
		}
	}

	private static final long[] LONG_TEN_POWERS_TABLE = { 1, // 0 / 10^0
			10, // 1 / 10^1
			100, // 2 / 10^2
			1000, // 3 / 10^3
			10000, // 4 / 10^4
			100000, // 5 / 10^5
			1000000, // 6 / 10^6
			10000000, // 7 / 10^7
			100000000, // 8 / 10^8
			1000000000, // 9 / 10^9
			10000000000L, // 10 / 10^10
			100000000000L, // 11 / 10^11
			1000000000000L, // 12 / 10^12
			10000000000000L, // 13 / 10^13
			100000000000000L, // 14 / 10^14
			1000000000000000L, // 15 / 10^15
			10000000000000000L, // 16 / 10^16
			100000000000000000L, // 17 / 10^17
			1000000000000000000L // 18 / 10^18
	};

	private static volatile BigInteger BIG_TEN_POWERS_TABLE[] = { BigInteger.ONE, BigInteger.valueOf(10),
			BigInteger.valueOf(100), BigInteger.valueOf(1000), BigInteger.valueOf(10000), BigInteger.valueOf(100000),
			BigInteger.valueOf(1000000), BigInteger.valueOf(10000000), BigInteger.valueOf(100000000),
			BigInteger.valueOf(1000000000), BigInteger.valueOf(10000000000L), BigInteger.valueOf(100000000000L),
			BigInteger.valueOf(1000000000000L), BigInteger.valueOf(10000000000000L),
			BigInteger.valueOf(100000000000000L), BigInteger.valueOf(1000000000000000L),
			BigInteger.valueOf(10000000000000000L), BigInteger.valueOf(100000000000000000L),
			BigInteger.valueOf(1000000000000000000L) };

	private static final int BIG_TEN_POWERS_TABLE_INITLEN = BIG_TEN_POWERS_TABLE.length;
	private static final int BIG_TEN_POWERS_TABLE_MAX = 16 * BIG_TEN_POWERS_TABLE_INITLEN;

	private static final long THRESHOLDS_TABLE[] = { Long.MAX_VALUE, // 0
			Long.MAX_VALUE / 10L, // 1
			Long.MAX_VALUE / 100L, // 2
			Long.MAX_VALUE / 1000L, // 3
			Long.MAX_VALUE / 10000L, // 4
			Long.MAX_VALUE / 100000L, // 5
			Long.MAX_VALUE / 1000000L, // 6
			Long.MAX_VALUE / 10000000L, // 7
			Long.MAX_VALUE / 100000000L, // 8
			Long.MAX_VALUE / 1000000000L, // 9
			Long.MAX_VALUE / 10000000000L, // 10
			Long.MAX_VALUE / 100000000000L, // 11
			Long.MAX_VALUE / 1000000000000L, // 12
			Long.MAX_VALUE / 10000000000000L, // 13
			Long.MAX_VALUE / 100000000000000L, // 14
			Long.MAX_VALUE / 1000000000000000L, // 15
			Long.MAX_VALUE / 10000000000000000L, // 16
			Long.MAX_VALUE / 100000000000000000L, // 17
			Long.MAX_VALUE / 1000000000000000000L // 18
	};

	/**
	 * Compute val * 10 ^ n; return this product if it is representable as a long, INFLATED otherwise.
	 */
	private static long longMultiplyPowerTen(long val, int n) {
		if (val == 0 || n <= 0) {
			return val;
		}
		long[] tab = LONG_TEN_POWERS_TABLE;
		long[] bounds = THRESHOLDS_TABLE;
		if (n < tab.length && n < bounds.length) {
			long tenpower = tab[n];
			if (val == 1) {
				return tenpower;
			}
			if (Math.abs(val) <= bounds[n]) {
				return val * tenpower;
			}
		}
		return INFLATED;
	}

	/**
	 * Compute this * 10 ^ n. Needed mainly to allow special casing to trap zero value
	 */
	private BigInteger bigMultiplyPowerTen(int n) {
		if (n <= 0) {
			return this.inflated();
		}

		if (this.intCompact != INFLATED) {
			return bigTenToThe(n).multiply(this.intCompact);
		} else {
			return this.intVal.multiply(bigTenToThe(n));
		}
	}

	/**
	 * Returns appropriate BigInteger from intVal field if intVal is null, i.e. the compact representation is in use.
	 */
	private BigInteger inflated() {
		if (this.intVal == null) {
			return BigInteger.valueOf(this.intCompact);
		}
		return this.intVal;
	}

	/**
	 * Match the scales of two {@code BigDecimal}s to align their least significant digits.
	 *
	 * <p>
	 * If the scales of val[0] and val[1] differ, rescale (non-destructively) the lower-scaled {@code BigDecimal} so
	 * they match. That is, the lower-scaled reference will be replaced by a reference to a new object with the same
	 * scale as the other {@code BigDecimal}.
	 *
	 * @param val
	 *            array of two elements referring to the two {@code BigDecimal}s to be aligned.
	 */
	private static void matchScale(BigDecimal[] val) {
		if (val[0].scale == val[1].scale) {
			return;
		} else if (val[0].scale < val[1].scale) {
			val[0] = val[0].setScale(val[1].scale, ROUND_UNNECESSARY);
		} else if (val[1].scale < val[0].scale) {
			val[1] = val[1].setScale(val[0].scale, ROUND_UNNECESSARY);
		}
	}

	/**
	 * Returns the length of the absolute value of a {@code long}, in decimal digits.
	 *
	 * @param x
	 *            the {@code long}
	 * @return the length of the unscaled value, in deciaml digits.
	 */
	static int longDigitLength(long x) {
		/*
		 * As described in "Bit Twiddling Hacks" by Sean Anderson, (http://graphics.stanford.edu/~seander/bithacks.html)
		 * integer log 10 of x is within 1 of (1233/4096)* (1 + integer log 2 of x). The fraction 1233/4096 approximates
		 * log10(2). So we first do a version of log2 (a variant of Long class with pre-checks and opposite
		 * directionality) and then scale and check against powers table. This is a little simpler in present context
		 * than the version in Hacker's Delight sec 11-4. Adding one to bit length allows comparing downward from the
		 * LONG_TEN_POWERS_TABLE that we need anyway.
		 */
		assert x != BigDecimal.INFLATED;
		if (x < 0) {
			x = -x;
		}
		if (x < 10) {
			return 1;
		}
		int r = ((64 - LongHelper.numberOfLeadingZeros(x) + 1) * 1233) >>> 12;
		long[] tab = LONG_TEN_POWERS_TABLE;
		// if r >= length, must have max possible digits for long
		return (r >= tab.length || x < tab[r]) ? r : r + 1;
	}

	/**
	 * Returns the length of the absolute value of a BigInteger, in decimal digits.
	 *
	 * @param b
	 *            the BigInteger
	 * @return the length of the unscaled value, in decimal digits
	 */
	private static int bigDigitLength(BigInteger b) {
		/*
		 * Same idea as the long version, but we need a better approximation of log10(2). Using 646456993/2^31 is
		 * accurate up to max possible reported bitLength.
		 */
		if (b.signum == 0) {
			return 1;
		}
		int r = (int) ((((long) b.bitLength() + 1) * 646456993) >>> 31);
		return b.compareMagnitude(bigTenToThe(r)) < 0 ? r : r + 1;
	}

	/**
	 * Check a scale for Underflow or Overflow. If this BigDecimal is nonzero, throw an exception if the scale is outof
	 * range. If this is zero, saturate the scale to the extreme value of the right sign if the scale is out of range.
	 *
	 * @param val
	 *            The new scale.
	 * @throws ArithmeticException
	 *             (overflow or underflow) if the new scale is out of range.
	 * @return validated scale as an int.
	 */
	private int checkScale(long val) {
		int asInt = (int) val;
		if (asInt != val) {
			asInt = val > Integer.MAX_VALUE ? Integer.MAX_VALUE : Integer.MIN_VALUE;
			BigInteger b;
			if (this.intCompact != 0 && ((b = this.intVal) == null || b.signum() != 0)) {
				throw new ArithmeticException(asInt > 0 ? "Underflow" : "Overflow");
			}
		}
		return asInt;
	}

	/**
	 * Returns the compact value for given {@code BigInteger}, or INFLATED if too big. Relies on internal representation
	 * of {@code BigInteger}.
	 */
	private static long compactValFor(BigInteger b) {
		int[] m = b.mag;
		int len = m.length;
		if (len == 0) {
			return 0;
		}
		int d = m[0];
		if (len > 2 || (len == 2 && d < 0)) {
			return INFLATED;
		}

		long u = (len == 2) ? ((m[1] & LONG_MASK) + (((long) d) << 32)) : ((d) & LONG_MASK);
		return (b.signum < 0) ? -u : u;
	}

	private static int longCompareMagnitude(long x, long y) {
		if (x < 0) {
			x = -x;
		}
		if (y < 0) {
			y = -y;
		}
		return (x < y) ? -1 : ((x == y) ? 0 : 1);
	}

	private static int saturateLong(long s) {
		int i = (int) s;
		return (s == i) ? i : (s < 0 ? Integer.MIN_VALUE : Integer.MAX_VALUE);
	}

	/* the same as checkScale where value!=0 */
	private static int checkScaleNonZero(long val) {
		int asInt = (int) val;
		if (asInt != val) {
			throw new ArithmeticException(asInt > 0 ? "Underflow" : "Overflow");
		}
		return asInt;
	}

	private static int checkScale(long intCompact, long val) {
		int asInt = (int) val;
		if (asInt != val) {
			asInt = val > Integer.MAX_VALUE ? Integer.MAX_VALUE : Integer.MIN_VALUE;
			if (intCompact != 0) {
				throw new ArithmeticException(asInt > 0 ? "Underflow" : "Overflow");
			}
		}
		return asInt;
	}

	private static int checkScale(BigInteger intVal, long val) {
		int asInt = (int) val;
		if (asInt != val) {
			asInt = val > Integer.MAX_VALUE ? Integer.MAX_VALUE : Integer.MIN_VALUE;
			if (intVal.signum() != 0) {
				throw new ArithmeticException(asInt > 0 ? "Underflow" : "Overflow");
			}
		}
		return asInt;
	}

	/**
	 * Returns a {@code BigDecimal} rounded according to the MathContext settings; If rounding is needed a new
	 * {@code BigDecimal} is created and returned.
	 *
	 * @param val
	 *            the value to be rounded
	 * @param mc
	 *            the context to use.
	 * @return a {@code BigDecimal} rounded according to the MathContext settings. May return {@code value}, if no
	 *         rounding needed.
	 * @throws ArithmeticException
	 *             if the rounding mode is {@code RoundingMode.UNNECESSARY} and the result is inexact.
	 */
	private static BigDecimal doRound(BigDecimal val, MathContext mc) {
		int mcp = mc.precision;
		boolean wasDivided = false;
		if (mcp > 0) {
			BigInteger intVal = val.intVal;
			long compactVal = val.intCompact;
			int scale = val.scale;
			int prec = val.precision();
			int mode = mc.roundingMode.oldMode;
			int drop;
			if (compactVal == INFLATED) {
				drop = prec - mcp;
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					intVal = divideAndRoundByTenPow(intVal, drop, mode);
					wasDivided = true;
					compactVal = compactValFor(intVal);
					if (compactVal != INFLATED) {
						prec = longDigitLength(compactVal);
						break;
					}
					prec = bigDigitLength(intVal);
					drop = prec - mcp;
				}
			}
			if (compactVal != INFLATED) {
				drop = prec - mcp; // drop can't be more than 18
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					compactVal = divideAndRound(compactVal, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
					wasDivided = true;
					prec = longDigitLength(compactVal);
					drop = prec - mcp;
					intVal = null;
				}
			}
			return wasDivided ? new BigDecimal(intVal, compactVal, scale, prec) : val;
		}
		return val;
	}

	/*
	 * Returns a {@code BigDecimal} created from {@code long} value with given scale rounded according to the
	 * MathContext settings
	 */
	private static BigDecimal doRound(long compactVal, int scale, MathContext mc) {
		int mcp = mc.precision;
		if (mcp > 0 && mcp < 19) {
			int prec = longDigitLength(compactVal);
			int drop = prec - mcp; // drop can't be more than 18
			while (drop > 0) {
				scale = checkScaleNonZero((long) scale - drop);
				compactVal = divideAndRound(compactVal, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
				prec = longDigitLength(compactVal);
				drop = prec - mcp;
			}
			return valueOf(compactVal, scale, prec);
		}
		return valueOf(compactVal, scale);
	}

	/*
	 * Returns a {@code BigDecimal} created from {@code BigInteger} value with given scale rounded according to the
	 * MathContext settings
	 */
	private static BigDecimal doRound(BigInteger intVal, int scale, MathContext mc) {
		int mcp = mc.precision;
		int prec = 0;
		if (mcp > 0) {
			long compactVal = compactValFor(intVal);
			int mode = mc.roundingMode.oldMode;
			int drop;
			if (compactVal == INFLATED) {
				prec = bigDigitLength(intVal);
				drop = prec - mcp;
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					intVal = divideAndRoundByTenPow(intVal, drop, mode);
					compactVal = compactValFor(intVal);
					if (compactVal != INFLATED) {
						break;
					}
					prec = bigDigitLength(intVal);
					drop = prec - mcp;
				}
			}
			if (compactVal != INFLATED) {
				prec = longDigitLength(compactVal);
				drop = prec - mcp; // drop can't be more than 18
				while (drop > 0) {
					scale = checkScaleNonZero((long) scale - drop);
					compactVal = divideAndRound(compactVal, LONG_TEN_POWERS_TABLE[drop], mc.roundingMode.oldMode);
					prec = longDigitLength(compactVal);
					drop = prec - mcp;
				}
				return valueOf(compactVal, scale, prec);
			}
		}
		return new BigDecimal(intVal, INFLATED, scale, prec);
	}

	/*
	 * Divides {@code BigInteger} value by ten power.
	 */
	private static BigInteger divideAndRoundByTenPow(BigInteger intVal, int tenPow, int roundingMode) {
		if (tenPow < LONG_TEN_POWERS_TABLE.length) {
			intVal = divideAndRound(intVal, LONG_TEN_POWERS_TABLE[tenPow], roundingMode);
		} else {
			intVal = divideAndRound(intVal, bigTenToThe(tenPow), roundingMode);
		}
		return intVal;
	}

	/**
	 * Internally used for division operation for division {@code long} by {@code long}. The returned {@code BigDecimal}
	 * object is the quotient whose scale is set to the passed in scale. If the remainder is not zero, it will be
	 * rounded based on the passed in roundingMode. Also, if the remainder is zero and the last parameter, i.e.
	 * preferredScale is NOT equal to scale, the trailing zeros of the result is stripped to match the preferredScale.
	 */
	private static BigDecimal divideAndRound(long ldividend, long ldivisor, int scale, int roundingMode,
			int preferredScale) {

		int qsign; // quotient sign
		long q = ldividend / ldivisor; // store quotient in long
		if (roundingMode == ROUND_DOWN && scale == preferredScale) {
			return valueOf(q, scale);
		}
		long r = ldividend % ldivisor; // store remainder in long
		qsign = ((ldividend < 0) == (ldivisor < 0)) ? 1 : -1;
		if (r != 0) {
			boolean increment = needIncrement(ldivisor, roundingMode, qsign, q, r);
			return valueOf((increment ? q + qsign : q), scale);
		} else {
			if (preferredScale != scale) {
				return createAndStripZerosToMatchScale(q, scale, preferredScale);
			} else {
				return valueOf(q, scale);
			}
		}
	}

	/**
	 * Divides {@code long} by {@code long} and do rounding based on the passed in roundingMode.
	 */
	private static long divideAndRound(long ldividend, long ldivisor, int roundingMode) {
		int qsign; // quotient sign
		long q = ldividend / ldivisor; // store quotient in long
		if (roundingMode == ROUND_DOWN) {
			return q;
		}
		long r = ldividend % ldivisor; // store remainder in long
		qsign = ((ldividend < 0) == (ldivisor < 0)) ? 1 : -1;
		if (r != 0) {
			boolean increment = needIncrement(ldivisor, roundingMode, qsign, q, r);
			return increment ? q + qsign : q;
		} else {
			return q;
		}
	}

	/**
	 * Shared logic of need increment computation.
	 */
	private static boolean commonNeedIncrement(int roundingMode, int qsign, int cmpFracHalf, boolean oddQuot) {
		switch (roundingMode) {
		case ROUND_UNNECESSARY:
			throw new ArithmeticException("Rounding necessary");

		case ROUND_UP: // Away from zero
			return true;

		case ROUND_DOWN: // Towards zero
			return false;

		case ROUND_CEILING: // Towards +infinity
			return qsign > 0;

		case ROUND_FLOOR: // Towards -infinity
			return qsign < 0;

		default: // Some kind of half-way rounding
			assert roundingMode >= ROUND_HALF_UP && roundingMode <= ROUND_HALF_EVEN : "Unexpected rounding mode"
					+ RoundingMode.valueOf(roundingMode);

			if (cmpFracHalf < 0) {
				return false;
			} else if (cmpFracHalf > 0) {
				return true;
			} else { // half-way
				assert cmpFracHalf == 0;

				switch (roundingMode) {
				case ROUND_HALF_DOWN:
					return false;

				case ROUND_HALF_UP:
					return true;

				case ROUND_HALF_EVEN:
					return oddQuot;

				default:
					throw new AssertionError("Unexpected rounding mode" + roundingMode);
				}
			}
		}
	}

	/**
	 * Tests if quotient has to be incremented according the roundingMode
	 */
	private static boolean needIncrement(long ldivisor, int roundingMode, int qsign, long q, long r) {
		assert r != 0L;

		int cmpFracHalf;
		if (r <= HALF_LONG_MIN_VALUE || r > HALF_LONG_MAX_VALUE) {
			cmpFracHalf = 1; // 2 * r can't fit into long
		} else {
			cmpFracHalf = longCompareMagnitude(2 * r, ldivisor);
		}

		return commonNeedIncrement(roundingMode, qsign, cmpFracHalf, (q & 1L) != 0L);
	}

	/**
	 * Divides {@code BigInteger} value by {@code long} value and do rounding based on the passed in roundingMode.
	 */
	private static BigInteger divideAndRound(BigInteger bdividend, long ldivisor, int roundingMode) {
		boolean isRemainderZero; // record remainder is zero or not
		int qsign; // quotient sign
		long r = 0; // store quotient & remainder in long
		MutableBigInteger mq = null; // store quotient
		// Descend into mutables for faster remainder checks
		MutableBigInteger mdividend = new MutableBigInteger(bdividend.mag);
		mq = new MutableBigInteger();
		r = mdividend.divide(ldivisor, mq);
		isRemainderZero = (r == 0);
		qsign = (ldivisor < 0) ? -bdividend.signum : bdividend.signum;
		if (!isRemainderZero) {
			if (needIncrement(ldivisor, roundingMode, qsign, mq, r)) {
				mq.add(MutableBigInteger.ONE);
			}
		}
		return mq.toBigInteger(qsign);
	}

	/**
	 * Internally used for division operation for division {@code BigInteger} by {@code long}. The returned
	 * {@code BigDecimal} object is the quotient whose scale is set to the passed in scale. If the remainder is not
	 * zero, it will be rounded based on the passed in roundingMode. Also, if the remainder is zero and the last
	 * parameter, i.e. preferredScale is NOT equal to scale, the trailing zeros of the result is stripped to match the
	 * preferredScale.
	 */
	private static BigDecimal divideAndRound(BigInteger bdividend, long ldivisor, int scale, int roundingMode,
			int preferredScale) {
		boolean isRemainderZero; // record remainder is zero or not
		int qsign; // quotient sign
		long r = 0; // store quotient & remainder in long
		MutableBigInteger mq = null; // store quotient
		// Descend into mutables for faster remainder checks
		MutableBigInteger mdividend = new MutableBigInteger(bdividend.mag);
		mq = new MutableBigInteger();
		r = mdividend.divide(ldivisor, mq);
		isRemainderZero = (r == 0);
		qsign = (ldivisor < 0) ? -bdividend.signum : bdividend.signum;
		if (!isRemainderZero) {
			if (needIncrement(ldivisor, roundingMode, qsign, mq, r)) {
				mq.add(MutableBigInteger.ONE);
			}
			return mq.toBigDecimal(qsign, scale);
		} else {
			if (preferredScale != scale) {
				long compactVal = mq.toCompactValue(qsign);
				if (compactVal != INFLATED) {
					return createAndStripZerosToMatchScale(compactVal, scale, preferredScale);
				}
				BigInteger intVal = mq.toBigInteger(qsign);
				return createAndStripZerosToMatchScale(intVal, scale, preferredScale);
			} else {
				return mq.toBigDecimal(qsign, scale);
			}
		}
	}

	/**
	 * Tests if quotient has to be incremented according the roundingMode
	 */
	private static boolean needIncrement(long ldivisor, int roundingMode, int qsign, MutableBigInteger mq, long r) {
		assert r != 0L;

		int cmpFracHalf;
		if (r <= HALF_LONG_MIN_VALUE || r > HALF_LONG_MAX_VALUE) {
			cmpFracHalf = 1; // 2 * r can't fit into long
		} else {
			cmpFracHalf = longCompareMagnitude(2 * r, ldivisor);
		}

		return commonNeedIncrement(roundingMode, qsign, cmpFracHalf, mq.isOdd());
	}

	/**
	 * Divides {@code BigInteger} value by {@code BigInteger} value and do rounding based on the passed in roundingMode.
	 */
	private static BigInteger divideAndRound(BigInteger bdividend, BigInteger bdivisor, int roundingMode) {
		boolean isRemainderZero; // record remainder is zero or not
		int qsign; // quotient sign
		// Descend into mutables for faster remainder checks
		MutableBigInteger mdividend = new MutableBigInteger(bdividend.mag);
		MutableBigInteger mq = new MutableBigInteger();
		MutableBigInteger mdivisor = new MutableBigInteger(bdivisor.mag);
		MutableBigInteger mr = mdividend.divide(mdivisor, mq);
		isRemainderZero = mr.isZero();
		qsign = (bdividend.signum != bdivisor.signum) ? -1 : 1;
		if (!isRemainderZero) {
			if (needIncrement(mdivisor, roundingMode, qsign, mq, mr)) {
				mq.add(MutableBigInteger.ONE);
			}
		}
		return mq.toBigInteger(qsign);
	}

	/**
	 * Internally used for division operation for division {@code BigInteger} by {@code BigInteger}. The returned
	 * {@code BigDecimal} object is the quotient whose scale is set to the passed in scale. If the remainder is not
	 * zero, it will be rounded based on the passed in roundingMode. Also, if the remainder is zero and the last
	 * parameter, i.e. preferredScale is NOT equal to scale, the trailing zeros of the result is stripped to match the
	 * preferredScale.
	 */
	private static BigDecimal divideAndRound(BigInteger bdividend, BigInteger bdivisor, int scale, int roundingMode,
			int preferredScale) {
		boolean isRemainderZero; // record remainder is zero or not
		int qsign; // quotient sign
		// Descend into mutables for faster remainder checks
		MutableBigInteger mdividend = new MutableBigInteger(bdividend.mag);
		MutableBigInteger mq = new MutableBigInteger();
		MutableBigInteger mdivisor = new MutableBigInteger(bdivisor.mag);
		MutableBigInteger mr = mdividend.divide(mdivisor, mq);
		isRemainderZero = mr.isZero();
		qsign = (bdividend.signum != bdivisor.signum) ? -1 : 1;
		if (!isRemainderZero) {
			if (needIncrement(mdivisor, roundingMode, qsign, mq, mr)) {
				mq.add(MutableBigInteger.ONE);
			}
			return mq.toBigDecimal(qsign, scale);
		} else {
			if (preferredScale != scale) {
				long compactVal = mq.toCompactValue(qsign);
				if (compactVal != INFLATED) {
					return createAndStripZerosToMatchScale(compactVal, scale, preferredScale);
				}
				BigInteger intVal = mq.toBigInteger(qsign);
				return createAndStripZerosToMatchScale(intVal, scale, preferredScale);
			} else {
				return mq.toBigDecimal(qsign, scale);
			}
		}
	}

	/**
	 * Tests if quotient has to be incremented according the roundingMode
	 */
	private static boolean needIncrement(MutableBigInteger mdivisor, int roundingMode, int qsign, MutableBigInteger mq,
			MutableBigInteger mr) {
		assert !mr.isZero();
		int cmpFracHalf = mr.compareHalf(mdivisor);
		return commonNeedIncrement(roundingMode, qsign, cmpFracHalf, mq.isOdd());
	}

	/**
	 * Remove insignificant trailing zeros from this {@code BigInteger} value until the preferred scale is reached or no
	 * more zeros can be removed. If the preferred scale is less than Integer.MIN_VALUE, all the trailing zeros will be
	 * removed.
	 *
	 * @return new {@code BigDecimal} with a scale possibly reduced to be closed to the preferred scale.
	 */
	private static BigDecimal createAndStripZerosToMatchScale(BigInteger intVal, int scale, long preferredScale) {
		BigInteger qr[]; // quotient-remainder pair
		while (intVal.compareMagnitude(BigInteger.TEN) >= 0 && scale > preferredScale) {
			if (intVal.testBit(0)) {
				break; // odd number cannot end in 0
			}
			qr = intVal.divideAndRemainder(BigInteger.TEN);
			if (qr[1].signum() != 0) {
				break; // non-0 remainder
			}
			intVal = qr[0];
			scale = checkScale(intVal, (long) scale - 1); // could Overflow
		}
		return valueOf(intVal, scale, 0);
	}

	/**
	 * Remove insignificant trailing zeros from this {@code long} value until the preferred scale is reached or no more
	 * zeros can be removed. If the preferred scale is less than Integer.MIN_VALUE, all the trailing zeros will be
	 * removed.
	 *
	 * @return new {@code BigDecimal} with a scale possibly reduced to be closed to the preferred scale.
	 */
	private static BigDecimal createAndStripZerosToMatchScale(long compactVal, int scale, long preferredScale) {
		while (Math.abs(compactVal) >= 10L && scale > preferredScale) {
			if ((compactVal & 1L) != 0L) {
				break; // odd number cannot end in 0
			}
			long r = compactVal % 10L;
			if (r != 0L) {
				break; // non-0 remainder
			}
			compactVal /= 10;
			scale = checkScale(compactVal, (long) scale - 1); // could Overflow
		}
		return valueOf(compactVal, scale);
	}

	private static BigDecimal stripZerosToMatchScale(BigInteger intVal, long intCompact, int scale,
			int preferredScale) {
		if (intCompact != INFLATED) {
			return createAndStripZerosToMatchScale(intCompact, scale, preferredScale);
		} else {
			return createAndStripZerosToMatchScale(intVal == null ? INFLATED_BIGINT : intVal, scale, preferredScale);
		}
	}

	/*
	 * returns INFLATED if oveflow
	 */
	private static long add(long xs, long ys) {
		long sum = xs + ys;
		// See "Hacker's Delight" section 2-12 for explanation of
		// the overflow test.
		if ((((sum ^ xs) & (sum ^ ys))) >= 0L) { // not overflowed
			return sum;
		}
		return INFLATED;
	}

	private static BigDecimal add(long xs, long ys, int scale) {
		long sum = add(xs, ys);
		if (sum != INFLATED) {
			return BigDecimal.valueOf(sum, scale);
		}
		return new BigDecimal(BigInteger.valueOf(xs).add(ys), scale);
	}

	private static BigDecimal add(final long xs, int scale1, final long ys, int scale2) {
		long sdiff = (long) scale1 - scale2;
		if (sdiff == 0) {
			return add(xs, ys, scale1);
		} else if (sdiff < 0) {
			int raise = checkScale(xs, -sdiff);
			long scaledX = longMultiplyPowerTen(xs, raise);
			if (scaledX != INFLATED) {
				return add(scaledX, ys, scale2);
			} else {
				BigInteger bigsum = bigMultiplyPowerTen(xs, raise).add(ys);
				return ((xs ^ ys) >= 0) ? // same sign test
						new BigDecimal(bigsum, INFLATED, scale2, 0) : valueOf(bigsum, scale2, 0);
			}
		} else {
			int raise = checkScale(ys, sdiff);
			long scaledY = longMultiplyPowerTen(ys, raise);
			if (scaledY != INFLATED) {
				return add(xs, scaledY, scale1);
			} else {
				BigInteger bigsum = bigMultiplyPowerTen(ys, raise).add(xs);
				return ((xs ^ ys) >= 0) ? new BigDecimal(bigsum, INFLATED, scale1, 0) : valueOf(bigsum, scale1, 0);
			}
		}
	}

	private static BigDecimal add(final long xs, int scale1, BigInteger snd, int scale2) {
		int rscale = scale1;
		long sdiff = (long) rscale - scale2;
		boolean sameSigns = (Long.signum(xs) == snd.signum);
		BigInteger sum;
		if (sdiff < 0) {
			int raise = checkScale(xs, -sdiff);
			rscale = scale2;
			long scaledX = longMultiplyPowerTen(xs, raise);
			if (scaledX == INFLATED) {
				sum = snd.add(bigMultiplyPowerTen(xs, raise));
			} else {
				sum = snd.add(scaledX);
			}
		} else { // if (sdiff > 0) {
			int raise = checkScale(snd, sdiff);
			snd = bigMultiplyPowerTen(snd, raise);
			sum = snd.add(xs);
		}
		return (sameSigns) ? new BigDecimal(sum, INFLATED, rscale, 0) : valueOf(sum, rscale, 0);
	}

	private static BigDecimal add(BigInteger fst, int scale1, BigInteger snd, int scale2) {
		int rscale = scale1;
		long sdiff = (long) rscale - scale2;
		if (sdiff != 0) {
			if (sdiff < 0) {
				int raise = checkScale(fst, -sdiff);
				rscale = scale2;
				fst = bigMultiplyPowerTen(fst, raise);
			} else {
				int raise = checkScale(snd, sdiff);
				snd = bigMultiplyPowerTen(snd, raise);
			}
		}
		BigInteger sum = fst.add(snd);
		return (fst.signum == snd.signum) ? new BigDecimal(sum, INFLATED, rscale, 0) : valueOf(sum, rscale, 0);
	}

	private static BigInteger bigMultiplyPowerTen(long value, int n) {
		if (n <= 0) {
			return BigInteger.valueOf(value);
		}
		return bigTenToThe(n).multiply(value);
	}

	private static BigInteger bigMultiplyPowerTen(BigInteger value, int n) {
		if (n <= 0) {
			return value;
		}
		if (n < LONG_TEN_POWERS_TABLE.length) {
			return value.multiply(LONG_TEN_POWERS_TABLE[n]);
		}
		return value.multiply(bigTenToThe(n));
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (xs /
	 * ys)}, with rounding according to the context settings.
	 *
	 * Fast path - used only when (xscale <= yscale && yscale < 18 && mc.presision<18) {
	 */
	private static BigDecimal divideSmallFastPath(final long xs, int xscale, final long ys, int yscale,
			long preferredScale, MathContext mc) {
		int mcp = mc.precision;
		int roundingMode = mc.roundingMode.oldMode;

		assert (xscale <= yscale) && (yscale < 18) && (mcp < 18);
		int xraise = yscale - xscale; // xraise >=0
		long scaledX = (xraise == 0) ? xs : longMultiplyPowerTen(xs, xraise); // can't overflow here!
		BigDecimal quotient;

		int cmp = longCompareMagnitude(scaledX, ys);
		if (cmp > 0) { // satisfy constraint (b)
			yscale -= 1; // [that is, divisor *= 10]
			int scl = checkScaleNonZero(preferredScale + yscale - xscale + mcp);
			if (checkScaleNonZero((long) mcp + yscale - xscale) > 0) {
				// assert newScale >= xscale
				int raise = checkScaleNonZero((long) mcp + yscale - xscale);
				long scaledXs;
				if ((scaledXs = longMultiplyPowerTen(xs, raise)) == INFLATED) {
					quotient = null;
					if ((mcp - 1) >= 0 && (mcp - 1) < LONG_TEN_POWERS_TABLE.length) {
						quotient = multiplyDivideAndRound(LONG_TEN_POWERS_TABLE[mcp - 1], scaledX, ys, scl,
								roundingMode, checkScaleNonZero(preferredScale));
					}
					if (quotient == null) {
						BigInteger rb = bigMultiplyPowerTen(scaledX, mcp - 1);
						quotient = divideAndRound(rb, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
					}
				} else {
					quotient = divideAndRound(scaledXs, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
				}
			} else {
				int newScale = checkScaleNonZero((long) xscale - mcp);
				// assert newScale >= yscale
				if (newScale == yscale) { // easy case
					quotient = divideAndRound(xs, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
				} else {
					int raise = checkScaleNonZero((long) newScale - yscale);
					long scaledYs;
					if ((scaledYs = longMultiplyPowerTen(ys, raise)) == INFLATED) {
						BigInteger rb = bigMultiplyPowerTen(ys, raise);
						quotient = divideAndRound(BigInteger.valueOf(xs), rb, scl, roundingMode,
								checkScaleNonZero(preferredScale));
					} else {
						quotient = divideAndRound(xs, scaledYs, scl, roundingMode, checkScaleNonZero(preferredScale));
					}
				}
			}
		} else {
			// abs(scaledX) <= abs(ys)
			// result is "scaledX * 10^msp / ys"
			int scl = checkScaleNonZero(preferredScale + yscale - xscale + mcp);
			if (cmp == 0) {
				// abs(scaleX)== abs(ys) => result will be scaled 10^mcp + correct sign
				quotient = roundedTenPower(((scaledX < 0) == (ys < 0)) ? 1 : -1, mcp, scl,
						checkScaleNonZero(preferredScale));
			} else {
				// abs(scaledX) < abs(ys)
				long scaledXs;
				if ((scaledXs = longMultiplyPowerTen(scaledX, mcp)) == INFLATED) {
					quotient = null;
					if (mcp < LONG_TEN_POWERS_TABLE.length) {
						quotient = multiplyDivideAndRound(LONG_TEN_POWERS_TABLE[mcp], scaledX, ys, scl, roundingMode,
								checkScaleNonZero(preferredScale));
					}
					if (quotient == null) {
						BigInteger rb = bigMultiplyPowerTen(scaledX, mcp);
						quotient = divideAndRound(rb, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
					}
				} else {
					quotient = divideAndRound(scaledXs, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
				}
			}
		}
		// doRound, here, only affects 1000000000 case.
		return doRound(quotient, mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (xs /
	 * ys)}, with rounding according to the context settings.
	 */
	private static BigDecimal divide(final long xs, int xscale, final long ys, int yscale, long preferredScale,
			MathContext mc) {
		int mcp = mc.precision;
		if (xscale <= yscale && yscale < 18 && mcp < 18) {
			return divideSmallFastPath(xs, xscale, ys, yscale, preferredScale, mc);
		}
		if (compareMagnitudeNormalized(xs, xscale, ys, yscale) > 0) {// satisfy constraint (b)
			yscale -= 1; // [that is, divisor *= 10]
		}
		int roundingMode = mc.roundingMode.oldMode;
		// In order to find out whether the divide generates the exact result,
		// we avoid calling the above divide method. 'quotient' holds the
		// return BigDecimal object whose scale will be set to 'scl'.
		int scl = checkScaleNonZero(preferredScale + yscale - xscale + mcp);
		BigDecimal quotient;
		if (checkScaleNonZero((long) mcp + yscale - xscale) > 0) {
			int raise = checkScaleNonZero((long) mcp + yscale - xscale);
			long scaledXs;
			if ((scaledXs = longMultiplyPowerTen(xs, raise)) == INFLATED) {
				BigInteger rb = bigMultiplyPowerTen(xs, raise);
				quotient = divideAndRound(rb, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
			} else {
				quotient = divideAndRound(scaledXs, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
			}
		} else {
			int newScale = checkScaleNonZero((long) xscale - mcp);
			// assert newScale >= yscale
			if (newScale == yscale) { // easy case
				quotient = divideAndRound(xs, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
			} else {
				int raise = checkScaleNonZero((long) newScale - yscale);
				long scaledYs;
				if ((scaledYs = longMultiplyPowerTen(ys, raise)) == INFLATED) {
					BigInteger rb = bigMultiplyPowerTen(ys, raise);
					quotient = divideAndRound(BigInteger.valueOf(xs), rb, scl, roundingMode,
							checkScaleNonZero(preferredScale));
				} else {
					quotient = divideAndRound(xs, scaledYs, scl, roundingMode, checkScaleNonZero(preferredScale));
				}
			}
		}
		// doRound, here, only affects 1000000000 case.
		return doRound(quotient, mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (xs /
	 * ys)}, with rounding according to the context settings.
	 */
	private static BigDecimal divide(BigInteger xs, int xscale, long ys, int yscale, long preferredScale,
			MathContext mc) {
		// Normalize dividend & divisor so that both fall into [0.1, 0.999...]
		if ((-compareMagnitudeNormalized(ys, yscale, xs, xscale)) > 0) {// satisfy constraint (b)
			yscale -= 1; // [that is, divisor *= 10]
		}
		int mcp = mc.precision;
		int roundingMode = mc.roundingMode.oldMode;

		// In order to find out whether the divide generates the exact result,
		// we avoid calling the above divide method. 'quotient' holds the
		// return BigDecimal object whose scale will be set to 'scl'.
		BigDecimal quotient;
		int scl = checkScaleNonZero(preferredScale + yscale - xscale + mcp);
		if (checkScaleNonZero((long) mcp + yscale - xscale) > 0) {
			int raise = checkScaleNonZero((long) mcp + yscale - xscale);
			BigInteger rb = bigMultiplyPowerTen(xs, raise);
			quotient = divideAndRound(rb, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
		} else {
			int newScale = checkScaleNonZero((long) xscale - mcp);
			// assert newScale >= yscale
			if (newScale == yscale) { // easy case
				quotient = divideAndRound(xs, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
			} else {
				int raise = checkScaleNonZero((long) newScale - yscale);
				long scaledYs;
				if ((scaledYs = longMultiplyPowerTen(ys, raise)) == INFLATED) {
					BigInteger rb = bigMultiplyPowerTen(ys, raise);
					quotient = divideAndRound(xs, rb, scl, roundingMode, checkScaleNonZero(preferredScale));
				} else {
					quotient = divideAndRound(xs, scaledYs, scl, roundingMode, checkScaleNonZero(preferredScale));
				}
			}
		}
		// doRound, here, only affects 1000000000 case.
		return doRound(quotient, mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (xs /
	 * ys)}, with rounding according to the context settings.
	 */
	private static BigDecimal divide(long xs, int xscale, BigInteger ys, int yscale, long preferredScale,
			MathContext mc) {
		// Normalize dividend & divisor so that both fall into [0.1, 0.999...]
		if (compareMagnitudeNormalized(xs, xscale, ys, yscale) > 0) {// satisfy constraint (b)
			yscale -= 1; // [that is, divisor *= 10]
		}
		int mcp = mc.precision;
		int roundingMode = mc.roundingMode.oldMode;

		// In order to find out whether the divide generates the exact result,
		// we avoid calling the above divide method. 'quotient' holds the
		// return BigDecimal object whose scale will be set to 'scl'.
		BigDecimal quotient;
		int scl = checkScaleNonZero(preferredScale + yscale - xscale + mcp);
		if (checkScaleNonZero((long) mcp + yscale - xscale) > 0) {
			int raise = checkScaleNonZero((long) mcp + yscale - xscale);
			BigInteger rb = bigMultiplyPowerTen(xs, raise);
			quotient = divideAndRound(rb, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
		} else {
			int newScale = checkScaleNonZero((long) xscale - mcp);
			int raise = checkScaleNonZero((long) newScale - yscale);
			BigInteger rb = bigMultiplyPowerTen(ys, raise);
			quotient = divideAndRound(BigInteger.valueOf(xs), rb, scl, roundingMode, checkScaleNonZero(preferredScale));
		}
		// doRound, here, only affects 1000000000 case.
		return doRound(quotient, mc);
	}

	/**
	 * Returns a {@code BigDecimal} whose value is {@code (xs /
	 * ys)}, with rounding according to the context settings.
	 */
	private static BigDecimal divide(BigInteger xs, int xscale, BigInteger ys, int yscale, long preferredScale,
			MathContext mc) {
		// Normalize dividend & divisor so that both fall into [0.1, 0.999...]
		if (compareMagnitudeNormalized(xs, xscale, ys, yscale) > 0) {// satisfy constraint (b)
			yscale -= 1; // [that is, divisor *= 10]
		}
		int mcp = mc.precision;
		int roundingMode = mc.roundingMode.oldMode;

		// In order to find out whether the divide generates the exact result,
		// we avoid calling the above divide method. 'quotient' holds the
		// return BigDecimal object whose scale will be set to 'scl'.
		BigDecimal quotient;
		int scl = checkScaleNonZero(preferredScale + yscale - xscale + mcp);
		if (checkScaleNonZero((long) mcp + yscale - xscale) > 0) {
			int raise = checkScaleNonZero((long) mcp + yscale - xscale);
			BigInteger rb = bigMultiplyPowerTen(xs, raise);
			quotient = divideAndRound(rb, ys, scl, roundingMode, checkScaleNonZero(preferredScale));
		} else {
			int newScale = checkScaleNonZero((long) xscale - mcp);
			int raise = checkScaleNonZero((long) newScale - yscale);
			BigInteger rb = bigMultiplyPowerTen(ys, raise);
			quotient = divideAndRound(xs, rb, scl, roundingMode, checkScaleNonZero(preferredScale));
		}
		// doRound, here, only affects 1000000000 case.
		return doRound(quotient, mc);
	}

	/*
	 * performs divideAndRound for (dividend0*dividend1, divisor) returns null if quotient can't fit into long value;
	 */
	private static BigDecimal multiplyDivideAndRound(long dividend0, long dividend1, long divisor, int scale,
			int roundingMode, int preferredScale) {
		int qsign = Long.signum(dividend0) * Long.signum(dividend1) * Long.signum(divisor);
		dividend0 = Math.abs(dividend0);
		dividend1 = Math.abs(dividend1);
		divisor = Math.abs(divisor);
		// multiply dividend0 * dividend1
		long d0_hi = dividend0 >>> 32;
		long d0_lo = dividend0 & LONG_MASK;
		long d1_hi = dividend1 >>> 32;
		long d1_lo = dividend1 & LONG_MASK;
		long product = d0_lo * d1_lo;
		long d0 = product & LONG_MASK;
		long d1 = product >>> 32;
		product = d0_hi * d1_lo + d1;
		d1 = product & LONG_MASK;
		long d2 = product >>> 32;
		product = d0_lo * d1_hi + d1;
		d1 = product & LONG_MASK;
		d2 += product >>> 32;
		long d3 = d2 >>> 32;
		d2 &= LONG_MASK;
		product = d0_hi * d1_hi + d2;
		d2 = product & LONG_MASK;
		d3 = ((product >>> 32) + d3) & LONG_MASK;
		final long dividendHi = make64(d3, d2);
		final long dividendLo = make64(d1, d0);
		// divide
		return divideAndRound128(dividendHi, dividendLo, divisor, qsign, scale, roundingMode, preferredScale);
	}

	private static final long DIV_NUM_BASE = (1L << 32); // Number base (32 bits).

	/*
	 * divideAndRound 128-bit value by long divisor. returns null if quotient can't fit into long value; Specialized
	 * version of Knuth's division
	 */
	private static BigDecimal divideAndRound128(final long dividendHi, final long dividendLo, long divisor, int sign,
			int scale, int roundingMode, int preferredScale) {
		if (dividendHi >= divisor) {
			return null;
		}
		final int shift = LongHelper.numberOfLeadingZeros(divisor);
		divisor <<= shift;

		final long v1 = divisor >>> 32;
		final long v0 = divisor & LONG_MASK;

		long q1, q0;
		long r_tmp;

		long tmp = dividendLo << shift;
		long u1 = tmp >>> 32;
		long u0 = tmp & LONG_MASK;

		tmp = (dividendHi << shift) | (dividendLo >>> 64 - shift);
		long u2 = tmp & LONG_MASK;
		tmp = divWord(tmp, v1);
		q1 = tmp & LONG_MASK;
		r_tmp = tmp >>> 32;
		while (q1 >= DIV_NUM_BASE || unsignedLongCompare(q1 * v0, make64(r_tmp, u1))) {
			q1--;
			r_tmp += v1;
			if (r_tmp >= DIV_NUM_BASE) {
				break;
			}
		}
		tmp = mulsub(u2, u1, v1, v0, q1);
		u1 = tmp & LONG_MASK;
		tmp = divWord(tmp, v1);
		q0 = tmp & LONG_MASK;
		r_tmp = tmp >>> 32;
		while (q0 >= DIV_NUM_BASE || unsignedLongCompare(q0 * v0, make64(r_tmp, u0))) {
			q0--;
			r_tmp += v1;
			if (r_tmp >= DIV_NUM_BASE) {
				break;
			}
		}
		if ((int) q1 < 0) {
			// result (which is positive and unsigned here)
			// can't fit into long due to sign bit is used for value
			MutableBigInteger mq = new MutableBigInteger(new int[] { (int) q1, (int) q0 });
			if (roundingMode == ROUND_DOWN && scale == preferredScale) {
				return mq.toBigDecimal(sign, scale);
			}
			long r = mulsub(u1, u0, v1, v0, q0) >>> shift;
			if (r != 0) {
				if (needIncrement(divisor >>> shift, roundingMode, sign, mq, r)) {
					mq.add(MutableBigInteger.ONE);
				}
				return mq.toBigDecimal(sign, scale);
			} else {
				if (preferredScale != scale) {
					BigInteger intVal = mq.toBigInteger(sign);
					return createAndStripZerosToMatchScale(intVal, scale, preferredScale);
				} else {
					return mq.toBigDecimal(sign, scale);
				}
			}
		}
		long q = make64(q1, q0);
		q *= sign;
		if (roundingMode == ROUND_DOWN && scale == preferredScale) {
			return valueOf(q, scale);
		}
		long r = mulsub(u1, u0, v1, v0, q0) >>> shift;
		if (r != 0) {
			boolean increment = needIncrement(divisor >>> shift, roundingMode, sign, q, r);
			return valueOf((increment ? q + sign : q), scale);
		} else {
			if (preferredScale != scale) {
				return createAndStripZerosToMatchScale(q, scale, preferredScale);
			} else {
				return valueOf(q, scale);
			}
		}
	}

	/*
	 * calculate divideAndRound for ldividend*10^raise / divisor when abs(dividend)==abs(divisor);
	 */
	private static BigDecimal roundedTenPower(int qsign, int raise, int scale, int preferredScale) {
		if (scale > preferredScale) {
			int diff = scale - preferredScale;
			if (diff < raise) {
				return scaledTenPow(raise - diff, qsign, preferredScale);
			} else {
				return valueOf(qsign, scale - raise);
			}
		} else {
			return scaledTenPow(raise, qsign, scale);
		}
	}

	static BigDecimal scaledTenPow(int n, int sign, int scale) {
		if (n < LONG_TEN_POWERS_TABLE.length) {
			return valueOf(sign * LONG_TEN_POWERS_TABLE[n], scale);
		} else {
			BigInteger unscaledVal = bigTenToThe(n);
			if (sign == -1) {
				unscaledVal = unscaledVal.negate();
			}
			return new BigDecimal(unscaledVal, INFLATED, scale, n + 1);
		}
	}

	private static long divWord(long n, long dLong) {
		long r;
		long q;
		if (dLong == 1) {
			q = (int) n;
			return (q & LONG_MASK);
		}
		// Approximate the quotient and remainder
		q = (n >>> 1) / (dLong >>> 1);
		r = n - q * dLong;

		// Correct the approximation
		while (r < 0) {
			r += dLong;
			q--;
		}
		while (r >= dLong) {
			r -= dLong;
			q++;
		}
		// n - q*dlong == r && 0 <= r <dLong, hence we're done.
		return (r << 32) | (q & LONG_MASK);
	}

	private static long make64(long hi, long lo) {
		return hi << 32 | lo;
	}

	private static long mulsub(long u1, long u0, final long v1, final long v0, long q0) {
		long tmp = u0 - q0 * v0;
		return make64(u1 + (tmp >>> 32) - q0 * v1, tmp & LONG_MASK);
	}

	private static boolean unsignedLongCompare(long one, long two) {
		return (one + Long.MIN_VALUE) > (two + Long.MIN_VALUE);
	}

	private static boolean unsignedLongCompareEq(long one, long two) {
		return (one + Long.MIN_VALUE) >= (two + Long.MIN_VALUE);
	}

	// Compare Normalize dividend & divisor so that both fall into [0.1, 0.999...]
	private static int compareMagnitudeNormalized(long xs, int xscale, long ys, int yscale) {
		// assert xs!=0 && ys!=0
		int sdiff = xscale - yscale;
		if (sdiff != 0) {
			if (sdiff < 0) {
				xs = longMultiplyPowerTen(xs, -sdiff);
			} else { // sdiff > 0
				ys = longMultiplyPowerTen(ys, sdiff);
			}
		}
		if (xs != INFLATED) {
			return (ys != INFLATED) ? longCompareMagnitude(xs, ys) : -1;
		} else {
			return 1;
		}
	}

	// Compare Normalize dividend & divisor so that both fall into [0.1, 0.999...]
	private static int compareMagnitudeNormalized(long xs, int xscale, BigInteger ys, int yscale) {
		// assert "ys can't be represented as long"
		if (xs == 0) {
			return -1;
		}
		int sdiff = xscale - yscale;
		if (sdiff < 0) {
			if (longMultiplyPowerTen(xs, -sdiff) == INFLATED) {
				return bigMultiplyPowerTen(xs, -sdiff).compareMagnitude(ys);
			}
		}
		return -1;
	}

	// Compare Normalize dividend & divisor so that both fall into [0.1, 0.999...]
	private static int compareMagnitudeNormalized(BigInteger xs, int xscale, BigInteger ys, int yscale) {
		int sdiff = xscale - yscale;
		if (sdiff < 0) {
			return bigMultiplyPowerTen(xs, -sdiff).compareMagnitude(ys);
		} else { // sdiff >= 0
			return xs.compareMagnitude(bigMultiplyPowerTen(ys, sdiff));
		}
	}

	private static long multiply(long x, long y) {
		long product = x * y;
		long ax = Math.abs(x);
		long ay = Math.abs(y);
		if (((ax | ay) >>> 31 == 0) || (y == 0) || (product / y == x)) {
			return product;
		}
		return INFLATED;
	}

	private static BigDecimal multiply(long x, long y, int scale) {
		long product = multiply(x, y);
		if (product != INFLATED) {
			return valueOf(product, scale);
		}
		return new BigDecimal(BigInteger.valueOf(x).multiply(y), INFLATED, scale, 0);
	}

	private static BigDecimal multiply(long x, BigInteger y, int scale) {
		if (x == 0) {
			return zeroValueOf(scale);
		}
		return new BigDecimal(y.multiply(x), INFLATED, scale, 0);
	}

	private static BigDecimal multiply(BigInteger x, BigInteger y, int scale) {
		return new BigDecimal(x.multiply(y), INFLATED, scale, 0);
	}

	/**
	 * Multiplies two long values and rounds according {@code MathContext}
	 */
	private static BigDecimal multiplyAndRound(long x, long y, int scale, MathContext mc) {
		long product = multiply(x, y);
		if (product != INFLATED) {
			return doRound(product, scale, mc);
		}
		// attempt to do it in 128 bits
		int rsign = 1;
		if (x < 0) {
			x = -x;
			rsign = -1;
		}
		if (y < 0) {
			y = -y;
			rsign *= -1;
		}
		// multiply dividend0 * dividend1
		long m0_hi = x >>> 32;
		long m0_lo = x & LONG_MASK;
		long m1_hi = y >>> 32;
		long m1_lo = y & LONG_MASK;
		product = m0_lo * m1_lo;
		long m0 = product & LONG_MASK;
		long m1 = product >>> 32;
		product = m0_hi * m1_lo + m1;
		m1 = product & LONG_MASK;
		long m2 = product >>> 32;
		product = m0_lo * m1_hi + m1;
		m1 = product & LONG_MASK;
		m2 += product >>> 32;
		long m3 = m2 >>> 32;
		m2 &= LONG_MASK;
		product = m0_hi * m1_hi + m2;
		m2 = product & LONG_MASK;
		m3 = ((product >>> 32) + m3) & LONG_MASK;
		final long mHi = make64(m3, m2);
		final long mLo = make64(m1, m0);
		BigDecimal res = doRound128(mHi, mLo, rsign, scale, mc);
		if (res != null) {
			return res;
		}
		res = new BigDecimal(BigInteger.valueOf(x).multiply(y * rsign), INFLATED, scale, 0);
		return doRound(res, mc);
	}

	private static BigDecimal multiplyAndRound(long x, BigInteger y, int scale, MathContext mc) {
		if (x == 0) {
			return zeroValueOf(scale);
		}
		return doRound(y.multiply(x), scale, mc);
	}

	private static BigDecimal multiplyAndRound(BigInteger x, BigInteger y, int scale, MathContext mc) {
		return doRound(x.multiply(y), scale, mc);
	}

	/**
	 * rounds 128-bit value according {@code MathContext} returns null if result can't be repsented as compact
	 * BigDecimal.
	 */
	private static BigDecimal doRound128(long hi, long lo, int sign, int scale, MathContext mc) {
		int mcp = mc.precision;
		int drop;
		BigDecimal res = null;
		if (((drop = precision(hi, lo) - mcp) > 0) && (drop < LONG_TEN_POWERS_TABLE.length)) {
			scale = checkScaleNonZero((long) scale - drop);
			res = divideAndRound128(hi, lo, LONG_TEN_POWERS_TABLE[drop], sign, scale, mc.roundingMode.oldMode, scale);
		}
		if (res != null) {
			return doRound(res, mc);
		}
		return null;
	}

	private static final long[][] LONGLONG_TEN_POWERS_TABLE = { { 0L, 0x8AC7230489E80000L }, // 10^19
			{ 0x5L, 0x6bc75e2d63100000L }, // 10^20
			{ 0x36L, 0x35c9adc5dea00000L }, // 10^21
			{ 0x21eL, 0x19e0c9bab2400000L }, // 10^22
			{ 0x152dL, 0x02c7e14af6800000L }, // 10^23
			{ 0xd3c2L, 0x1bcecceda1000000L }, // 10^24
			{ 0x84595L, 0x161401484a000000L }, // 10^25
			{ 0x52b7d2L, 0xdcc80cd2e4000000L }, // 10^26
			{ 0x33b2e3cL, 0x9fd0803ce8000000L }, // 10^27
			{ 0x204fce5eL, 0x3e25026110000000L }, // 10^28
			{ 0x1431e0faeL, 0x6d7217caa0000000L }, // 10^29
			{ 0xc9f2c9cd0L, 0x4674edea40000000L }, // 10^30
			{ 0x7e37be2022L, 0xc0914b2680000000L }, // 10^31
			{ 0x4ee2d6d415bL, 0x85acef8100000000L }, // 10^32
			{ 0x314dc6448d93L, 0x38c15b0a00000000L }, // 10^33
			{ 0x1ed09bead87c0L, 0x378d8e6400000000L }, // 10^34
			{ 0x13426172c74d82L, 0x2b878fe800000000L }, // 10^35
			{ 0xc097ce7bc90715L, 0xb34b9f1000000000L }, // 10^36
			{ 0x785ee10d5da46d9L, 0x00f436a000000000L }, // 10^37
			{ 0x4b3b4ca85a86c47aL, 0x098a224000000000L }, // 10^38
	};

	/*
	 * returns precision of 128-bit value
	 */
	private static int precision(long hi, long lo) {
		if (hi == 0) {
			if (lo >= 0) {
				return longDigitLength(lo);
			}
			return (unsignedLongCompareEq(lo, LONGLONG_TEN_POWERS_TABLE[0][1])) ? 20 : 19;
			// 0x8AC7230489E80000L = unsigned 2^19
		}
		int r = ((128 - LongHelper.numberOfLeadingZeros(hi) + 1) * 1233) >>> 12;
		int idx = r - 19;
		return (idx >= LONGLONG_TEN_POWERS_TABLE.length || longLongCompareMagnitude(hi, lo,
				LONGLONG_TEN_POWERS_TABLE[idx][0], LONGLONG_TEN_POWERS_TABLE[idx][1])) ? r : r + 1;
	}

	/*
	 * returns true if 128 bit number <hi0,lo0> is less then <hi1,lo1> hi0 & hi1 should be non-negative
	 */
	private static boolean longLongCompareMagnitude(long hi0, long lo0, long hi1, long lo1) {
		if (hi0 != hi1) {
			return hi0 < hi1;
		}
		return (lo0 + Long.MIN_VALUE) < (lo1 + Long.MIN_VALUE);
	}

	private static BigDecimal divide(long dividend, int dividendScale, long divisor, int divisorScale, int scale,
			int roundingMode) {
		if (checkScale(dividend, (long) scale + divisorScale) > dividendScale) {
			int newScale = scale + divisorScale;
			int raise = newScale - dividendScale;
			if (raise < LONG_TEN_POWERS_TABLE.length) {
				long xs = dividend;
				if ((xs = longMultiplyPowerTen(xs, raise)) != INFLATED) {
					return divideAndRound(xs, divisor, scale, roundingMode, scale);
				}
				BigDecimal q = multiplyDivideAndRound(LONG_TEN_POWERS_TABLE[raise], dividend, divisor, scale,
						roundingMode, scale);
				if (q != null) {
					return q;
				}
			}
			BigInteger scaledDividend = bigMultiplyPowerTen(dividend, raise);
			return divideAndRound(scaledDividend, divisor, scale, roundingMode, scale);
		} else {
			int newScale = checkScale(divisor, (long) dividendScale - scale);
			int raise = newScale - divisorScale;
			if (raise < LONG_TEN_POWERS_TABLE.length) {
				long ys = divisor;
				if ((ys = longMultiplyPowerTen(ys, raise)) != INFLATED) {
					return divideAndRound(dividend, ys, scale, roundingMode, scale);
				}
			}
			BigInteger scaledDivisor = bigMultiplyPowerTen(divisor, raise);
			return divideAndRound(BigInteger.valueOf(dividend), scaledDivisor, scale, roundingMode, scale);
		}
	}

	private static BigDecimal divide(BigInteger dividend, int dividendScale, long divisor, int divisorScale, int scale,
			int roundingMode) {
		if (checkScale(dividend, (long) scale + divisorScale) > dividendScale) {
			int newScale = scale + divisorScale;
			int raise = newScale - dividendScale;
			BigInteger scaledDividend = bigMultiplyPowerTen(dividend, raise);
			return divideAndRound(scaledDividend, divisor, scale, roundingMode, scale);
		} else {
			int newScale = checkScale(divisor, (long) dividendScale - scale);
			int raise = newScale - divisorScale;
			if (raise < LONG_TEN_POWERS_TABLE.length) {
				long ys = divisor;
				if ((ys = longMultiplyPowerTen(ys, raise)) != INFLATED) {
					return divideAndRound(dividend, ys, scale, roundingMode, scale);
				}
			}
			BigInteger scaledDivisor = bigMultiplyPowerTen(divisor, raise);
			return divideAndRound(dividend, scaledDivisor, scale, roundingMode, scale);
		}
	}

	private static BigDecimal divide(long dividend, int dividendScale, BigInteger divisor, int divisorScale, int scale,
			int roundingMode) {
		if (checkScale(dividend, (long) scale + divisorScale) > dividendScale) {
			int newScale = scale + divisorScale;
			int raise = newScale - dividendScale;
			BigInteger scaledDividend = bigMultiplyPowerTen(dividend, raise);
			return divideAndRound(scaledDividend, divisor, scale, roundingMode, scale);
		} else {
			int newScale = checkScale(divisor, (long) dividendScale - scale);
			int raise = newScale - divisorScale;
			BigInteger scaledDivisor = bigMultiplyPowerTen(divisor, raise);
			return divideAndRound(BigInteger.valueOf(dividend), scaledDivisor, scale, roundingMode, scale);
		}
	}

	private static BigDecimal divide(BigInteger dividend, int dividendScale, BigInteger divisor, int divisorScale,
			int scale, int roundingMode) {
		if (checkScale(dividend, (long) scale + divisorScale) > dividendScale) {
			int newScale = scale + divisorScale;
			int raise = newScale - dividendScale;
			BigInteger scaledDividend = bigMultiplyPowerTen(dividend, raise);
			return divideAndRound(scaledDividend, divisor, scale, roundingMode, scale);
		} else {
			int newScale = checkScale(divisor, (long) dividendScale - scale);
			int raise = newScale - divisorScale;
			BigInteger scaledDivisor = bigMultiplyPowerTen(divisor, raise);
			return divideAndRound(dividend, scaledDivisor, scale, roundingMode, scale);
		}
	}

}
