/*
 * Java
 *
 * Copyright 2019-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import ej.fp.FrontPanel;

/**
 * This class holds all methods to send MicroUI Pointer events to MicroUI application.
 */
@SuppressWarnings("nls")
public class EventPointer {

	/**
	 * Usual MicroUI Pointer generator tag used in platform configuration project (<code>microui.xml</code>).
	 */
	public static final String COMMON_MICROUI_GENERATOR_TAG = "POINTER";

	// MicroUI pointer events representation

	// masks & shifts
	private static final int POINTER_DRIVER_TYPE_SHIFT = 28;
	private static final int POINTER_DRIVER_TYPE_MASK = 0x1 << POINTER_DRIVER_TYPE_SHIFT;
	private static final int POINTER_DRIVER_CMD_SHIFT = 24;
	private static final int POINTER_DRIVER_CMD_MASK = 0xf << POINTER_DRIVER_CMD_SHIFT;
	private static final int POINTER_DRIVER_BUTTON_SHIFT = 16;
	private static final int POINTER_DRIVER_BUTTON_MASK = 0xff << POINTER_DRIVER_BUTTON_SHIFT;
	private static final int POINTER_DRIVER_DATA_SHIFT = 0;
	private static final int POINTER_DRIVER_DATA_MASK = 0xffff << POINTER_DRIVER_DATA_SHIFT;

	// types values
	private static final int POINTER_DRIVER_TYPE_ABS = 0x0;
	private static final int POINTER_DRIVER_TYPE_DELTA = 0x1;

	// cmd values
	private static final int POINTER_DRIVER_CMD_MOVE = 0x0;
	private static final int POINTER_DRIVER_CMD_PRESS = 0x1;
	private static final int POINTER_DRIVER_CMD_RELEASE = 0x2;

	private EventPointer() {
		// hide implicit constructor
	}

	/**
	 * Send a PRESSED event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 * @param absolute
	 *            <code>true</code> when (x,y) coordinates are absolute, <code>false</code> when there are relative to
	 *            the previous pointer position.
	 *
	 * @see #sendPressedEvent(String, int, int, int, boolean)
	 */
	public static void sendPressedEvent(int buttonId, int x, int y, boolean absolute) {
		sendPressedEvent(COMMON_MICROUI_GENERATOR_TAG, buttonId, x, y, absolute);
	}

	/**
	 * Send a PRESSED event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Pointer</code> event
	 * generator.
	 *
	 * @param pointerEventGenerator
	 *            the MicroUI <code>Pointer</code> event generator.
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 * @param absolute
	 *            <code>true</code> when (x,y) coordinates are absolute, <code>false</code> when there are relative to
	 *            the previous pointer position.
	 *
	 */
	public static void sendPressedEvent(String pointerEventGenerator, int buttonId, int x, int y, boolean absolute) {
		try {
			sendEvents(pointerEventGenerator, POINTER_DRIVER_CMD_PRESS, x, y, absolute, buttonId);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			StringBuilder buf = new StringBuilder();
			buf.append("press button ");
			buf.append(buttonId);
			buf.append(" at ");
			buf.append(x);
			buf.append(",");
			buf.append(y);
			traceEvent(pointerEventGenerator, buf.toString());
		}
	}

	/**
	 * Send a RELEASED event to the event generator with the MicroUI generator tag
	 * {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 * @see #sendReleasedEvent(String, int)
	 */
	public static void sendReleasedEvent(int buttonId) {
		sendReleasedEvent(COMMON_MICROUI_GENERATOR_TAG, buttonId);
	}

	/**
	 * Send a RELEASED event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Pointer</code> event
	 * generator.
	 *
	 * @param pointerEventGenerator
	 *            the MicroUI <code>Pointer</code> event generator.
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 */
	public static void sendReleasedEvent(String pointerEventGenerator, int buttonId) {
		try {
			sendEvents(pointerEventGenerator, POINTER_DRIVER_CMD_RELEASE, 0/* don't care */, 0/* don't care */,
					false/* don't care */, buttonId);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			traceEvent(pointerEventGenerator, "release button " + buttonId);
		}
	}

	/**
	 * Send a MOVE event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 * @param absolute
	 *            <code>true</code> when (x,y) coordinates are absolute, <code>false</code> when there are relative to
	 *            the previous pointer position.
	 *
	 * @see #sendMovedEvent(String, int, int, boolean)
	 */
	public static void sendMovedEvent(int x, int y, boolean absolute) {
		sendMovedEvent(COMMON_MICROUI_GENERATOR_TAG, x, y, absolute);
	}

	/**
	 * Send a MOVE event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Pointer</code> event
	 * generator.
	 *
	 * @param pointerEventGenerator
	 *            the MicroUI <code>Pointer</code> event generator.
	 * @param x
	 *            the pointer X coordinate.
	 * @param y
	 *            the pointer Y coordinate.
	 * @param absolute
	 *            <code>true</code> when (x,y) coordinates are absolute, <code>false</code> when there are relative to
	 *            the previous pointer position.
	 *
	 */
	public static void sendMovedEvent(String pointerEventGenerator, int x, int y, boolean absolute) {
		try {
			sendEvents(pointerEventGenerator, POINTER_DRIVER_CMD_MOVE, x, y, absolute, 0/* useless */);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			StringBuilder buf = new StringBuilder();
			buf.append("move at ");
			buf.append(x);
			buf.append(",");
			buf.append(y);
			traceEvent(pointerEventGenerator, buf.toString());
		}
	}

	private static void sendEvents(String eventGen, int cmd, int x, int y, boolean abs, int button) {
		int eventX = ((x << POINTER_DRIVER_DATA_SHIFT) & POINTER_DRIVER_DATA_MASK);

		int eventY = 0
				| (((abs ? POINTER_DRIVER_TYPE_ABS : POINTER_DRIVER_TYPE_DELTA) << POINTER_DRIVER_TYPE_SHIFT)
						& POINTER_DRIVER_TYPE_MASK)
				| ((cmd << POINTER_DRIVER_CMD_SHIFT) & POINTER_DRIVER_CMD_MASK)
				| ((button << POINTER_DRIVER_BUTTON_SHIFT) & POINTER_DRIVER_BUTTON_MASK)
				| ((y << POINTER_DRIVER_DATA_SHIFT) & POINTER_DRIVER_DATA_MASK);

		int[] events = new int[2];
		events[0] = LLUIInput.buildEvent(eventGen, eventX);
		events[1] = eventY;
		LLUIInput.sendEvents(events);
	}

	private static void traceEvent(String pointerEventGenerator, String desc) {
		StringBuilder buf = new StringBuilder();
		buf.append("[");
		buf.append(EventPointer.class.getName());
		buf.append("] ");
		buf.append(desc);
		buf.append(" (");
		buf.append(pointerEventGenerator);
		buf.append(")");
		FrontPanel.out.println(buf.toString());
	}
}
