# Overview
AWS IoT Java SDK is a set of Java APIs enabling MicroEJ developers to securely access and interact with
[AWS IoT platform](https://docs.aws.amazon.com/iot/latest/developerguide/what-is-aws-iot.html) from an embedded device.

It can be used to create embedded applications that runs on MicroEJ VEE.

It provides the following functionalities:

- AWS MQTT connection over TLS
- AWS MQTT Topics Pub/Sub
- AWS Device Shadow management (CRUD)


# Usage
Add the following line to your `module.ivy`:

    <dependency org="ej.library.iot" name="aws-iot" rev="2.0.0"/>

# Requirements
This library requires the following Foundation Libraries to be provided by the MicroEJ VEE:

    BON-1.4, EDC-1.3, NET-1.1, SECURITY-1.3, SSL-2.2, TRACE-1.1

# Dependencies
All dependencies are retrieved transitively by MicroEJ Module Manager.
    
# Source
N/A

      
## Code Memory Consumption
AWS IoT Java SDK code memory consumption is approximately ``3.1KB``
	
It has been measured using the [Memory Map Analyzer](https://docs.microej.com/en/latest/ApplicationDeveloperGuide/memoryMapAnalyzer.html)
on the [WROVER 4.1 Platform](https://github.com/MicroEJ/Platform-Espressif-ESP-WROVER-KIT-V4.1) based on Xtensa LX6 Architecture version ``7.16.0``.

The original ``SOAR.map`` file can be found [here](./memoryrequirements/SOAR.map) and can be browsed using the Memory Map Analyzer in the MicroEJ SDK.

This library relies on MicroPaho (MicroEJ lightweight MQTT client implementation). MicroPaho is about ``5.5KB``.



# Documentation
## Client options & connection

The MQTT connections options are provided by the  class ``AwsIotClientOptions``.

```java

	// Build the client connection options using the AwsIotClientOptions.Builder
	AwsIotClientOptions options = AwsIotClientOptions.Builder
		.builder()
		.host("<AWS IoT server URL>")
		.port(8883)// AWS IoT server port
		.clientID("<MQTT client id>")// MQTT client id
		.thingName("<AWS IoT thing name>")
		.timeout(30) //Optional, client connection timeout in seconds. default is 30
		.keepAlive(60) //Optional, client connection keep alive in seconds. default is 60
		.secure(socketFactory)// Optional, an SSL Socket Factory for TLS connection
		.username("username") // Optional, User name to use in MQTT CONNECT
		.password("changeit") // Optional, Password to use in MQTT CONNECT
		.build();


	// Create an instance of AwsIotClient using the connection options
	AwsIotClient client = new AwsIotClient(options);
	client.connect(); // connect

```

**Host**

To get your AWS IoT ``host`` address (endpoint):

- Connect to ``AWS Console``
- Go to ``Iot Core`` Service
- From the left menu, go to ``Setting`` 
- Copy the endpoint from the setting page. 

It looks something like ``*.iot.eu-west-3.amazonaws.com``

**Port**

The MQTT server port. default is ``8883`` for a connection over TLS.

**Client ID**

MQTT client id.

>MQTT client IDs uniquely identify MQTT connections. If a new connection is established using a client ID that is already claimed for another connection, the AWS IoT message broker drops the old connection to allow the new connection. Client IDs must be unique within each AWS account and each AWS Region. This means that you don't need to enforce global uniqueness of client IDs outside of your AWS account or across Regions within your AWS account.
>Source: [AWS IoT Security Best-Practices](https://docs.aws.amazon.com/iot/latest/developerguide/security-best-practices.html)

For Test Only: ``AwsIotClientOptions#generateClientId()`` method can be used to generate a random client id.
The id will have the following format ``MicroEJ<platformTimeNanos>`` where *<platformTimeNanos>* is the current platform time in nanoseconds.


**Thing Name**

The device thing name that was used to register or provision the device in the AWS Iot platform.

**Timeout**

MQTT connection timeout value. This value, measured in seconds,
defines the maximum time interval the client will wait for the network connection to the MQTT server to be established.
The default timeout is 30 seconds. 
A value of 0 disables timeout processing meaning the client will wait until the network connection is made successfully or fails.

**Keep Alive**

This value, measured in seconds, defines the maximum time interval between messages sent or received.
It enables the client to detect if the server is no longer available, without having to wait for the TCP/IP timeout.
The client will ensure that at least one message travels across the network within each keep alive period.
In the absence of a data-related message during the time period, 
the client sends a very small "ping" message, which the server will acknowledge. 
A value of 0 disables keep alive processing in the client. 

The default value is 60 seconds

**Secure (Socket Factory)**

This options let you configure the underlying socket factory to setup a TLS connection for example.


**Username & Password**

MQTT credentials to be used in CONNECT.


## Client Disconnect
----------
Use ``AwsIotClient#disconnect()`` method on a connected client to disconnect the client. This will close the underlying connection socket.

The same client instance can be reused to connect the device again.

```java
client.disconnect();
```

## Client Close
Use ``AwsIotClient#close()`` after disconnecting the clients to clear all the registered subscribers callbacks and
stop the executor service responsible of executing the callbacks when a message is received.

```java
client.close();
```

# AWS MQTT Pub/Sub

## Publish

Message can be published using QOS 0 or 1 to a specific topic using the following APIs:

* ``publish(AwsIotMessage message)``
* ``publish(String topic, byte[] data)``
* ``publish(String topic, byte[] data, int qos)``
* ``publish(String topic, byte[] data, int qos, boolean retained)``

### Publish Example

```java
	String message = "{\"message\" : \"Hello world\"}";
	client.publish("foo/bar", message.getBytes());
```


## Subscribe
Topic subscriptions are managed using a callback mechanism. A callback can be registered on a topic 
and will be executed when a message is received on that specific topic.

Callbacks are executed on a separated thread.

### Subscribe Example
```java
	AwsIotMessageCallback callback =  new AwsIotMessageCallback() {
		
		@Override
		public void onMessageReceived(AwsIotMessage message) {
		    
			// message received..
			System.out.println(message.getTopic());
			System.out.println(message.getQos());
			System.out.println(message.isRetained());
			System.out.println(new String(message.getPayload()));
		}
	};

	// Subscribe to topic
	client.subscribe("foo/bar", callback);
```


# AWS Device Shadow
The AWS IoT Device Shadow service adds shadows to AWS IoT thing objects. 
Shadows can make a device’s state available to apps and other services whether the device is connected to AWS IoT or not. 
AWS IoT thing objects can have multiple named shadows so that your IoT solution has more options for connecting your devices to other apps and services.

[Read more about Device Shadow](https://docs.aws.amazon.com/iot/latest/developerguide/iot-device-shadows.html)

The SDK provides the following API to manage Device Shadow.

**Subscribe to shadow results before performing an action to get the result.**

You can subscribe to none, one or all the possible results ACCEPTED, REJECTED, DELTA, and DOCUMENTS.

## Classic Shadow:

* ``getShadow()``
* ``getShadow(int qos)``
* ``deleteShadow()``
* ``deleteShadow(int qos)``
* ``updateShadow(byte[] message)``
* ``updateShadow(byte[] message, int qos)``
* ``subscribeToShadow(ShadowAction action, ShadowResult result, AwsIotMessageCallback callback)``
* ``subscribeToShadow(ShadowAction action, ShadowResult result, AwsIotMessageCallback callback, int qos)``
* ``unsubscribeFromShadow(ShadowAction action, ShadowResult result)``
		
## Named Shadow:

* ``getShadow(String shadowName)``
* ``getShadow(String shadowName, int qos)``
* ``deleteShadow(String shadowName)``
* ``deleteShadow(String shadowName, int qos)``
* ``updateShadow(String shadowName, byte[] message)``
* ``updateShadow(String shadowName, byte[] message, int qos)``
* ``subscribeToShadow(String shadowName, ShadowAction action, ShadowResult result, AwsIotMessageCallback callback)``
* ``subscribeToShadow(String shadowName, ShadowAction action, ShadowResult result, AwsIotMessageCallback callback, int qos)``
* ``unsubscribeFromShadow(String shadowName, ShadowAction action, ShadowResult result)``


## Get Shadow Example
```	java
		AwsIotMessageCallback onSuccess = new AwsIotMessageCallback() {
			@Override
			public void onMessageReceived(AwsIotMessage message) {
				System.out.println("accepted");
				System.out.println(new String(message.getPayload()));
			}
		}

		AwsIotMessageCallback onError = new AwsIotMessageCallback() {
			@Override
			public void onMessageReceived(AwsIotMessage message) {
				System.out.println("rejected");
				System.out.println(new String(message.getPayload()));
			}
		}

		// subscribe to get results
		client.subscribeToShadow(ShadowAction.get, ShadowResult.accepted, onSuccess);
		client.subscribeToShadow(ShadowAction.get, ShadowResult.rejected, onError);
		
		// Get for the shadow data
		client.getShadow();
```


## Delete Shadow Example
```java

		AwsIotMessageCallback onSuccess = new AwsIotMessageCallback() {
			@Override
			public void onMessageReceived(AwsIotMessage message) {
				System.out.println("accepted");
				System.out.println(new String(message.getPayload()));
			}
		}

		AwsIotMessageCallback onError = new AwsIotMessageCallback() {
			@Override
			public void onMessageReceived(AwsIotMessage message) {
				System.out.println("rejected");
				System.out.println(new String(message.getPayload()));
			}
		}

		//subscribe to delete shadow events
		client.subscribeToShadow(ShadowAction.delete, ShadowResult.accepted, onSuccess);
		client.subscribeToShadow(ShadowAction.delete, ShadowResult.rejected, onError);

		// delete  shadow
		client.deleteShadow()
```

## Create/Update Shadow Example
```java

	String AWS_SHADOW_NAME = "shadowName";
	String message = "{\"state\": {\"reported\": {\"welcome\": \"AWS updated\"}}}";
	
	AwsIotMessageCallback onSuccess = new AwsIotMessageCallback() {
		@Override
		public void onMessageReceived(AwsIotMessage message) {
			System.out.println("accepted");
			System.out.println(new String(message.getPayload()));
		}
	}
	
	AwsIotMessageCallback onError = new AwsIotMessageCallback() {
		@Override
		public void onMessageReceived(AwsIotMessage message) {
			System.out.println("rejected");
			System.out.println(new String(message.getPayload()));
		}
	}
	
	AwsIotMessageCallback delta = new AwsIotMessageCallback() {
		@Override
		public void onMessageReceived(AwsIotMessage message) {
			System.out.println("delta");
			System.out.println(new String(message.getPayload()));
		}
	}
	
	AwsIotMessageCallback documents = new AwsIotMessageCallback() {
		@Override
		public void onMessageReceived(AwsIotMessage message) {
			System.out.println("documents");
			System.out.println(new String(message.getPayload()));
		}
	}

	// Subscribe update shadow results 
	client.subscribeToShadow(ShadowAction.update, ShadowResult.accepted, onSuccess);
	client.subscribeToShadow(ShadowAction.update, ShadowResult.rejected, onError);
	client.subscribeToShadow(ShadowAction.update, ShadowResult.delta, delta);
	client.subscribeToShadow(ShadowAction.update, ShadowResult.documents, documents);
	
    // Create or Update a shadow and
	client.updateShadow(message.getBytes());
```


# Restrictions
The following features specified by MQTT ``3.1.1`` are not supported by this client:

* Other MQTT versions than ``3.1.1``
* Clean Session 0
* QoS 2
* Will messages
* Multiple in-flight messages
* When QoS is 1, the message acknowledgment is sent before the callback method is executed, thus the message is acknowledged even if an exception is thrown by the callback
* Within an implementation of ``AwsIotMessageCallback`` callback, it is not possible to send a new message. Only the following methods are allowed to be called:

  * ``ej.aws.iot.AwsIotClient.getClientOptions()``
  * ``ej.aws.iot.AwsIotClient.isConnected()``

* [AWS IoT differences from MQTT version 3.1.1 specification](https://docs.aws.amazon.com/iot/latest/developerguide/mqtt.html#mqtt-differences)
* Topic filters are not supported when registering a callback during subscribe. The full topic name should be provided. [Read more about AWS Topic filters](https://docs.aws.amazon.com/iot/latest/developerguide/topics.html#topicfilters).

---
_Copyright 2019-2021 MicroEJ Corp. All rights reserved._
_This library is provided in source code for use, modification and test, subject to license terms._
_Any modification of the source code will break MicroEJ Corp. warranties on the whole library._