/*
 * Copyright 2017-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.property.loader;

import ej.annotation.Nullable;
import ej.basictool.ArrayTools;
import ej.property.PropertyLoader;

/**
 * A composite property loader contains a list a property loaders. It searches the requested properties in its children
 * sequentially respecting the order they are added.
 */
public class CompositePropertyLoader implements PropertyLoader {

	/**
	 * Ordered array of property loader to look-up into.
	 */
	private PropertyLoader[] propertyLoaders;

	/**
	 * Creates a new empty composite service loader.
	 */
	public CompositePropertyLoader() {
		this.propertyLoaders = new PropertyLoader[0];
	}

	/**
	 * Adds a property loader at the end of the list a property loaders used by this composite property loader.
	 *
	 * @param propertyLoader
	 *            the property loader to add.
	 * @throws NullPointerException
	 *             if the given property loader is <code>null</code>.
	 */
	public void addPropertyLoader(PropertyLoader propertyLoader) {
		this.propertyLoaders = ArrayTools.add(this.propertyLoaders, propertyLoader);
	}

	@Override
	@Nullable
	public String getProperty(String key) {
		for (PropertyLoader propertyLoader : this.propertyLoaders) {
			String value = propertyLoader.getProperty(key);
			if (null != value) {
				return value;
			}
		}
		return null;
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		String property = getProperty(key);
		if (property != null) {
			return property;
		}
		return defaultValue;
	}
}
