/*
 * 2011-2022 ESR - Not subject to Copyright.
 *
 * This document has been released and published by E-S-R consortium, a non-profit entity.
 * To learn more about E-S-R consortium, please visit http://www.e-s-r.net/.
 * The matter contained in this document is not subject to copyright; you are free to use it for any purpose, for more information see E-S-R consortium policies.
 */
package ej.bon;

import java.util.Date;

/**
 * A task that can be scheduled for one-time or repeated execution by a Timer.
 */
public abstract class TimerTask implements Runnable {

	/**
	 * Creates a new timer task.
	 */
	protected TimerTask() {
		throw new RuntimeException();
	}

	/**
	 * The action to be performed by this timer task.
	 *
	 * @see Thread#run()
	 */
	@Override
	public abstract void run();

	/**
	 * Cancels this timer task. If the task has been scheduled for one-time
	 * execution and has not yet run, or has not yet been scheduled, it will never
	 * run. If the task has been scheduled for repeated execution, it will never run
	 * again. (If the task is running when this call occurs, the task will run to
	 * completion, but will never run again.)
	 *
	 * Note that calling this method from within the run method of a repeating timer
	 * task absolutely guarantees that the timer task will not run again.
	 *
	 * This method may be called repeatedly; the second and subsequent calls have no
	 * effect.
	 *
	 * @return <code>true</code> if this task is scheduled for one-time execution
	 *         and has not yet run, or this task is scheduled for repeated
	 *         execution. Returns <code>false</code> if the task was scheduled for
	 *         one-time execution and has already run, or if the task was never
	 *         scheduled, or if the task was already canceled (Loosely speaking,
	 *         this method returns <code>true</code> if it prevents one or more
	 *         scheduled executions from taking place.)
	 */
	public boolean cancel() {
		throw new RuntimeException();
	}

	/**
	 * Returns the scheduled execution time of the most recent actual execution of
	 * this task. (If this method is invoked while task execution is in progress,
	 * the return value is the scheduled execution time of the ongoing task
	 * execution.)
	 * <p>
	 * This method is typically invoked from within a task's run method, to
	 * determine whether the current execution of the task is sufficiently timely to
	 * warrant performing the scheduled activity:
	 * <pre>
	 * public void run() {
	 * 	if (CurrentTime.get() - scheduledExecutionTime() &gt;= MAX_TARDINESS)
	 * 		return; // Too late; skip this execution.
	 * 	// Perform the task
	 * }
	 * </pre>
	 *
	 * <p>
	 * This method is typically not used in conjunction with fixed-delay execution
	 * repeating tasks, as their scheduled execution times are allowed to drift over
	 * time, and so are not terribly significant.
	 *
	 * @return the time at which the most recent execution of this task was
	 *         scheduled to occur, in the format returned by Date.getTime(). The
	 *         return value is undefined if the task has yet to commence its first
	 *         execution.
	 * @see Date#getTime()
	 */
	public long scheduledExecutionTime() {
		throw new RuntimeException();
	}

	/**
	 * Method invoked when this {@link TimerTask} terminates due to an uncaught
	 * exception thrown by {@link TimerTask#run()}.
	 * <p>
	 * The default implementation of this method handles the uncaught exception as
	 * following:
	 * <ol>
	 * <li>if the {@link Timer} instance has a registered handler, this handler is
	 * invoked,</li>
	 * <li>otherwise, if the {@link Timer} class has a registered default handler,
	 * this handler is invoked,</li>
	 * <li>otherwise, the task is canceled and {@link Throwable#printStackTrace()}
	 * is invoked on the given exception.
	 * </ol>
	 * <p>
	 * Any exception thrown by this method will be ignored.
	 *
	 * @param timer
	 *            The {@link Timer} on which this {@link TimerTask} is scheduled.
	 * @param e
	 *            The uncaught exception.
	 * @see Timer#setDefaultUncaughtExceptionHandler(Thread.UncaughtExceptionHandler)
	 * @see Timer#setUncaughtExceptionHandler(Thread.UncaughtExceptionHandler)
	 */
	public void uncaughtException(Timer timer, Throwable e) {
		throw new RuntimeException();
	}

}
