/*
 * Copyright 2014-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.websocket.frame;

import java.io.PrintStream;
import java.util.logging.Logger;

/**
 * PrettyFramePrinter will analyze the content of a {@link RawFrame} and print it in a pretty human readable manner to a
 * given {@link PrintStream} output.
 *
 *
 *
 */
public class PrettyFramePrinter {

	private static final String BEGIN_END_MARKER = "---------------------------------------------";
	private final Logger out;

	/**
	 * Create a new PrettyFramePrinter that will print on the given output.
	 *
	 * @param out
	 *            the PrintStream to produce the output
	 */
	public PrettyFramePrinter(Logger out) {
		this.out = out;
	}

	/**
	 * Equivalent to <code>PrettyFramePrinter(System.out)</code>.
	 */
	public PrettyFramePrinter() {
		this(Logger.getGlobal());
	}

	/**
	 * Print a frame.
	 *
	 * @param f
	 *            the frame to print
	 */
	public void print(RawFrame f) {
		StringBuilder builder = new StringBuilder();

		builder.append(BEGIN_END_MARKER + "\n");

		// FIN bit
		builder.append("FIN\t" + f.getFIN() + "\n");

		// RSV bits
		builder.append("RSV\t" + f.getRSV() + "\n");

		// Opcode
		builder.append("Opcode\t" + f.getOpcode() + "\n");

		// MASK bit
		builder.append("Mask\t" + f.getMASK() + "\n");

		// Length
		builder.append("Length\t" + f.getLength() + "\n");

		// Extended length
		if (f.hasExtendedLength()) {
			long extLength = f.getExtendedLength();
			builder.append("Extended Length\t" + extLength + "\n");
		}

		// Masking key
		if (f.isMasked()) {
			builder.append("Masking key\t");
			byte[] maskingKey = f.getMaskingKey();
			for (byte b : maskingKey) {
				builder.append(b);
				builder.append("\t");
			}
			builder.append("" + "\n");
		}

		// Payload
		builder.append("Payload\t");
		switch (f.getOpcode()) {
		case RawFrame.OPCODE_TEXT_FRAME: {
			byte[] payload = f.getPayload(false);
			builder.append(new String(payload) + "\n"); // TODO 'true' when unmasking will be implemented
		}

		case RawFrame.OPCODE_BINARY_FRAME:
		case RawFrame.OPCODE_PING:
		case RawFrame.OPCODE_PONG: {
			byte[] payload = f.getPayload(false); // TODO 'true' when unmasking will be implemented
			for (byte b : payload) {
				builder.append(b);
				builder.append(' ');
			}
			builder.append('\n');
			break;
		}

		default:
			builder.append("ERROR: not managed " + f.getOpcode() + "\n");
		}

		builder.append(BEGIN_END_MARKER + "\n");
		this.out.info(builder.toString());
	}
}
