/*
 * Copyright 2014-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.websocket.frame;

import java.io.IOException;
import java.io.InputStream;

import ej.bon.ByteArray;
import ej.util.message.Level;
import ej.websocket.Messages;
import ej.websocket.WebSocket;
import ej.websocket.WebSocketException;

/**
 * A FrameScanner keeps reading the {@link InputStream} (which is supposed to be the input stream of a
 * {@link WebSocket}. Each time the {@link #getFrame()} method is called, it tries to create a {@link RawFrame} from
 * bytes that arrived on the input stream.
 *
 *
 *
 */
public class FrameScanner {

	// Possible states of the scanner. Denote the kind of information it is waiting for.
	private enum WAITING_FOR {
		FIN_RSV_OPCODE, MASK_AND_LENGTH, EXTENDED_LENGTH, VERY_EXTENDED_LENGTH, MASKING_KEY, PAYLOAD, COMPLETE
	}

	private final InputStream is;
	private WAITING_FOR currently;

	/**
	 * Create a new FrameScanner for the given {@link InputStream}.
	 *
	 * @param is
	 *            the input stream to read from. It is supposed to be the stream of a {@link WebSocket}
	 */
	public FrameScanner(InputStream is) {
		this.is = is;
		this.currently = WAITING_FOR.FIN_RSV_OPCODE;
	}

	/**
	 * Get a complete frame from the input stream. This method will block until a frame is completely received or an
	 * {@link IOException} occurs.
	 *
	 * @return a websocket frame as a byte array
	 * @throws IOException
	 *             if an {@link IOException} is encountered by the input stream
	 * @throws WebSocketException
	 *             if the frame is invalid.
	 */
	public RawFrame getFrame() throws IOException, WebSocketException {
		byte finRsvOpcode = 0;
		byte maskLength = 0;

		byte[] extendedLength = null;
		// the actual length of the payload (using both payload length and extended payload length)
		long actualPayloadLength = 0;
		byte[] bytes = new byte[0];

		while (this.currently != WAITING_FOR.COMPLETE) {
			switch (this.currently) {

			case FIN_RSV_OPCODE: {
				finRsvOpcode = getByteFromStream();

				// Move to next state
				this.currently = WAITING_FOR.MASK_AND_LENGTH;
				break;
			}

			case MASK_AND_LENGTH: {
				maskLength = mask();
				actualPayloadLength = getLength(maskLength, actualPayloadLength);
				break;
			}

			case EXTENDED_LENGTH: {
				// Length is a 16-bit wide integer
				extendedLength = new byte[2];
				extendedLength[0] = getByteFromStream();
				extendedLength[1] = getByteFromStream();

				// MSB are received first
				actualPayloadLength = ((extendedLength[0] & 0xFF) << 8) + (extendedLength[1] & 0xFF);

				// Move to next state
				// TODO future implementation: we may receive a masked frame and next case will then be MASKING KEY
				this.currently = WAITING_FOR.PAYLOAD;
				break;
			}

			case VERY_EXTENDED_LENGTH:
				// Length is a 64-bit wide integer
				extendedLength = new byte[8];
				int totalRead = 0;
				while (totalRead < extendedLength.length) {
					int read = this.is.read(extendedLength);
					if (read == -1) {
						throw new IOException();
					} else {
						totalRead += read;
					}
				}

				// MSB are received first
				actualPayloadLength = ByteArray.readLong(extendedLength, 0, ByteArray.BIG_ENDIAN);

				// Move to next state
				// TODO future implementation: we may receive a masked frame and next case will then be MASKING KEY
				this.currently = WAITING_FOR.PAYLOAD;
				break;
			case MASKING_KEY:
				// TODO future version
				// Receiving masked frame is not managed by this implementation
				throw new UnsupportedOperationException(
						Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.UNSUPPORTED_MASKED));

			case PAYLOAD: {
				bytes = getPayload(finRsvOpcode, maskLength, extendedLength, actualPayloadLength);

				// Move to next state
				this.currently = WAITING_FOR.COMPLETE;
				break;
			}

			default:
				// FrameScanner encountered an unexpected case
				throw new IllegalStateException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
						Messages.INVALID_FRAME, this.currently));
			}
		}

		// Reset the scanner state
		this.currently = WAITING_FOR.FIN_RSV_OPCODE;

		// Return the received frame
		return new RawFrame(bytes);
	}

	private byte[] getPayload(byte finRsvOpcode, byte maskLength, byte[] extendedLength, long actualPayloadLength)
			throws IOException {
		byte[] bytes;
		// 'actualPayloadLength' has already been set to the appropriate value
		// We can now create the frame, add header and optional extended length, then finally receive payload and add it
		// to this frame
		long frameLength = ClientFrameBuilder.computeFrameLength(actualPayloadLength);
		if (frameLength > Integer.MAX_VALUE) {
			throw new UnsupportedOperationException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
					Messages.UNSUPPORTED_VERY_EXTENDED_LENGTH));
		}
		bytes = new byte[(int) frameLength];

		// Header
		int offset = 0;
		bytes[offset] = finRsvOpcode;
		offset++;
		bytes[offset] = maskLength;
		offset++;

		// Optional extended length
		if (extendedLength != null) {
			for (byte b : extendedLength) {
				bytes[offset] = b;
				offset++;
			}
		}

		// Receive the payload
		int totalRead = 0;
		while (totalRead < actualPayloadLength) {
			int length = (int) Math.min(Integer.MAX_VALUE, actualPayloadLength - totalRead);
			int chunkRead = this.is.read(bytes, offset + totalRead, length);
			if (chunkRead == -1) {
				throw new IOException();
			}
			totalRead += chunkRead;
		}
		return bytes;
	}

	private byte mask() throws IOException {
		byte maskLength;
		maskLength = getByteFromStream();

		// Mask
		if ((maskLength & 0x80) != 0) {
			// TODO future version
			// FrameScanner cannot receive a masked key
			throw new UnsupportedOperationException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.UNSUPPORTED_MASKED));
		}
		return maskLength;
	}

	private long getLength(byte maskLength, long actualPayloadLength) throws WebSocketException {
		byte payloadLength;
		// Move to next state
		payloadLength = (byte) (maskLength & 0x7F);
		if (payloadLength <= 125) {
			actualPayloadLength = payloadLength;
			this.currently = WAITING_FOR.PAYLOAD;
		} else if (payloadLength == 126) {
			this.currently = WAITING_FOR.EXTENDED_LENGTH;
		} else if (payloadLength == 127) {
			this.currently = WAITING_FOR.VERY_EXTENDED_LENGTH;
		} else {
			// We should never get here
			// FrameScanner failed while decoding length
			throw new WebSocketException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.INVALID_FRAME));
		}
		return actualPayloadLength;
	}

	/**
	 * A convenient method to manage end of stream (should not occur before {@link WebSocket} goes to CLOSING/CLOSED
	 * state and {@link IOException}s.
	 *
	 * @return the byte read from the stream
	 * @throws IOException
	 *             if -1 is read from the stream or if a {@link IOException} occurs
	 */
	private byte getByteFromStream() throws IOException {
		int b = this.is.read();
		if (b == -1) {
			throw new IOException();
		} else {
			return (byte) (b & 0xFF);
		}
	}
}
