/*
 * Java
 *
 * Copyright 2014-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.rcommand.impl;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import ej.annotation.Nullable;
import ej.rcommand.CommandSender;
import ej.rcommand.RemoteCommandListener;
import ej.rcommand.RemoteCommandManager;
import ej.rcommand.RemoteConnection;
import ej.rcommand.RemoteConnectionListener;
import ej.rcommand.RemoteNotification;

public class DefaultRemoteCommandManager implements RemoteCommandManager {

	private static final String THREAD_NAME_PREFIX = "Rcommand ";
	private static final String UNKNOWN_CONNNECTION = "unknown";

	private final HashMap<String, RemoteCommandListener> listeners = new HashMap<>();
	private final Map<RemoteConnection, CommandReader> readers = new HashMap<>();

	private final boolean registerOverwrite;

	/**
	 * Creates a default remote command manager.
	 * <p>
	 * It is possible to choose the policy when a listener is registered that manage the same command as another one.
	 * Either drop the new one or overwrite the old one.
	 *
	 * @param overwrite
	 *            the listener policy, <code>true</code> to overwrite the old one, <code>false</code> otherwise.
	 */
	public DefaultRemoteCommandManager(boolean overwrite) {
		this.registerOverwrite = overwrite;
	}

	/**
	 * Creates a default remote command manager.
	 * <p>
	 * Equivalent as calling {@link DefaultRemoteCommandManager#DefaultRemoteCommandManager(boolean)} with
	 * <code>false</code> as parameter.
	 */
	public DefaultRemoteCommandManager() {
		this(false);
	}

	private class CommandReader implements Runnable {
		private final RemoteConnection connection;
		private final @Nullable RemoteConnectionListener listener;
		private volatile boolean running;

		public CommandReader(RemoteConnection connection, @Nullable RemoteConnectionListener listener) {
			this.connection = connection;
			this.listener = listener;
			this.running = true;
		}

		@Override
		public void run() {
			RemoteConnection connection = this.connection;

			if (this.listener != null) {
				this.listener.startRead();
			}

			while (this.running) {
				String command;
				try {
					try {
						command = connection.readCommand();
					}
					catch(IllegalArgumentException e) {
						// Probably an invalid client is connected
						getLogger().log(Level.SEVERE, e.getMessage(), e);
						throw new IOException(e);
					}
					getLogger().finer("Command received: " + command);
					RemoteCommandListener listener = getListenerFor(command);
					if (listener != null) {
						listener.commandReceived(connection, connection, command);
						// read the "End Of Command" and other remaining
						// potential parameters
						connection.skipParameters();
					} else if (this.running) {
						List<Object> params = connection.readParameters();
						// read the "End Of Command" and other remaining
						// potential parameters
						connection.skipParameters();
						getLogger().severe("Command not managed: " + command + toString(params));
					}
				} catch (IllegalArgumentException e) {
					// Unexpected command parameters received
					getLogger().log(Level.SEVERE, e.getMessage(), e);
					try {
						// read the "End Of Command" and other remaining
						// potential parameters
						connection.skipParameters();
					} catch (IOException e1) {
					}
				} catch (IOException e) {
					if (this.running) {
						onError(connection);
					}
					break;
				}
			}

			if (this.listener != null) {
				this.listener.stopRead();
			}
		}

		public void stopReading() {
			this.running = false;
			this.connection.close();
		}

		public String toString(List<Object> params) {
			StringBuilder sb = new StringBuilder("(");
			for (Object param : params) {
				if (param.getClass().isArray()) {
					byte[] array = (byte[]) param;
					sb.append("byte[" + array.length + "], ");
				} else if (param instanceof String) {
					String s = (String) param;
					sb.append("\"" + s + "\", ");

				} else {
					sb.append(param.toString());
				}
			}
			// remove last ", " if any
			if (sb.length() > 2) {
				sb.delete(sb.length() - 2, sb.length());
			}
			sb.append(')');
			return sb.toString();
		}
	}

	private Logger getLogger() {
		Logger logger = Logger.getLogger(DefaultRemoteCommandManager.class.getName());
		assert (logger != null);
		return logger;
	}

	@Override
	public void registerListener(RemoteCommandListener listener) throws IllegalStateException {
		String[] commands = listener.getManagedCommands();
		for (String command : commands) {
			assert (command != null);
			if (!this.registerOverwrite && this.listeners.containsKey(command)) {
				throw new IllegalStateException("Duplicate listener for " + command);
			}
			this.listeners.put(command, listener);
		}
	}

	/**
	 * Returns the {@link RemoteConnectionListener} for the given command.
	 */
	protected @Nullable RemoteCommandListener getListenerFor(String command) {
		return this.listeners.get(command);
	}

	/**
	 * Can be called once.
	 */
	@Override
	public void startReading(RemoteConnection connection) {
		startReading(connection, UNKNOWN_CONNNECTION, null);
	}

	/**
	 * Can be called once.
	 */
	@Override
	public void startReading(RemoteConnection connection, RemoteConnectionListener listener) {
		startReading(connection, UNKNOWN_CONNNECTION, listener);
	}

	/**
	 * Can be called once.
	 */
	@Override
	public void startReading(RemoteConnection connection, String connectionName) {
		startReading(connection, connectionName, null);
	}

	/**
	 * Can be called once.
	 */
	@Override
	public synchronized void startReading(RemoteConnection connection, String connectionName,
			@Nullable RemoteConnectionListener listener) {
		CommandReader reader = new CommandReader(connection, listener);
		this.readers.put(connection, reader);
		Thread thread = new Thread(reader);
		thread.setName(THREAD_NAME_PREFIX + connectionName);
		thread.start();
	}

	@Override
	public synchronized void stopReading(RemoteConnection connection) {
		CommandReader reader = this.readers.remove(connection);
		if (reader != null) {
			reader.stopReading();
		}
	}

	/** Called when an error occurs in the given connection */
	public void onError(RemoteConnection connection) {
		stopReading(connection);
	}

	@Override
	public synchronized void stopAll() {
		for (CommandReader reader : this.readers.values()) {
			assert (reader != null);
			reader.stopReading();
		}
		this.readers.clear();
	}

	@Override
	public Collection<? extends CommandSender> getCommandSenders(String command) {
		// Currently no filter on command
		return this.readers.keySet();
	}

	@Override
	public Collection<String> getRegisteredCommands() {
		return this.listeners.keySet();
	}

	@Override
	public void sendNotification(RemoteNotification notification) {
		for (RemoteConnection connection : this.readers.keySet()) {
			try {
				connection.startCommand(notification.getName());
				connection.sendInt(-1); // command id
				notification.writeBody(connection);
				connection.flushCommand();
			} catch (IOException e) {
				// ignore exception
			}
		}
	}
}
