/*
 * Java
 *
 * Copyright 2017-2024 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 */
package ej.junit;

public class CheckHelperTestListener implements TestListener {

	private final boolean verbose = Boolean.getBoolean("ej.junit.checkhelper.verbose");

	@Override
	public void testBeforeClassMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.err.println("[JUNIT] Running @BeforeClass method: " + methodName);
		}
	}

	@Override
	public void testInstance(String testClass) {
		if (this.verbose) {
			System.err.println("[JUNIT] Instantiating: " + testClass);
		}
	}

	@Override
	public void testBeforeMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.err.println("[JUNIT] Running @Before method: " + methodName);
		}
	}

	@Override
	public void testAfterMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.err.println("[JUNIT] Running @After method: " + methodName);
		}
	}

	@Override
	public void testAfterClassMethod(String testClass, String methodName) {
		if (this.verbose) {
			System.err.println("[JUNIT] Running @AfterClass method: " + methodName);
		}
	}

	private void log(String text) {
		System.err.println("[JUNIT] " + text);
	}

	@Override
	public void onSuiteStart(String name) {
		log("Suite start\t" + name);
	}

	@Override
	public void onSuiteEnd(String name, int runs, int failures, int errors) {
		log("Suite end\t" + name);
		System.err.println((failures == 0 && errors == 0 ? "PASSED:\t" : "FAILED:\t") + runs + " tests\t" + failures
				+ " failures\t" + errors + " errors");
	}

	@Override
	public void onCaseStart(String name, String className, String file, int line) {
		log("Test start\t" + name + "\t" + className + "\t" + file + "\t" + line);
	}

	@Override
	public void onCaseEnd(String name, double duration) {
		log("Test end\t" + name + "\t" + duration + " seconds");

	}

	@Override
	public void onFailure(String message, String type, String trace) {
		log("Failure\t" + type + "\t" + message);
		log("TRACE START\n" + trace);
		log("TRACE END");
	}

	@Override
	public void onError(String message, String type, String trace) {
		log("Error\t" + type + "\t" + message);
		log("TRACE START\n" + trace);
		log("TRACE END");
	}

	@Override
	public void onSkip(String message) {
		log("Skipped\t" + message);
	}
}
