package java.lang;

import java.util.Map;

import ej.annotation.Nullable;

/**
 * A <i>thread</i> is a thread of execution in a program. The Java Virtual Machine allows an
 * application to have multiple threads of execution running concurrently.
 * <p>
 * Every thread has a priority. Threads with higher priority are executed in preference to threads
 * with lower priority. Each thread may or may not also be marked as a daemon. When code running in
 * some thread creates a new <code>Thread</code> object, the new thread has its priority initially
 * set equal to the priority of the creating thread, and is a daemon thread if and only if the
 * creating thread is a daemon.
 * <p>
 * When a Java Virtual Machine starts up, there is usually a single non-daemon thread (which
 * typically calls the method named <code>main</code> of some designated class). The Java Virtual
 * Machine continues to execute threads until either of the following occurs:
 * <ul>
 * <li>The <code>exit</code> method of class <code>Runtime</code> has been called and the security
 * manager has permitted the exit operation to take place.
 * <li>All threads that are not daemon threads have died, either by returning from the call to the
 * <code>run</code> method or by throwing an exception that propagates beyond the <code>run</code>
 * method.
 * </ul>
 * <p>
 * There are two ways to create a new thread of execution. One is to declare a class to be a
 * subclass of <code>Thread</code>. This subclass should override the <code>run</code> method of
 * class <code>Thread</code>. An instance of the subclass can then be allocated and started. For
 * example, a thread that computes primes larger than a stated value could be written as follows:
 * <hr>
 * <blockquote>
 *
 * <pre>
 * class PrimeThread extends Thread {
 * 	long minPrime;
 *
 * 	PrimeThread(long minPrime) {
 * 		this.minPrime = minPrime;
 * 	}
 *
 * 	public void run() {
 *             // compute primes larger than minPrime
 *              . . .
 *         }
 * }
 * </pre>
 *
 * </blockquote>
 * <hr>
 * <p>
 * The following code would then create a thread and start it running:
 * <blockquote>
 *
 * <pre>
 * PrimeThread p = new PrimeThread(143);
 * p.start();
 * </pre>
 *
 * </blockquote>
 * <p>
 * The other way to create a thread is to declare a class that implements the <code>Runnable</code>
 * interface. That class then implements the <code>run</code> method. An instance of the class can
 * then be allocated, passed as an argument when creating <code>Thread</code>, and started. The same
 * example in this other style looks like the following:
 * <hr>
 * <blockquote>
 *
 * <pre>
 * class PrimeRun implements Runnable {
 * 	long minPrime;
 *
 * 	PrimeRun(long minPrime) {
 * 		this.minPrime = minPrime;
 * 	}
 *
 * 	public void run() {
 *             // compute primes larger than minPrime
 *              . . .
 *         }
 * }
 * </pre>
 *
 * </blockquote>
 * <hr>
 * <p>
 * The following code would then create a thread and start it running:
 * <blockquote>
 *
 * <pre>
 * PrimeRun p = new PrimeRun(143);
 * new Thread(p).start();
 * </pre>
 *
 * </blockquote>
 * <p>
 * Every thread has a name for identification purposes. More than one thread may have the same name.
 * If a name is not specified when a thread is created, a new name is generated for it.
 * <p>
 * Unless otherwise noted, passing a {@code null} argument to a constructor or method in this class
 * will cause a {@link NullPointerException} to be thrown.
 *
 * @see Runnable
 * @see Runtime#exit(int)
 * @see #run()
 */
public class Thread implements Runnable {
	/**
	 * A thread state. A thread can be in one of the following states:
	 * <ul>
	 * <li>{@link #NEW}<br>
	 * A thread that has not yet started is in this state.</li>
	 * <li>{@link #RUNNABLE}<br>
	 * A thread executing in the Java virtual machine is in this state.</li>
	 * <li>{@link #BLOCKED}<br>
	 * A thread that is blocked waiting for a monitor lock is in this state.</li>
	 * <li>{@link #WAITING}<br>
	 * A thread that is waiting indefinitely for another thread to perform a particular action is in
	 * this state.</li>
	 * <li>{@link #TIMED_WAITING}<br>
	 * A thread that is waiting for another thread to perform an action for up to a specified waiting
	 * time is in this state.</li>
	 * <li>{@link #TERMINATED}<br>
	 * A thread that has exited is in this state.</li>
	 * </ul>
	 *
	 * <p>
	 * A thread can be in only one state at a given point in time. These states are virtual machine
	 * states which do not reflect any operating system thread states.
	 *
	 * @see #getState
	 */
	public enum State {
		/**
		 * Thread state for a thread blocked waiting for a monitor lock. A thread in the blocked state is
		 * waiting for a monitor lock to enter a synchronized block/method or reenter a synchronized
		 * block/method after calling {@link Object#wait() Object.wait}.
		 */
		BLOCKED,

		/**
		 * Thread state for a thread which has not yet started.
		 */
		NEW,

		/**
		 * Thread state for a runnable thread. A thread in the runnable state is executing in the Java
		 * virtual machine but it may be waiting for other resources from the operating system such as
		 * processor.
		 */
		RUNNABLE,

		/**
		 * Thread state for a terminated thread. The thread has completed execution.
		 */
		TERMINATED,

		/**
		 * Thread state for a waiting thread with a specified waiting time. A thread is in the timed waiting
		 * state due to calling one of the following methods with a specified positive waiting time:
		 * <ul>
		 * <li>{@link #sleep Thread.sleep}</li>
		 * <li>{@link Object#wait(long) Object.wait} with timeout</li>
		 * <li>{@link #join(long) Thread.join} with timeout</li>
		 * </ul>
		 */
		TIMED_WAITING,

		/**
		 * Thread state for a waiting thread. A thread is in the waiting state due to calling one of the
		 * following methods:
		 * <ul>
		 * <li>{@link Object#wait() Object.wait} with no timeout</li>
		 * <li>{@link #join() Thread.join} with no timeout</li>
		 * </ul>
		 *
		 * <p>
		 * A thread in the waiting state is waiting for another thread to perform a particular action.
		 *
		 * For example, a thread that has called <tt>Object.wait()</tt> on an object is waiting for another
		 * thread to call <tt>Object.notify()</tt> or <tt>Object.notifyAll()</tt> on that object. A thread
		 * that has called <tt>Thread.join()</tt> is waiting for a specified thread to terminate.
		 */
		WAITING;
	}

	/**
	 * Interface for handlers invoked when a <tt>Thread</tt> abruptly terminates due to an uncaught
	 * exception.
	 * <p>
	 * When a thread is about to terminate due to an uncaught exception the Java Virtual Machine will
	 * query the thread for its <tt>UncaughtExceptionHandler</tt> using
	 * {@link #getUncaughtExceptionHandler} and will invoke the handler's <tt>uncaughtException</tt>
	 * method, passing the thread and the exception as arguments.
	 * {@linkplain #getDefaultUncaughtExceptionHandler default uncaught exception handler}.
	 *
	 * @see #setDefaultUncaughtExceptionHandler
	 * @see #setUncaughtExceptionHandler
	 */
	public interface UncaughtExceptionHandler {
		/**
		 * Method invoked when the given thread terminates due to the given uncaught exception.
		 * <p>
		 * Any exception thrown by this method will be ignored by the Java Virtual Machine.
		 *
		 * @param t
		 *        the thread
		 * @param e
		 *        the exception
		 */
		void uncaughtException(Thread t, Throwable e);
	}

	/**
	 * The maximum priority that a thread can have.
	 */
	public static final int MAX_PRIORITY = 10;

	/**
	 * The minimum priority that a thread can have.
	 */
	public static final int MIN_PRIORITY = 1;

	/**
	 * The default priority that is assigned to a thread.
	 */
	public static final int NORM_PRIORITY = 5;

	/**
	 * Returns the current number of active threads in the virtual machine.
	 *
	 * @return the number of active threads
	 */
	public static int activeCount() {
		throw new RuntimeException();
	}

	/**
	 * Returns a reference to the currently executing thread object.
	 *
	 * @return the currently executing thread.
	 */
	public static Thread currentThread() {
		throw new RuntimeException();
	}

	/**
	 * Prints a stack trace of the current thread to the standard error stream. This method is used only
	 * for debugging.
	 *
	 * @see Throwable#printStackTrace()
	 */
	public static void dumpStack() {
		throw new RuntimeException();
	}

	/**
	 * Copies into the specified array every active thread.
	 *
	 * <p>
	 * An application might use the {@linkplain #activeCount activeCount} method to get an estimate of
	 * how big the array should be, however <i>if the array is too short to hold all the threads, the
	 * extra threads are silently ignored.</i> If it is critical to obtain every active thread in the
	 * current thread's thread group and its subgroups, the invoker should verify that the returned int
	 * value is strictly less than the length of {@code tarray}.
	 *
	 * <p>
	 * Due to the inherent race condition in this method, it is recommended that the method only be used
	 * for debugging and monitoring purposes.
	 *
	 * @param tarray
	 *        an array into which to put the list of threads
	 *
	 * @return the number of threads put into the array
	 */
	public static int enumerate(Thread tarray[]) {
		throw new RuntimeException();
	}

	/**
	 * Returns a map of stack traces for all live threads. The map keys are threads and each map value
	 * is an array of <tt>StackTraceElement</tt> that represents the stack dump of the corresponding
	 * <tt>Thread</tt>. The returned stack traces are in the format specified for the
	 * {@link #getStackTrace getStackTrace} method.
	 *
	 * <p>
	 * The threads may be executing while this method is called. The stack trace of each thread only
	 * represents a snapshot and each stack trace may be obtained at different time. A zero-length array
	 * will be returned in the map value if the virtual machine has no stack trace information about a
	 * thread.
	 *
	 * <p>
	 * If there is a security manager, then the security manager's <tt>checkPermission</tt> method is
	 * called with a <tt>RuntimePermission("getStackTrace")</tt> permission to see if it is ok to get
	 * the stack trace of all threads.
	 *
	 * @return a <tt>Map</tt> from <tt>Thread</tt> to an array of <tt>StackTraceElement</tt> that
	 *         represents the stack trace of the corresponding thread.
	 *
	 * @throws SecurityException
	 *         if a security manager exists and its <tt>checkPermission</tt> method doesn't allow
	 *         getting the stack trace of thread.
	 * @see #getStackTrace
	 * @see SecurityManager#checkPermission
	 * @see RuntimePermission
	 * @see Throwable#getStackTrace
	 *
	 */
	public static Map<Thread, StackTraceElement[]> getAllStackTraces() {
		throw new RuntimeException();
	}

	/**
	 * Returns the default handler invoked when a thread abruptly terminates due to
	 * an uncaught exception. If the returned value is <tt>null</tt>, there is no
	 * default.
	 *
	 * @return the default uncaught exception handler for all threads
	 *
	 * @see #setDefaultUncaughtExceptionHandler
	 */
	@Nullable
	public static UncaughtExceptionHandler getDefaultUncaughtExceptionHandler() {
		throw new RuntimeException();
	}

	/**
	 * Returns <tt>true</tt> if and only if the current thread holds the monitor lock on the specified
	 * object.
	 *
	 * <p>
	 * This method is designed to allow a program to assert that the current thread already holds a
	 * specified lock:
	 *
	 * <pre>
	 * assert Thread.holdsLock(obj);
	 * </pre>
	 *
	 * @param obj
	 *        the object on which to test lock ownership
	 * @throws NullPointerException
	 *         if obj is <tt>null</tt>
	 * @return <tt>true</tt> if the current thread holds the monitor lock on the specified object.
	 */
	public static boolean holdsLock(Object obj) {
		throw new RuntimeException();
	}

	/**
	 * Tests whether the current thread has been interrupted. The <i>interrupted status</i> of the
	 * thread is cleared by this method. In other words, if this method were to be called twice in
	 * succession, the second call would return false (unless the current thread were interrupted again,
	 * after the first call had cleared its interrupted status and before the second call had examined
	 * it).
	 *
	 * <p>
	 * A thread interruption ignored because a thread was not alive at the time of the interrupt will be
	 * reflected by this method returning false.
	 *
	 * @return <code>true</code> if the current thread has been interrupted; <code>false</code>
	 *         otherwise.
	 * @see #isInterrupted()
	 */
	public static boolean interrupted() {
		throw new RuntimeException();
	}

	/**
	 * Set the default handler invoked when a thread abruptly terminates due to an uncaught exception,
	 * and no other handler has been defined for that thread.
	 *
	 * <p>
	 * Uncaught exception handling is controlled first by the thread and by the default uncaught
	 * exception handler. If the thread does not have an explicit uncaught exception handler set, and
	 * the thread's thread group (including parent thread groups) does not specialize its
	 * <tt>uncaughtException</tt> method, then the default handler's <tt>uncaughtException</tt> method
	 * will be invoked.
	 * <p>
	 * By setting the default uncaught exception handler, an application can change the way in which
	 * uncaught exceptions are handled (such as logging to a specific device, or file) for those threads
	 * that would already accept whatever &quot;default&quot; behavior the system provided.
	 *
	 * @param eh
	 *        the object to use as the default uncaught exception handler. If <tt>null</tt> then there
	 *        is no default handler.
	 *
	 * @throws SecurityException
	 *         if a security manager is present and it denies <tt>{@link RuntimePermission}
	 *         (&quot;setDefaultUncaughtExceptionHandler&quot;)</tt>
	 *
	 * @see #setUncaughtExceptionHandler
	 * @see #getUncaughtExceptionHandler
	 */
	public static void setDefaultUncaughtExceptionHandler(@Nullable UncaughtExceptionHandler eh) {
		throw new RuntimeException();
	}

	/**
	 * Causes the currently executing thread to sleep (temporarily cease execution) for the specified
	 * number of milliseconds, subject to the precision and accuracy of system timers and schedulers.
	 * The thread does not lose ownership of any monitors.
	 *
	 * @param millis
	 *        the length of time to sleep in milliseconds
	 *
	 * @throws IllegalArgumentException
	 *         if the value of {@code millis} is negative
	 *
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of the
	 *         current thread is cleared when this exception is thrown.
	 */
	public static void sleep(long millis) throws InterruptedException {
		throw new RuntimeException();
	}

	/**
	 * Causes the currently executing thread to sleep (temporarily cease execution) for the specified
	 * number of milliseconds plus the specified number of nanoseconds, subject to the precision and
	 * accuracy of system timers and schedulers. The thread does not lose ownership of any monitors.
	 *
	 * @param millis
	 *        the length of time to sleep in milliseconds
	 *
	 * @param nanos
	 *        {@code 0-999999} additional nanoseconds to sleep
	 *
	 * @throws IllegalArgumentException
	 *         if the value of {@code millis} is negative, or the value of {@code nanos} is not in the
	 *         range {@code 0-999999}
	 *
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of the
	 *         current thread is cleared when this exception is thrown.
	 */
	public static void sleep(long millis, int nanos) throws InterruptedException {
		throw new RuntimeException();
	}

	/**
	 * A hint to the scheduler that the current thread is willing to yield its current use of a
	 * processor. The scheduler is free to ignore this hint.
	 *
	 * <p>
	 * Yield is a heuristic attempt to improve relative progression between threads that would otherwise
	 * over-utilise a CPU. Its use should be combined with detailed profiling and benchmarking to ensure
	 * that it actually has the desired effect.
	 */
	public static void yield() {
		throw new RuntimeException();
	}

	/**
	 * Allocates a new {@code Thread} object.
	 */
	public Thread() {
		throw new RuntimeException();
	}

	/**
	 * Allocates a new Thread object with a specific target object whose run method is called.
	 *
	 * @param target
	 *        the object whose {@code run} method is invoked when this thread is started. If
	 *        {@code null}, this classes {@code run} method does nothing.
	 */
	public Thread(@Nullable Runnable target) {
		throw new RuntimeException();
	}

	/**
	 * Allocates a new {@code Thread} object.
	 *
	 * @param target
	 *        the object whose {@code run} method is invoked when this thread is started. If
	 *        {@code null}, this classes {@code run} method does nothing.
	 * @param name
	 *        the name of the new thread
	 */
	public Thread(@Nullable Runnable target, String name) {
		throw new RuntimeException();
	}

	/**
	 * Allocates a new {@code Thread} object.
	 *
	 * @param name
	 *        the name of the new thread
	 */
	public Thread(String name) {
		throw new RuntimeException();
	}

	/**
	 * Determines if the currently running thread has permission to modify this thread.
	 * <p>
	 * If there is a security manager, its <code>checkAccess</code> method is called with this thread as
	 * its argument. This may result in throwing a <code>SecurityException</code>.
	 *
	 * @exception SecurityException
	 *            if the current thread is not allowed to access this thread.
	 * @see SecurityManager#checkAccess(Thread)
	 */
	public final void checkAccess() {
		throw new RuntimeException();
	}

	/**
	 * Returns the identifier of this Thread. The thread ID is a positive <tt>long</tt> number generated
	 * when this thread was created. The thread ID is unique and remains unchanged during its lifetime.
	 * When a thread is terminated, this thread ID may be reused.
	 *
	 * @return this thread's ID.
	 */
	public long getId() {
		throw new RuntimeException();
	}

	/**
	 * Returns this thread's name.
	 *
	 * @return this thread's name.
	 * @see #setName(String)
	 */
	public final String getName() {
		throw new RuntimeException();
	}

	/**
	 * Returns this thread's priority.
	 *
	 * @return this thread's priority.
	 * @see #setPriority
	 */
	public final int getPriority() {
		throw new RuntimeException();
	}

	/**
	 * Returns an array of stack trace elements representing the stack dump of this thread. This method
	 * will return a zero-length array if this thread has not started, has started but has not yet been
	 * scheduled to run by the system, or has terminated. If the returned array is of non-zero length
	 * then the first element of the array represents the top of the stack, which is the most recent
	 * method invocation in the sequence. The last element of the array represents the bottom of the
	 * stack, which is the least recent method invocation in the sequence.
	 *
	 * <p>
	 * If there is a security manager, and this thread is not the current thread, then the security
	 * manager's <tt>checkPermission</tt> method is called with a
	 * <tt>RuntimePermission("getStackTrace")</tt> permission to see if it's ok to get the stack trace.
	 *
	 * <p>
	 * Some virtual machines may, under some circumstances, omit one or more stack frames from the stack
	 * trace. In the extreme case, a virtual machine that has no stack trace information concerning this
	 * thread is permitted to return a zero-length array from this method.
	 *
	 * @return an array of <tt>StackTraceElement</tt>, each represents one stack frame.
	 *
	 * @throws SecurityException
	 *         if a security manager exists and its <tt>checkPermission</tt> method doesn't allow
	 *         getting the stack trace of thread.
	 * @see SecurityManager#checkPermission
	 * @see RuntimePermission
	 * @see Throwable#getStackTrace
	 *
	 */
	public StackTraceElement[] getStackTrace() {
		throw new RuntimeException();
	}

	/**
	 * Returns the state of this thread. This method is designed for use in monitoring of the system
	 * state, not for synchronization control.
	 *
	 * @return this thread's state.
	 */
	public State getState() {
		throw new RuntimeException();
	}

	/**
	 * Returns the handler invoked when this thread abruptly terminates due to an
	 * uncaught exception. If this thread has not had an uncaught exception handler
	 * explicitly set then <tt>null</tt> is returned.
	 *
	 * @return the uncaught exception handler for this thread
	 */
	@Nullable
	public UncaughtExceptionHandler getUncaughtExceptionHandler() {
		throw new RuntimeException();
	}

	/**
	 * Interrupts this thread. In an implementation conforming to the CLDC Specification, this operation
	 * is not required to cancel or clean up any pending I/O operations that the thread may be waiting
	 * for.
	 */
	public void interrupt() {
		throw new RuntimeException();
	}

	/**
	 * Tests if this thread is alive. A thread is alive if it has been started and has not yet died.
	 *
	 * @return <code>true</code> if this thread is alive; <code>false</code> otherwise.
	 */
	public final boolean isAlive() {
		throw new RuntimeException();
	}

	/**
	 * Tests whether this thread has been interrupted. The <i>interrupted status</i> of the thread is
	 * unaffected by this method.
	 *
	 * <p>
	 * A thread interruption ignored because a thread was not alive at the time of the interrupt will be
	 * reflected by this method returning false.
	 *
	 * @return <code>true</code> if this thread has been interrupted; <code>false</code> otherwise.
	 * @see #interrupted()
	 */
	public boolean isInterrupted() {
		throw new RuntimeException();
	}

	/**
	 * Waits for this thread to die.
	 *
	 * <p>
	 * An invocation of this method behaves in exactly the same way as the invocation
	 *
	 * <blockquote> {@linkplain #join(long) join}{@code (0)} </blockquote>
	 *
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of the
	 *         current thread is cleared when this exception is thrown.
	 */
	public final void join() throws InterruptedException {
		throw new RuntimeException();
	}

	/**
	 * Waits at most {@code millis} milliseconds for this thread to die. A timeout of {@code 0} means to
	 * wait forever.
	 *
	 * <p>
	 * This implementation uses a loop of {@code this.wait} calls conditioned on {@code this.isAlive}.
	 * As a thread terminates the {@code this.notifyAll} method is invoked. It is recommended that
	 * applications not use {@code wait}, {@code notify}, or {@code notifyAll} on {@code Thread}
	 * instances.
	 *
	 * @param millis
	 *        the time to wait in milliseconds
	 *
	 * @throws IllegalArgumentException
	 *         if the value of {@code millis} is negative
	 *
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of the
	 *         current thread is cleared when this exception is thrown.
	 */
	public final synchronized void join(long millis) throws InterruptedException {
		throw new RuntimeException();
	}

	/**
	 * Waits at most {@code millis} milliseconds plus {@code nanos} nanoseconds for this thread to die.
	 *
	 * <p>
	 * This implementation uses a loop of {@code this.wait} calls conditioned on {@code this.isAlive}.
	 * As a thread terminates the {@code this.notifyAll} method is invoked. It is recommended that
	 * applications not use {@code wait}, {@code notify}, or {@code notifyAll} on {@code Thread}
	 * instances.
	 *
	 * @param millis
	 *        the time to wait in milliseconds
	 *
	 * @param nanos
	 *        {@code 0-999999} additional nanoseconds to wait
	 *
	 * @throws IllegalArgumentException
	 *         if the value of {@code millis} is negative, or the value of {@code nanos} is not in the
	 *         range {@code 0-999999}
	 *
	 * @throws InterruptedException
	 *         if any thread has interrupted the current thread. The <i>interrupted status</i> of the
	 *         current thread is cleared when this exception is thrown.
	 */
	public final synchronized void join(long millis, int nanos) throws InterruptedException {
		throw new RuntimeException();
	}

	/**
	 * If this thread was constructed using a separate Runnable run object, then that Runnable object's
	 * run method is called; otherwise, this method does nothing and returns. Subclasses of Thread
	 * should override this method.
	 */
	@Override
	public void run() {
		throw new RuntimeException();
	}

	/**
	 * Changes the name of this thread to be equal to the argument <code>name</code>.
	 * <p>
	 * First the <code>checkAccess</code> method of this thread is called with no arguments. This may
	 * result in throwing a <code>SecurityException</code>.
	 *
	 * @param name
	 *        the new name for this thread.
	 * @exception SecurityException
	 *            if the current thread cannot modify this thread.
	 * @see #getName
	 */
	public final void setName(String name) {
		throw new RuntimeException();
	}

	/**
	 * Marks this thread as either a {@linkplain #isDaemon daemon} thread
	 * or a user thread. The Java Virtual Machine exits when the only
	 * threads running are all daemon threads.
	 *
	 * <p> This method must be invoked before the thread is started.
	 *
	 * @param  on
	 *         if {@code true}, marks this thread as a daemon thread
	 *
	 * @throws  IllegalThreadStateException
	 *          if this thread is {@linkplain #isAlive alive}
	 *
	 * @throws  SecurityException
	 *          if {@link #checkAccess} determines that the current
	 *          thread cannot modify this thread
	 */
	public final void setDaemon(boolean on) {
		throw new RuntimeException();
	}

	/**
	 * Tests if this thread is a daemon thread.
	 *
	 * @return  <code>true</code> if this thread is a daemon thread;
	 *          <code>false</code> otherwise.
	 * @see     #setDaemon(boolean)
	 */
	public final boolean isDaemon() {
		throw new RuntimeException();
	}

	/**
	 * Changes the priority of this thread.
	 * <p>
	 *
	 * @param newPriority
	 *        priority to set this thread to
	 * @exception IllegalArgumentException
	 *            If the priority is not in the range <code>MIN_PRIORITY</code> to
	 *            <code>MAX_PRIORITY</code>.
	 * @exception SecurityException
	 *            if the current thread cannot modify this thread.
	 * @see #getPriority
	 * @see #MAX_PRIORITY
	 * @see #MIN_PRIORITY
	 */
	public final void setPriority(int newPriority) {
		throw new RuntimeException();
	}

	/**
	 * Set the handler invoked when this thread abruptly terminates due to an uncaught exception.
	 * <p>
	 * A thread can take full control of how it responds to uncaught exceptions by having its uncaught
	 * exception handler explicitly set.
	 *
	 * @param eh
	 *        the object to use as this thread's uncaught exception handler. If <tt>null</tt> then this
	 *        thread has no explicit handler.
	 * @throws SecurityException
	 *         if the current thread is not allowed to modify this thread.
	 * @see #setDefaultUncaughtExceptionHandler
	 */
	public void setUncaughtExceptionHandler(@Nullable UncaughtExceptionHandler eh) {
		throw new RuntimeException();
	}

	/**
	 * Causes this thread to begin execution; the Java Virtual Machine calls the <code>run</code> method
	 * of this thread.
	 * <p>
	 * The result is that two threads are running concurrently: the current thread (which returns from
	 * the call to the <code>start</code> method) and the other thread (which executes its
	 * <code>run</code> method).
	 * <p>
	 * It is never legal to start a thread more than once. In particular, a thread may not be restarted
	 * once it has completed execution.
	 *
	 * @exception IllegalThreadStateException
	 *            if the thread was already started.
	 * @see #run()
	 */
	public synchronized void start() {
		throw new RuntimeException();
	}

	/**
	 * Returns a string representation of this thread, including the thread's name, priority, and thread
	 * group.
	 *
	 * @return a string representation of this thread.
	 */
	@Override
	public String toString() {
		throw new RuntimeException();
	}

}