/*
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.wear.framework.components;

import ej.microui.display.GraphicsContext;

/**
 * Represents a source of data that can be displayed on a watchface complication.
 */
public interface ComplicationDataSource {

	/**
	 * Returns whether this source provides a text.
	 *
	 * @return whether this source provides a text.
	 */
	boolean hasText();

	/**
	 * Returns whether this source provides an icon.
	 *
	 * @return whether this source provides an icon.
	 */
	boolean hasIcon();

	/**
	 * Returns whether this source provides a progress.
	 *
	 * @return whether this source provides a progress.
	 */
	boolean hasProgress();

	/**
	 * Returns the text.
	 *
	 * @return the text.
	 * @throws IllegalStateException
	 *             if this source does not provide a text.
	 */
	String getText();

	/**
	 * Renders the icon.
	 *
	 * @param g
	 *            the graphics context to draw with.
	 * @param x
	 *            the x coordinate of the region to draw in.
	 * @param y
	 *            the y coordinate of the region to draw in.
	 * @param width
	 *            the width of the region to draw in.
	 * @param height
	 *            the height of the region to draw in.
	 * @throws IllegalStateException
	 *             if this source does not provide an icon.
	 */
	void renderIcon(GraphicsContext g, int x, int y, int width, int height);

	/**
	 * Returns the progress.
	 *
	 * @return the progress (between <code>0.0f</code> and <code>1.0f</code>).
	 * @throws IllegalStateException
	 *             if this source does not provide a progress.
	 */
	float getProgress();
}
