/*
 * Java
 *
 * Copyright 2016-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.nls;

import java.io.IOException;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.bon.ResourceArray;
import ej.bon.ResourceBuffer;
import ej.nls.NLS;
import ej.util.message.Level;
import ej.util.message.MessageBuilder;
import ej.util.message.MessageLogger;
import ej.util.message.basic.BasicMessageBuilder;
import ej.util.message.basic.BasicMessageLogger;

/**
 * This {@link NLS} implementation uses a binary file as entry point. This binary file is encoded by the NLS PO addon
 * processor.
 */
public class BinaryNLS implements NLS {

	/**
	 * Binary NLS format version. Set by NLS PO compiler.
	 */
	private static final int FORMAT_VERSION = 1;

	/**
	 * Property which allows to fix the default locale at compiletime.
	 */
	public static final String NLS_DEFAULT_LOCALE = "com.microej.nlspo.defaultLocale"; //$NON-NLS-1$

	/**
	 * Property which allows to define a fallback NLS implementation.
	 *
	 * @see #newBinaryNLS(String, int)
	 */
	public static final String NLS_DEFAULT_IMPLEMENTATION = "com.microej.nlspo.defaultImplementation"; //$NON-NLS-1$

	/**
	 * Error messages category.
	 */
	private static final String NLS_PO_CATEGORY = "NLS-PO"; //$NON-NLS-1$
	/**
	 * Error code when binary is missing.
	 */
	public static final int MISSING_BINARY = 1;
	/**
	 * Error code when binary is incompatible: wrong header.
	 */
	public static final int WRONG_HEADER = 2;
	/**
	 * Error code when binary is incompatible: wrong format version.
	 */
	public static final int WRONG_FORMAT_VERSION = 3;
	/**
	 * Error code when binary is incompatible: wrong CRC.
	 */
	public static final int WRONG_CRC = 4;
	/**
	 * Error code when default implementation cannot be instantiated.
	 */
	public static final int DEFAULT_IMPLEMENTATION_ERROR = 5;
	/**
	 * Error code when required implementation cannot be instantiated.
	 */
	public static final int REQUIRED_IMPLEMENTATION_ERROR = 6;

	/**
	 * Error message builder instance.
	 */
	private static final MessageBuilder ERROR_MESSAGE_BUILDER = new BasicMessageBuilder();

	/**
	 * Error message Logger instance.
	 */
	private static final MessageLogger ERROR_MESSAGE_LOGGER = new BasicMessageLogger(ERROR_MESSAGE_BUILDER);

	/**
	 * Array of offsets locales data
	 */
	private final ResourceArray localesArray;

	/**
	 * Array of locales names
	 */
	private final String[] localesNames;

	/**
	 * Current locale set by user
	 */
	private Locale currentLocale;

	/**
	 * Decodes a binary NLS file for the given interface.
	 * <p>
	 * If an error occurs while creating the NLS, a default NLS implementation instance is returned if the property
	 * {@link #NLS_DEFAULT_IMPLEMENTATION} is defined, otherwise, <code>null</code> is returned.
	 *
	 * @param nlsInterfaceName
	 *            the name of the NLS interface which lists the messages ID.
	 * @param keysCRC32
	 *            the CRC32 of all messages keys.
	 * @return an NLS instance.
	 * @throws IllegalArgumentException
	 *             if the default implementation cannot be instantiated.
	 * @see #BinaryNLS(String, int)
	 */
	@Nullable
	public static NLS newBinaryNLS(String nlsInterfaceName, int keysCRC32) {
		try {
			return new BinaryNLS(nlsInterfaceName, keysCRC32);
		} catch (IOException e) {
			ERROR_MESSAGE_LOGGER.log(Level.INFO, NLS_PO_CATEGORY, REQUIRED_IMPLEMENTATION_ERROR, e);
			return createDefaultImplementation();
		}
	}

	@Nullable
	private static NLS createDefaultImplementation() {
		String implementationName = System.getProperty(NLS_DEFAULT_IMPLEMENTATION);
		if (implementationName != null) {
			try {
				return (NLS) Class.forName(implementationName).newInstance();
			} catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
				throw new IllegalArgumentException(ERROR_MESSAGE_BUILDER.buildMessage(ej.util.message.Level.SEVERE,
						NLS_PO_CATEGORY, DEFAULT_IMPLEMENTATION_ERROR, implementationName));
			}
		}
		return null;
	}

	/**
	 * Decodes a binary NLS file for the given interface. The name of binary file is the interface full name (package +
	 * interface simple name) + ".nls" extension. This binary file must have been compiled thanks the binary NLS addon
	 * processor.
	 *
	 * @param nlsInterfaceName
	 *            the name of the NLS interface which lists the messages ID.
	 * @param keysCRC32
	 *            the CRC32 of all messages keys. This allows to verify the coherence between the encoded messages and
	 *            the given interface. This CRC32 is a field of the given interface.
	 * @throws IOException
	 *             if the expected binary file is not available in application classpath or if the binary file content
	 *             is not recognized (probably because not encoded by the Binary NLS addon processor).
	 * @throws IllegalArgumentException
	 *             if the default locale defined by the property {@link #NLS_DEFAULT_LOCALE} is not available in the
	 *             binary file.
	 */
	public BinaryNLS(String nlsInterfaceName, int keysCRC32) throws IOException {
		ResourceBuffer buf = loadBinFile(nlsInterfaceName, keysCRC32);
		@SuppressWarnings("null")
		@NonNull
		ResourceArray localesArray = buf.readArray();
		this.localesArray = localesArray;
		this.localesNames = loadLocales(this.localesArray);
		this.currentLocale = getDefaultLocale();
	}

	/**
	 * Decodes a binary NLS file for the given interface. The name of binary file is the interface full name (package +
	 * interface simple name) + ".nls" extension. This binary file must have been compiled thanks the binary NLS addon
	 * processor.
	 *
	 * @param nlsInterface
	 *            the NLS interface which lists the messages ID.
	 * @param keysCRC32
	 *            the CRC32 of all messages keys. This allows to verify the coherence between the encoded messages and
	 *            the given interface. This CRC32 is a field of the given interface.
	 * @throws IOException
	 *             if the expected binary file is not available in application classpath or if the binary file content
	 *             is not recognized (probably because not encoded by the Binary NLS addon processor).
	 * @throws IllegalArgumentException
	 *             if the default locale defined by the property {@link #NLS_DEFAULT_LOCALE} is not available in the
	 *             binary file.
	 */
	@SuppressWarnings("null") // The name of a class is not null for sure.
	public BinaryNLS(Class<? extends Object> nlsInterface, int keysCRC32) throws IOException {
		this(nlsInterface.getName(), keysCRC32);
	}

	@Override
	public String getDisplayName(String locale) {
		return getMessage(0, locale);
	}

	@Override
	public String getMessage(int messageID) {
		try {
			return this.currentLocale.getMessage(messageID);
		} catch (IndexOutOfBoundsException e) {
			// invalid message ID
			throw new ArrayIndexOutOfBoundsException();
		} catch (IOException e) {
			// cannot throw another exception (NLS API)
			throw new RuntimeException();
		}
	}

	@Override
	public String getMessage(int messageID, String locale) {
		try {
			return getLocale(locale).getMessage(messageID);
		} catch (IndexOutOfBoundsException e) {
			// invalid message ID
			throw new ArrayIndexOutOfBoundsException();
		} catch (IOException e) {
			// cannot throw another exception (NLS API)
			throw new RuntimeException();
		}
	}

	@Override
	public void setCurrentLocale(String locale) {
		if (!locale.equals(this.localesNames[this.currentLocale.index])) {
			try {
				this.currentLocale = getLocale(locale);
			} catch (IOException e) {
				// no error: keep previous locale
			}
		}
	}

	@Override
	public String[] getAvailableLocales() {
		// perform a copy to not alterate original locales array
		int l = this.localesNames.length;
		String[] out = new String[l];
		System.arraycopy(this.localesNames, 0, out, 0, l);
		return out;
	}

	@Override
	public String getCurrentLocale() {
		@SuppressWarnings("null")
		@NonNull
		String locale = this.localesNames[this.currentLocale.index];
		return locale;
	}

	private Locale getDefaultLocale() throws IOException, IllegalArgumentException {
		// check if a default locale is defined
		String defaultLocale = System.getProperty(NLS_DEFAULT_LOCALE);
		if (defaultLocale != null) {
			try {
				return getLocale(defaultLocale);
			} catch (IOException e) {
				throw new IllegalArgumentException(e);
			}
		}

		// sanity check
		if (this.localesNames.length == 0) {
			// should never occur because the binary NLS processor does not generate a file without locale
			throw new RuntimeException();
		}

		// no defined default locales, take the first available one
		return getLocale(0);
	}

	private String[] loadLocales(ResourceArray localesArray) throws IOException {
		int nbLocales = localesArray.length();
		String[] locales = new String[nbLocales];

		for (int i = -1; ++i < nbLocales;) {

			// go to the locale data position
			ResourceBuffer buf = localesArray.seekToElementPointer(i);

			// skip locale messages array
			buf.readArray();
			buf.align(4);

			// here: buf points on locale name
			locales[i] = buf.readString();
		}

		return locales;
	}

	private ResourceBuffer loadBinFile(String nlsInterfaceName, int keysCRC32) throws IOException {
		String binFilePath = '/' + nlsInterfaceName.replace('.', '/') + ".nls"; //$NON-NLS-1$
		ResourceBuffer rb;
		try {
			rb = new ResourceBuffer(binFilePath);
		} catch (IOException e) {
			throw new IOException(ERROR_MESSAGE_BUILDER.buildMessage(Level.SEVERE, NLS_PO_CATEGORY, MISSING_BINARY));
		}

		// check signature, version and crc32
		if (rb.readByte() != (byte) 'M' || rb.readByte() != (byte) 'E' || rb.readByte() != (byte) 'J'
				|| rb.readByte() != (byte) '_' || rb.readByte() != (byte) 'N' || rb.readByte() != (byte) 'L'
				|| rb.readByte() != (byte) 'S') {
			throw new IOException(ERROR_MESSAGE_BUILDER.buildMessage(Level.SEVERE, NLS_PO_CATEGORY, WRONG_HEADER));
		}
		byte readFormat = rb.readByte();
		if (readFormat != FORMAT_VERSION) {
			throw new IOException(ERROR_MESSAGE_BUILDER.buildMessage(Level.SEVERE, NLS_PO_CATEGORY,
					WRONG_FORMAT_VERSION, new Integer(readFormat), new Integer(FORMAT_VERSION)));
		}
		int readCRC = rb.readInt();
		if (readCRC != keysCRC32) {
			throw new IOException(ERROR_MESSAGE_BUILDER.buildMessage(Level.SEVERE, NLS_PO_CATEGORY, WRONG_CRC,
					new Integer(readCRC), new Integer(keysCRC32)));
		}

		return rb;
	}

	private int getLocaleIndex(String locale) throws IOException {
		String[] locales = this.localesNames;
		for (int i = locales.length; --i >= 0;) {
			if (locales[i].equals(locale)) {
				// expected locale is defined
				return i;
			}
		}
		// expected locale not found
		throw new IOException();
	}

	private Locale getLocale(String locale) throws IOException {
		return getLocale(getLocaleIndex(locale));
	}

	private Locale getLocale(int localeIndex) throws IOException {
		synchronized (this.localesArray.getBuffer()) {
			@SuppressWarnings("null")
			@NonNull
			ResourceArray array = this.localesArray.seekToElementPointer(localeIndex).readArray();
			return new Locale(localeIndex, array);
		}
	}
}

class Locale {

	final ResourceArray messagesOffsets;
	final int index;

	Locale(int index, ResourceArray messagesOffsets) {
		this.index = index;
		this.messagesOffsets = messagesOffsets;
	}

	String getMessage(int messageID) throws IOException {
		synchronized (this.messagesOffsets.getBuffer()) {
			@SuppressWarnings("null")
			@NonNull
			String string = this.messagesOffsets.seekToElementPointer(messageID).readString();
			return string;
		}
	}
}
