/*
 * Java
 *
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.service;

import ej.kf.Kernel;
import ej.service.ServiceRegistry;
import ej.service.loader.SystemPropertiesServiceLoader;
import ej.service.registry.SimpleServiceRegistry;

/**
 * The Class ServiceRegistryKF which contains the Local Service mapping for each
 * context.
 */
public class ServiceRegistryKF extends SimpleServiceRegistry implements ServiceRegistry {

	/** The shared service registry KF. */
	private static SharedServiceRegistry sharedServiceRegistryKF;

	/**
	 * Instantiates a new service registry KF.
	 */
	public ServiceRegistryKF() {
		if (sharedServiceRegistryKF == null) {
			sharedServiceRegistryKF = new SystemPropertiesServiceLoader().getService(SharedServiceRegistry.class);
		}
	}

	@Override
	public <T> T getService(Class<T> service) {
		T implementation = super.getService(service);
		if (implementation != null) {
			return implementation;
		}

		// if no instance found in local registry, check for Shared Service Registry
		return sharedServiceRegistryKF.getService(service);
	}

	@Override
	public <T> void register(Class<T> service, T instance) {
		if (Kernel.isInKernelMode()) {
			super.register(service, instance);
		} else {
			if (Kernel.isSharedInterface(service)) {
				sharedServiceRegistryKF.register(service, instance);
			} else {
				super.register(service, instance);
			}
		}
	}

	/**
	 * Registers a Service Instance for the Kernel.
	 *
	 * @param service       the class type of the service
	 * @param instance      the instance of the service
	 * @param isKernelLocal if the service should be stored in the local context of
	 *                      the Kernel or shared with other features
	 */
	public <T> void register(Class<T> service, T instance, boolean isKernelLocal) {
		if (Kernel.isInKernelMode()) {
			if (isKernelLocal) {
				super.register(service, instance);
			} else {
				sharedServiceRegistryKF.register(service, instance);
			}
		} else {
			// This method should not be called from a feature
			throw new IllegalStateException();
		}

	}

	@Override
	public <T> void unregister(Class<T> service, T instance) {
		if (!Kernel.isInKernelMode()) {
			if (Kernel.isSharedInterface(service)) {
				sharedServiceRegistryKF.unregister(service, instance);
			} else {
				super.unregister(service, instance);
			}
		} else {
			super.unregister(service, instance);
		}
	}

	/**
	 * Unregisters a Service Instance from the Kernel.
	 *
	 * @param service       the class type of the service
	 * @param instance      the instance of the service
	 * @param isKernelLocal if the service should be removed from the local context
	 *                      of the Kernel or shared with other features
	 */
	public <T> void unregister(Class<T> service, T instance, boolean isKernelLocal) {
		if (Kernel.isInKernelMode()) {
			if (isKernelLocal) {
				super.unregister(service, instance);
			} else {
				sharedServiceRegistryKF.unregister(service, instance);
			}
		} else {
			// This method should not be called from a feature
			throw new IllegalStateException();
		}

	}

}
