/*
 * Copyright 2018-2019 IS2T. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break IS2T warranties on the whole library.
 */
package ej.property.loader;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.property.PropertyLoader;
import ej.property.PropertyPermission;
import ej.service.loader.FilePropertiesServiceLoader;

/**
 * A property loader that searches in a properties file.
 */
public class FilePropertyLoader implements PropertyLoader {

	private final String file;

	/**
	 * Creates a file property loader.
	 *
	 * @param path
	 *            the properties file path.
	 */
	public FilePropertyLoader(String path) {
		this.file = path;
	}

	@Override
	@Nullable
	public String getProperty(String key) {
		SecurityManager securityManager = System.getSecurityManager();
		if (securityManager != null) {
			@SuppressWarnings("null") // Class name is not null for sure.
			@NonNull
			String className = getClass().getSimpleName();
			securityManager.checkPermission(new PropertyPermission(className, key, PropertyPermission.GET_ACTION));
		}

		Properties properties = new Properties();
		try (InputStream is = FilePropertiesServiceLoader.class.getResourceAsStream(this.file)) {
			properties.load(is);
		} catch (IOException | NullPointerException e) {
			// cannot read file
		}
		String property = properties.getProperty(key);
		if (property != null) {
			return property;
		}
		return null;
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		String property = getProperty(key);
		if (property != null) {
			return property;
		}
		return defaultValue;
	}

}
