/*
 * Java
 *
 * Copyright 2021-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorImage;

/**
 * Tests the drawing of animated images.
 */
@SuppressWarnings("nls")
public class TestDrawAnimatedImage {

	private static final String ANIMATED_PATHS_AVD = "/com/microej/microvg/test/animated_paths.xml";
	private static final String ANIMATED_COLORS_AVD = "/com/microej/microvg/test/animated_colors.xml";
	private static final String ANIMATED_TRANSLATEXY_AVD = "/com/microej/microvg/test/animated_translatexy.xml";
	private static final String ANIMATED_PATHINTERPOLATOR_AVD = "/com/microej/microvg/test/animated_pathinterpolator.xml";
	private static final String ANIMATED_PATHINTERPOLATOR2_AVD = "/com/microej/microvg/test/animated_pathinterpolator2.xml";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests drawing an animated image.
	 */
	@Test
	public static void testDrawAnimatedImageAtEnd() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_PATHS_AVD);

		// Test at start of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 0);

		display.flush();

		TestUtilities.check("start - blue", 125, 125, g, Colors.BLUE);
		TestUtilities.check("start - green", 175, 125, g, Colors.LIME);
		TestUtilities.check("start - red", 125, 175, g, Colors.RED);
		TestUtilities.check("start - yellow", 325, 125, g, Colors.YELLOW);
		TestUtilities.check("start - black", 175, 175, g, TestUtilities.BACKGROUND_COLOR);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, image.getDuration());

		display.flush();

		TestUtilities.check("start - blue", 175, 175, g, Colors.BLUE);
		TestUtilities.check("start - green", 325, 125, g, Colors.LIME);
		TestUtilities.check("start - red", 125, 325, g, Colors.RED);
		TestUtilities.check("start - yellow", 250, 300, g, Colors.YELLOW);
	}

	/**
	 * Tests drawing an animated image with x,y parameters.
	 */
	@Test
	public static void testDrawAnimatedImageXY() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage image = VectorImage.getImage(ANIMATED_PATHS_AVD);

		// Test at start of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, 100, 100, 0);

		display.flush();

		TestUtilities.check("start - blue", 125, 125, g, Colors.BLUE);
		TestUtilities.check("start - green", 175, 125, g, Colors.LIME);
		TestUtilities.check("start - red", 125, 175, g, Colors.RED);
		TestUtilities.check("start - yellow", 325, 125, g, Colors.YELLOW);
		TestUtilities.check("start - black", 175, 175, g, TestUtilities.BACKGROUND_COLOR);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, 100, 100, image.getDuration());

		display.flush();

		TestUtilities.check("start - blue", 175, 175, g, Colors.BLUE);
		TestUtilities.check("start - green", 325, 125, g, Colors.LIME);
		TestUtilities.check("start - red", 125, 325, g, Colors.RED);
		TestUtilities.check("start - yellow", 250, 300, g, Colors.YELLOW);
	}

	// /**
	// * Tests drawing an animated image with transformation.
	// */
	// @Test
	// public static void testDrawTransformedImage() {
	// Display display = Display.getDisplay();
	// GraphicsContext g = display.getGraphicsContext();
	//
	// Matrix matrix = new Matrix();
	// matrix.setTranslate(100, 100);
	// VectorImage image = VectorImage.getImage(ANIMATED_PATHS_AVD);
	// VectorImage transformed = image.transformWithFilter(0xffff00ff, BlendMode.SRC);
	//
	// // Test at start of animation
	// VectorGraphicsPainter.drawAnimatedImage(g, transformed, matrix, 0);
	//
	// display.flush();
	//
	// TestUtilities.check("start - blue", 125, 125, g, Colors.MAGENTA);
	// TestUtilities.check("start - green", 175, 125, g, Colors.MAGENTA);
	// TestUtilities.check("start - red", 125, 175, g, Colors.MAGENTA);
	// TestUtilities.check("start - yellow", 325, 125, g, Colors.MAGENTA);
	// TestUtilities.check("start - black", 175, 175, g, TestUtilities.BACKGROUND_COLOR);
	//
	// g.setColor(TestUtilities.BACKGROUND_COLOR);
	// Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	//
	// // Test at end of animation
	// VectorGraphicsPainter.drawAnimatedImage(g, transformed, matrix, transformed.getDuration());
	//
	// display.flush();
	//
	// TestUtilities.check("end - blue", 175, 175, g, Colors.MAGENTA);
	// TestUtilities.check("end - green", 325, 125, g, Colors.MAGENTA);
	// TestUtilities.check("end - red", 125, 325, g, Colors.MAGENTA);
	// TestUtilities.check("end - yellow", 250, 300, g, Colors.MAGENTA);
	// }

	/**
	 * Tests drawing a color animated image.
	 */
	@Test
	public static void testDrawColorImage() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);

		// Test at start of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 0);

		display.flush();

		TestUtilities.check("start - fillColor", 100, 100, g, Colors.RED);
		TestUtilities.check("start - strokeColor", 200, 100, g, Colors.RED);
		TestUtilities.check("start - fillAlpha", 100, 200, g, Colors.LIME);
		TestUtilities.check("start - strokeAlpha", 200, 200, g, Colors.LIME);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at middle of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, image.getDuration() / 2);

		display.flush();

		TestUtilities.check("middle - fillColor", 100, 100, g, 0xba00ba);
		TestUtilities.check("middle - strokeColor", 200, 100, g, 0xba00ba);
		TestUtilities.check("middle - fillAlpha", 100, 200, g, 0x00C000);
		TestUtilities.check("middle - strokeAlpha", 200, 200, g, 0x00C000);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// Test at end of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, image.getDuration());

		display.flush();

		TestUtilities.check("end - fillColor", 100, 100, g, Colors.BLUE);
		TestUtilities.check("end - strokeColor", 200, 100, g, Colors.BLUE);
		TestUtilities.check("end - fillAlpha", 100, 200, g, 0x008000);
		TestUtilities.check("end - strokeAlpha", 200, 200, g, 0x008000);
	}

	/**
	 * Tests drawing an animated image with transparency.
	 */
	@Test
	public static void testDrawAnimatedImageTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();

		Path path = new Path();
		path.moveTo(25, 25);
		path.lineTo(125, 25);
		path.lineTo(125, 125);
		path.lineTo(25, 125);
		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillPath(g, path, matrix);

		VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 0, 0x88);

		display.flush();

		TestUtilities.check("start - fillColor", 0, 0, g, 0x880000);
		TestUtilities.check("start - strokeColor", 149, 0, g, 0x880000);
		TestUtilities.check("start - fillAlpha", 0, 148, g, 0x008800);
		TestUtilities.check("start - strokeAlpha", 149, 149, g, 0x008800); // $NON-NLS-1
		TestUtilities.check("start - fillColor overlap", 45, 45, g, 0x880077);
		TestUtilities.check("start - strokeColor overlap", 100, 50, g, 0x880077);
		TestUtilities.check("start - fillAlpha overlap", 45, 105, g, 0x008877);
		TestUtilities.check("start - strokeAlpha overlap", 100, 100, g, 0x008877); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillPath(g, path, matrix);
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, image.getDuration() / 2, 0x88);

		display.flush();

		TestUtilities.check("middle - fillColor", 0, 0, g, 0x630063);
		TestUtilities.check("middle - strokeColor", 149, 0, g, 0x630063);
		TestUtilities.check("middle - fillAlpha", 0, 148, g, 0x006600);
		TestUtilities.check("middle - strokeAlpha", 149, 149, g, 0x006600); // $NON-NLS-1
		TestUtilities.check("middle - fillColor overlap", 45, 45, g, 0x6300da);
		TestUtilities.check("middle - strokeColor overlap", 100, 50, g, 0x6300da);
		TestUtilities.check("middle - fillAlpha overlap", 45, 105, g, 0x006699);
		TestUtilities.check("middle - strokeAlpha overlap", 100, 100, g, 0x006699); // $NON-NLS-1

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		g.setColor(Colors.BLUE);
		VectorGraphicsPainter.fillPath(g, path, matrix);
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, image.getDuration(), 0x88);

		display.flush();

		TestUtilities.check("end - fillColor", 0, 0, g, 0x000088);
		TestUtilities.check("end - strokeColor", 149, 0, g, 0x000088);
		TestUtilities.check("end - fillAlpha", 0, 148, g, 0x004400);
		TestUtilities.check("end - strokeAlpha", 149, 149, g, 0x004400); // $NON-NLS-1
		TestUtilities.check("end - fillColor overlap", 45, 45, g, Colors.BLUE);
		TestUtilities.check("end - strokeColor overlap", 100, 50, g, Colors.BLUE);
		TestUtilities.check("end - fillAlpha overlap", 45, 105, g, 0x0044bb);
		TestUtilities.check("end - strokeAlpha overlap", 100, 100, g, 0x0044bb); // $NON-NLS-1
	}

	/**
	 * Tests drawing an animated image with overlapping paths with transparency.
	 */
	@Test
	public static void testDrawAnimatedImageTransparentOverlap() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_PATHS_AVD);

		boolean exception;
		try {
			VectorGraphicsPainter.drawAnimatedImage(g, image, new Matrix(), 0, 0x88);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);

		try {
			VectorGraphicsPainter.drawAnimatedImage(g, image, new Matrix(), image.getDuration(), 0x88);
			exception = false;
		} catch (VectorGraphicsException e) {
			exception = true;
		}
		Assert.assertTrue(exception);
	}

	// /**
	// * Tests drawing a color transformed animated image.
	// */
	// @Test
	// public static void testDrawColorTransformedImage() {
	// Display display = Display.getDisplay();
	// GraphicsContext g = display.getGraphicsContext();
	//
	// Matrix matrix = new Matrix();
	// matrix.setTranslate(100, 100);
	// VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);
	// VectorImage transformed = image.transformWithFilter(0x40ffff00, BlendMode.SRC_OVER);
	//
	// // Test at start of animation
	// VectorGraphicsPainter.drawAnimatedImage(g, transformed, matrix, 0);
	//
	// display.flush();
	//
	// TestUtilities.check("start - fillColor", 100, 100, g, Colors.YELLOW);
	// TestUtilities.check("start - strokeColor", 200, 100, g, Colors.YELLOW);
	// TestUtilities.check("start - fillAlpha", 100, 200, g, Colors.YELLOW);
	// TestUtilities.check("start - strokeAlpha", 200, 200, g, Colors.YELLOW);
	//
	// g.setColor(TestUtilities.BACKGROUND_COLOR);
	// Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	//
	// // Test at end of animation
	// VectorGraphicsPainter.drawAnimatedImage(g, transformed, matrix, transformed.getDuration());
	//
	// display.flush();
	//
	// TestUtilities.check("end - fillColor", 100, 100, g, 0xffffbf);
	// TestUtilities.check("end - strokeColor", 200, 100, g, 0xffffbf);
	// TestUtilities.check("end - fillAlpha", 100, 200, g, 0x9e9e00);
	// TestUtilities.check("end - strokeAlpha", 200, 200, g, 0x9e9e00);
	// }

	// /**
	// * Tests drawing a color animated image with transformationn.
	// */
	// @Test
	// public static void testDrawColorImageWithTransform() {
	// Display display = Display.getDisplay();
	// GraphicsContext g = display.getGraphicsContext();
	//
	// Matrix matrix = new Matrix();
	// matrix.setTranslate(100, 100);
	// VectorImage image = VectorImage.getImage(ANIMATED_COLORS_AVD);
	//
	// // Test at start of animation
	// VectorGraphicsPainter.drawAnimatedImageWithFilter(g, image, matrix, 0, 0x4000ffff, BlendMode.SRC_OVER);
	//
	// display.flush();
	//
	// TestUtilities.check("start - fillColor", 100, 100, g, 0xbfffff);
	// TestUtilities.check("start - strokeColor", 200, 100, g, 0xbfffff);
	// TestUtilities.check("start - fillAlpha", 100, 200, g, 0x00ffff);
	// TestUtilities.check("start - strokeAlpha", 200, 200, g, 0x00ffff);
	//
	// g.setColor(TestUtilities.BACKGROUND_COLOR);
	// Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	//
	// // Test at end of animation
	// VectorGraphicsPainter.drawAnimatedImageWithFilter(g, image, matrix, image.getDuration(), 0x40ffff00,
	// BlendMode.SRC_OVER);
	//
	// display.flush();
	//
	// TestUtilities.check("end - fillColor", 100, 100, g, 0xffffbf);
	// TestUtilities.check("end - strokeColor", 200, 100, g, 0xffffbf);
	// TestUtilities.check("end - fillAlpha", 100, 200, g, 0x9e9e00);
	// TestUtilities.check("end - strokeAlpha", 200, 200, g, 0x9e9e00);
	// }

	/**
	 * Tests drawing a animated image with translateXY animations.
	 */
	@Test
	public static void testDrawTranslateXYImage() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_TRANSLATEXY_AVD);

		// Top left
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 0);

		display.flush();

		TestUtilities.check("0 - top left", 101, 101, g, Colors.YELLOW);
		TestUtilities.check("0 - center", 121, 121, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("0 - top right", 146, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("0 - bottom right", 146, 146, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("0 - bottom left", 101, 146, g, TestUtilities.BACKGROUND_COLOR);

		// Center

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 2500);

		display.flush();

		TestUtilities.check("1 - top left", 101, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("1 - center", 121, 121, g, Colors.YELLOW);
		TestUtilities.check("1 - top right", 146, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("1 - bottom right", 146, 146, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("1 - bottom left", 101, 146, g, TestUtilities.BACKGROUND_COLOR);

		// Top Right

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 5500);

		display.flush();

		TestUtilities.check("2 - top left", 101, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("2 - center", 121, 121, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("2 - top right", 146, 101, g, Colors.YELLOW);
		TestUtilities.check("2 - bottom right", 146, 146, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("2 - bottom left", 101, 146, g, TestUtilities.BACKGROUND_COLOR);

		// Bottom Right

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 8500);

		display.flush();

		TestUtilities.check("3 - top left", 101, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("3 - center", 121, 121, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("3 - top right", 146, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("3 - bottom right", 146, 146, g, Colors.YELLOW);
		TestUtilities.check("3 - bottom left", 101, 146, g, TestUtilities.BACKGROUND_COLOR);

		// Bottom Left

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 11500);

		display.flush();

		TestUtilities.check("4 - top left", 101, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("4 - center", 121, 121, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("4 - top right", 146, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("4 - bottom right", 146, 146, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("4 - bottom left", 101, 146, g, Colors.YELLOW);

		// top Left

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 18000);

		display.flush();

		TestUtilities.check("5 - top left", 101, 101, g, Colors.YELLOW);
		TestUtilities.check("5 - center", 121, 121, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("5 - top right", 146, 101, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("5 - bottom right", 146, 146, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.check("5 - bottom left", 101, 146, g, TestUtilities.BACKGROUND_COLOR);
	}

	/**
	 * Tests drawing a animated image with path interpolator animations.
	 */
	@Test
	public static void testDrawPathInterpolatorImage() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_PATHINTERPOLATOR_AVD);

		// 0 millisec - tx = 0%
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 0);

		display.flush();

		TestUtilities.check("0", 101, 125, g, Colors.YELLOW);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// 1000 millisec - tx = 74.3%
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 1000);

		display.flush();

		TestUtilities.check("0", 134, 125, g, Colors.YELLOW);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// 2000 millisec - tx = 0%
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 2000);

		display.flush();

		TestUtilities.check("0", 101, 125, g, Colors.YELLOW);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// 3000 millisec - tx = 75%
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 3000);

		display.flush();

		TestUtilities.check("0", 137, 125, g, Colors.YELLOW);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		// 4000 millisec - tx = 100%
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, 4000);

		display.flush();

		TestUtilities.check("0", 146, 125, g, Colors.YELLOW);

		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	}

	/**
	 * Tests drawing an animated image with an elapsed time negative.
	 */
	@Test
	public static void testDrawAnimatedImageNegativeTime() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_PATHS_AVD);
		long elapsedTime = (long) (-image.getDuration() * 0.5f);

		// Test at start of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, elapsedTime);

		display.flush();

		TestUtilities.check("start (1) - blue", 125, 125, g, Colors.BLUE);
		TestUtilities.check("start (1) - green", 175, 125, g, Colors.LIME);
		TestUtilities.check("start (1) - red", 125, 175, g, Colors.RED);
		TestUtilities.check("start (1) - yellow", 325, 125, g, Colors.YELLOW);
		TestUtilities.check("start (1) - black", 175, 175, g, TestUtilities.BACKGROUND_COLOR);

		TestUtilities.clearScreen();

		VectorGraphicsPainter.drawAnimatedImage(g, image, 100, 100, elapsedTime);
		display.flush();

		TestUtilities.check("start (2) - blue", 125, 125, g, Colors.BLUE);
		TestUtilities.check("start (2) - green", 175, 125, g, Colors.LIME);
		TestUtilities.check("start (2) - red", 125, 175, g, Colors.RED);
		TestUtilities.check("start (2) - yellow", 325, 125, g, Colors.YELLOW);
		TestUtilities.check("start (2) - black", 175, 175, g, TestUtilities.BACKGROUND_COLOR);

		TestUtilities.clearScreen();

		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, elapsedTime, GraphicsContext.OPAQUE);
		display.flush();

		TestUtilities.check("start (3) - blue", 125, 125, g, Colors.BLUE);
		TestUtilities.check("start (3) - green", 175, 125, g, Colors.LIME);
		TestUtilities.check("start (3) - red", 125, 175, g, Colors.RED);
		TestUtilities.check("start (3) - yellow", 325, 125, g, Colors.YELLOW);
		TestUtilities.check("start (3) - black", 175, 175, g, TestUtilities.BACKGROUND_COLOR);
	}

	/**
	 * Tests drawing an animated image with an elapsed time greater than duration.
	 */
	@Test
	public static void testDrawAnimatedImageAfterTime() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_PATHS_AVD);
		long elapsedTime = (long) (image.getDuration() * 1.5f);

		// Test at end of animation
		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, elapsedTime);
		display.flush();

		TestUtilities.check("start - blue", 175, 175, g, Colors.BLUE);
		TestUtilities.check("start - green", 325, 125, g, Colors.LIME);
		TestUtilities.check("start - red", 125, 325, g, Colors.RED);
		TestUtilities.check("start - yellow", 250, 300, g, Colors.YELLOW);

		TestUtilities.clearScreen();

		VectorGraphicsPainter.drawAnimatedImage(g, image, 100, 100, elapsedTime);
		display.flush();

		TestUtilities.check("start - blue", 175, 175, g, Colors.BLUE);
		TestUtilities.check("start - green", 325, 125, g, Colors.LIME);
		TestUtilities.check("start - red", 125, 325, g, Colors.RED);
		TestUtilities.check("start - yellow", 250, 300, g, Colors.YELLOW);

		TestUtilities.clearScreen();

		VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, elapsedTime, GraphicsContext.OPAQUE);
		display.flush();

		TestUtilities.check("start - blue", 175, 175, g, Colors.BLUE);
		TestUtilities.check("start - green", 325, 125, g, Colors.LIME);
		TestUtilities.check("start - red", 125, 325, g, Colors.RED);
		TestUtilities.check("start - yellow", 250, 300, g, Colors.YELLOW);
	}

	/**
	 * Tests drawing an animated image with different path interpolators: the paths have got a different lenght than the
	 * original shape (smaller or bigger) and the bounding box differs according the path interpolator.
	 */
	@Test
	public static void testDrawAnimatedImageDifferentPathSizes() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Matrix matrix = new Matrix();
		matrix.setScale(5, 5);
		matrix.postTranslate(100, 100);
		VectorImage image = VectorImage.getImage(ANIMATED_PATHINTERPOLATOR2_AVD);

		int drawn = Colors.WHITE;
		int bg = g.getColor();
		int nbTests = 0;

		// Test at end of animation
		for (int i = 0; i <= image.getDuration(); i++) {
			VectorGraphicsPainter.drawAnimatedImage(g, image, matrix, i);
			display.flush();
			display.waitFlushCompleted();

			switch (i) {
			case 0:
			case 30:
			case 60:
				// big square
				TestUtilities.check("square TL", 105, 105, g, drawn);
				TestUtilities.check("square T", 150, 105, g, drawn);
				TestUtilities.check("square TR", 195, 105, g, drawn);
				TestUtilities.check("square L", 105, 150, g, drawn);
				TestUtilities.check("square R", 195, 150, g, drawn);
				TestUtilities.check("square BL", 105, 195, g, drawn);
				TestUtilities.check("square B", 150, 195, g, drawn);
				TestUtilities.check("square BR", 195, 195, g, drawn);
				TestUtilities.check("square C", 150, 150, g, drawn);
				++nbTests;
				break;
			case 10:
				// rectangle
				TestUtilities.check("rectangle TL", 105, 105, g, bg);
				TestUtilities.check("rectangle T", 150, 105, g, drawn);
				TestUtilities.check("rectangle TR", 195, 105, g, bg);
				TestUtilities.check("rectangle L", 105, 150, g, bg);
				TestUtilities.check("rectangle R", 195, 150, g, bg);
				TestUtilities.check("rectangle BL", 105, 195, g, bg);
				TestUtilities.check("rectangle B", 150, 195, g, drawn);
				TestUtilities.check("rectangle BR", 195, 195, g, bg);
				TestUtilities.check("rectangle C", 150, 150, g, drawn);
				++nbTests;
				break;
			case 20:
				// small square
				TestUtilities.check("ssquare TL", 105, 105, g, bg);
				TestUtilities.check("ssquare T", 150, 105, g, bg);
				TestUtilities.check("ssquare TR", 195, 105, g, bg);
				TestUtilities.check("ssquare L", 105, 150, g, bg);
				TestUtilities.check("ssquare R", 195, 150, g, bg);
				TestUtilities.check("ssquare BL", 105, 195, g, bg);
				TestUtilities.check("ssquare B", 150, 195, g, bg);
				TestUtilities.check("ssquare BR", 195, 195, g, bg);
				TestUtilities.check("ssquare C", 150, 150, g, drawn);
				++nbTests;
				break;
			case 40:
				// triangleA
				TestUtilities.check("triangleA TL", 105, 105, g, bg);
				TestUtilities.check("triangleA T", 150, 105, g, drawn);
				TestUtilities.check("triangleA TR", 195, 105, g, bg);
				TestUtilities.check("triangleA L", 105, 150, g, bg);
				TestUtilities.check("triangleA R", 195, 150, g, bg);
				TestUtilities.check("triangleA BL", 105, 195, g, drawn);
				TestUtilities.check("triangleA B", 150, 195, g, drawn);
				TestUtilities.check("triangleA BR", 195, 195, g, drawn);
				TestUtilities.check("triangleA C", 150, 150, g, drawn);
				++nbTests;
				break;
			case 50:
				// triangleB
				TestUtilities.check("triangleB TL", 105, 105, g, drawn);
				TestUtilities.check("triangleB T", 150, 105, g, drawn);
				TestUtilities.check("triangleB TR", 195, 105, g, drawn);
				TestUtilities.check("triangleB L", 105, 150, g, bg);
				TestUtilities.check("triangleB R", 195, 150, g, bg);
				TestUtilities.check("triangleB BL", 105, 195, g, bg);
				TestUtilities.check("triangleB B", 150, 195, g, drawn);
				TestUtilities.check("triangleB BR", 195, 195, g, bg);
				TestUtilities.check("triangleB C", 150, 150, g, drawn);
				++nbTests;
				break;

			default:
				break;
			}

			try {
				Thread.sleep(50);
			} catch (InterruptedException e) {
				// drop
			}
			TestUtilities.clearScreen();
		}

		assertEquals("check number of tests", 7, nbTests);

	}
}
