/*
 * Java
 *
 * Copyright 2023-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BufferedVectorImage;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.ResourceVectorImage;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorImage;

/**
 * Tests that BufferedVectorImage can be used in place of VectorImage.
 *
 * Some tests of this class do not test that the calls actually work, but only that the functions do not fail abruptly.
 */
@SuppressWarnings("nls")
public class TestBufferedVectorImageCompatibility {

	private static final String RED_RECTANGLE_SVG = "/com/microej/microvg/test/red_rectangle.xml";
	private static final String GRADIENT_RECTANGLE_AVD = "/com/microej/microvg/test/gradient.xml";

	/**
	 * Initializes MicroUI.
	 */
	@BeforeClass
	public static void setUpBeforeClass() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void tearDownAfterClass() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawAnimatedImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawAnimatedImage1() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawAnimatedImage(gc, bvi, 10.0f, 20.0f, 2);
		}
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawAnimatedImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawAnimatedImage2() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		final Matrix matrix = new Matrix();

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawAnimatedImage(gc, bvi, matrix, 1, 128);
		}
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawAnimatedImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawAnimatedImage3() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		final Matrix matrix = new Matrix();

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawAnimatedImage(gc, bvi, matrix, 3);
		}
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawFilteredAnimatedImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawFilteredAnimatedImage() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		final Matrix matrix = new Matrix();
		final float[] colorMatrix = new float[] { //
				1.0f, 0.0f, 0.0f, 0.0f, 0.0f, // Red
				0.0f, 1.0f, 0.0f, 0.0f, 0.0f, // Green
				0.0f, 0.0f, 1.0f, 0.0f, 0.0f, // Blue
				0.0f, 0.0f, 0.0f, 1.0f, 0.0f // Alpha
		};

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawFilteredAnimatedImage(gc, bvi, matrix, 0, colorMatrix);
		}
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawFilteredImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawFilteredImage() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		final Matrix matrix = new Matrix();
		final float[] colorMatrix = new float[] { //
				1.0f, 0.0f, 0.0f, 0.0f, 0.0f, // Red
				0.0f, 1.0f, 0.0f, 0.0f, 0.0f, // Green
				0.0f, 0.0f, 1.0f, 0.0f, 0.0f, // Blue
				0.0f, 0.0f, 0.0f, 1.0f, 0.0f // Alpha
		};

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawFilteredImage(gc, bvi, matrix, colorMatrix);
		}
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawImage1() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawImage(gc, bvi, 39, 14);
		}
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawImage2() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		final Matrix matrix = new Matrix();

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawImage(gc, bvi, matrix, 64);
		}
	}

	/**
	 * Tests that calling VectorGraphicsPainter.drawImage on a BufferedVectorImage does not fail.
	 */
	@Test
	public void testDrawImage3() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		final Matrix matrix = new Matrix();

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			VectorGraphicsPainter.drawImage(gc, bvi, matrix);
		}
	}

	/**
	 * Tests that calling BufferedVectorImage.clear does not fail.
	 */
	@Test
	public void testClear() {
		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			bvi.clear();
		}
	}

	/**
	 * Tests that calling BufferedVectorImage.close does not fail.
	 */
	@Test
	public void testClose() {
		@SuppressWarnings("resource")
		BufferedVectorImage bvi = new BufferedVectorImage(200, 150);
		bvi.close();
	}

	/**
	 * Tests that calling BufferedVectorImage.filterImage does not fail.
	 */
	@Test
	public void testFilterImage() {
		final float[] colorMatrix = new float[] { //
				1.0f, 0.0f, 0.0f, 0.0f, 0.0f, // Red
				0.0f, 1.0f, 0.0f, 0.0f, 0.0f, // Green
				0.0f, 0.0f, 1.0f, 0.0f, 0.0f, // Blue
				0.0f, 0.0f, 0.0f, 1.0f, 0.0f // Alpha
		};

		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			@SuppressWarnings("resource")
			ResourceVectorImage rvi = bvi.filterImage(colorMatrix);
			rvi.close();
		}
	}

	/**
	 * Tests that calling BufferedVectorImage.getDuration does not fail.
	 */
	@Test
	public void testGetDuration() {
		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			bvi.getDuration();
		}
	}

	/**
	 * Tests that calling BufferedVectorImage.getGraphicsContext does not fail.
	 */
	@Test
	public void testGetGraphicsContext() {
		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			bvi.getGraphicsContext();
		}
	}

	/**
	 * Tests that calling BufferedVectorImage.getWidth does not fail.
	 */
	@Test
	public void testGetWidth() {
		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			bvi.getWidth();
		}
	}

	/**
	 * Tests that calling BufferedVectorImage.getHeight does not fail.
	 */
	@Test
	public void testGetHeight() {
		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			bvi.getHeight();
		}
	}

	/**
	 * Tests that calling BufferedVectorImage.isClosed does not fail.
	 */
	@Test
	public void testIsClosed() {
		try (final BufferedVectorImage bvi = new BufferedVectorImage(200, 150)) {
			bvi.isClosed();
		}
	}

	/**
	 * Tests drawing a buffered vector image with a color matrix filter changing the color by setting a new one.
	 */
	@Test
	public static void testColorFilterNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(150, 150)) {
			GraphicsContext igc = image.getGraphicsContext();
			igc.setColor(Colors.RED);
			igc.translate(50, 50);
			VectorGraphicsPainter.fillPath(igc, getPath(100, 100), new Matrix());

			Matrix matrix = new Matrix();
			float[] colorMatrix = new float[] { //
					0, 0, 0, 0, 0, // red
					0, 0, 0, 0, 0x88, // green
					0, 0, 0, 0, 0x88, // blue
					0, 0, 0, 1, 0, // alpha
			};
			VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
			display.flush();

			TestUtilities.check("top left", 50, 50, g, 0x008888);
			TestUtilities.check("center", 99, 99, g, 0x008888);
			TestUtilities.check("bottom right", 149, 149, g, 0x008888);
		}
	}

	/**
	 * Tests drawing a buffered vector gradient image with a color matrix filter changing the color by setting a new
	 * one.
	 */
	@Test
	public static void testGradientColorFilterNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(100, 100)) {
			GraphicsContext igc = image.getGraphicsContext();
			LinearGradient gradient = new LinearGradient(50, 0, 50, 99,
					new int[] { 0xffff0000, 0xff00ff00, 0xff0000ff });
			VectorGraphicsPainter.fillGradientPath(igc, getPath(100, 100), new Matrix(), gradient);

			Matrix matrix = new Matrix();
			float[] colorMatrix = new float[] { //
					0, 0, 0, 0, 0, // red
					0, 0, 0, 0, 0x88, // green
					0, 0, 0, 0, 0x88, // blue
					0, 0, 0, 1, 0, // alpha
			};
			VectorGraphicsPainter.drawFilteredImage(g, image, matrix, colorMatrix);
			display.flush();

			TestUtilities.check("top left", 0, 0, g, 0x008888);
			TestUtilities.check("center", 49, 49, g, 0x008888);
			TestUtilities.check("bottom right", 99, 99, g, 0x008888);
		}
	}

	/**
	 * Tests drawing a buffered vector image after transforming it with a color matrix filter changing the color by
	 * setting a new one.
	 */
	@Test
	public static void testColorFilterImageNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(150, 150)) {
			GraphicsContext igc = image.getGraphicsContext();
			igc.setColor(Colors.RED);
			igc.translate(50, 50);
			VectorGraphicsPainter.fillPath(igc, getPath(100, 100), new Matrix());

			float[] colorMatrix = new float[] { //
					0, 0, 0, 0, 0, // red
					0, 0, 0, 0, 0x88, // green
					0, 0, 0, 0, 0x88, // blue
					0, 0, 0, 1, 0, // alpha
			};
			try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
				VectorGraphicsPainter.drawImage(g, filterImage, new Matrix());
				display.flush();

				TestUtilities.check("top left", 50, 50, g, 0x008888);
				TestUtilities.check("center", 99, 99, g, 0x008888);
				TestUtilities.check("bottom right", 149, 149, g, 0x008888);
			}
		}
	}

	/**
	 * Tests drawing a buffered vector gradient image after transforming it with a color matrix filter changing the
	 * color by setting a new one.
	 */
	@Test
	public static void testGradientColorFilterImageNewColor() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(100, 100)) {
			GraphicsContext igc = image.getGraphicsContext();
			LinearGradient gradient = new LinearGradient(50, 0, 50, 99,
					new int[] { 0xffff0000, 0xff00ff00, 0xff0000ff });
			VectorGraphicsPainter.fillGradientPath(igc, getPath(100, 100), new Matrix(), gradient);

			Matrix matrix = new Matrix();
			float[] colorMatrix = new float[] { //
					0, 0, 0, 0, 0, // red
					0, 0, 0, 0, 0x88, // green
					0, 0, 0, 0, 0x88, // blue
					0, 0, 0, 1, 0, // alpha
			};
			try (ResourceVectorImage filterImage = image.filterImage(colorMatrix)) {
				VectorGraphicsPainter.drawImage(g, filterImage, matrix);
				display.flush();

				TestUtilities.check("top left", 0, 0, g, 0x008888);
				TestUtilities.check("center", 49, 49, g, 0x008888);
				TestUtilities.check("bottom right", 99, 99, g, 0x008888);
			}
		}
	}

	/**
	 * Tests drawing a buffered vector image that points on a RAW image and that holds paths after transforming it with
	 * a color matrix filter changing the color by setting a new one.
	 */
	@Test
	public static void testColorFilterImageWithRawImage() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage imageColor = VectorImage.getImage(RED_RECTANGLE_SVG);
		VectorImage imageGradient = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		try (BufferedVectorImage image = new BufferedVectorImage(300, 300)) {
			GraphicsContext igc = image.getGraphicsContext();

			// TOP LEFT: color
			igc.setColor(Colors.RED);
			igc.translate(50, 50);
			VectorGraphicsPainter.fillPath(igc, getPath(100, 100), new Matrix());
			igc.translate(-50, -50);

			// TOP RIGHT: raw image with color
			igc.translate(150, 0);
			VectorGraphicsPainter.drawImage(igc, imageColor, new Matrix());
			igc.translate(-150, 0);

			// BOTTOM LEFT: gradient
			LinearGradient gradient = new LinearGradient(50, 0, 50, 99,
					new int[] { 0xffff0000, 0xff00ff00, 0xff0000ff });
			igc.translate(0, 150);
			VectorGraphicsPainter.fillGradientPath(igc, getPath(100, 100), new Matrix(), gradient);
			igc.translate(0, -150);

			// BOTTOM RIGHT: raw image with gradient
			igc.translate(150, 150);
			VectorGraphicsPainter.drawImage(igc, imageGradient, new Matrix());
			igc.translate(-150, -150);

			float[] colorMatrix1 = new float[] { //
					0.5f, 0, 0, 0, 0, // red
					0, 0, 0, 0, 0x88, // green
					1, 0, 0, 0, 0, // blue
					0, 0, 0, 1, 0, // alpha
			};

			try (ResourceVectorImage filterImage = image.filterImage(colorMatrix1)) {

				float[] colorMatrix2 = new float[] { //
						0.5f, 0, 0, 0, 0, // red
						0, 0.5f, 0, 0, 0, // green
						0, 0, 0, 0, 0x88, // blue
						0, 0, 0, 1, 0, // alpha
				};

				VectorGraphicsPainter.drawFilteredImage(g, filterImage, new Matrix(), colorMatrix2);
				display.flush();
			}

			// color
			TestUtilities.check("[path color] top left", 50, 50, g, 0x3f4488);
			TestUtilities.check("[path color] center", 99, 99, g, 0x3f4488);
			TestUtilities.check("[path color] bottom right", 149, 149, g, 0x3f4488);

			// image color
			TestUtilities.check("[image color] top left", 150 + 50, 50, g, 0x3f4488);
			TestUtilities.check("[image color] center", 150 + 99, 99, g, 0x3f4488);
			TestUtilities.check("[image color] bottom right", 150 + 149, 149, g, 0x3f4488);

			// gradient
			TestUtilities.check("[path gradient] top left", 0, 150 + 0, g, 0x3f4488);
			TestUtilities.check("[path gradient] center", 49, 150 + 49, g, 0x004488);
			TestUtilities.check("[path gradient] bottom right", 99, 150 + 99, g, 0x004488);

			// image gradient
			TestUtilities.check("[image gradient] top left", 150 + 0, 150 + 0, g, 0x3f4488);
			TestUtilities.check("[image gradient] center", 150 + 49, 150 + 49, g, 0x004488);
			TestUtilities.check("[image gradient] bottom right", 150 + 99, 150 + 99, g, 0x004488);
		}
	}

	/**
	 * Tests drawing a buffered vector image that points on a RAW image transformed with a color matrix filter changing
	 * the color by setting a new one.
	 */
	@Test
	public static void testColorFilterImageWithFilteredRawImage() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorImage imageColor = VectorImage.getImage(RED_RECTANGLE_SVG);
		VectorImage imageGradient = VectorImage.getImage(GRADIENT_RECTANGLE_AVD);

		try (BufferedVectorImage image = new BufferedVectorImage(300, 300)) {
			GraphicsContext igc = image.getGraphicsContext();

			float[] colorMatrix1 = new float[] { //
					0.5f, 0, 0, 0, 0, // red
					0, 0, 0, 0, 0x88, // green
					1, 0, 0, 0, 0, // blue
					0, 0, 0, 1, 0, // alpha
			};

			// TOP: raw image with color
			igc.translate(50, 0);
			VectorGraphicsPainter.drawFilteredImage(igc, imageColor, new Matrix(), colorMatrix1);

			// BOTTOM: raw image with gradient
			igc.translate(50, 170);
			VectorGraphicsPainter.drawFilteredImage(igc, imageGradient, new Matrix(), colorMatrix1);

			VectorGraphicsPainter.drawImage(g, image, new Matrix());

			display.flush();

			// image color
			TestUtilities.check("[image color] top left", 50 + 50, 0 + 50, g, 0x7888f8);
			TestUtilities.check("[image color] center", 50 + 99, 0 + 99, g, 0x7888f8);
			TestUtilities.check("[image color] bottom right", 50 + 149, 0 + 149, g, 0x7888f8);

			// image gradient
			TestUtilities.check("[image gradient] top left", 50 + 50, 170 + 0, g, 0x7888f8);
			TestUtilities.check("[image gradient] center", 50 + 99, 170 + 49, g, 0x008800);
			TestUtilities.check("[image gradient] bottom right", 50 + 149, 170 + 99, g, 0x008800);
		}
	}

	private static Path getPath(int w, int h) {
		Path path = new Path();
		path.lineTo(w, 0);
		path.lineTo(w, h);
		path.lineTo(0, h);
		path.close();
		return path;
	}
}
