/*
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.wear.services;

import ej.microui.display.Format;

import java.io.Closeable;
import java.io.IOException;

/**
 * This service allows Features to create and delete external resources.
 */
public interface ExternalResourceService {

	/**
	 * Enumerates supported image formats.
	 */
	enum ImageFormat {

		/** RGB565 format. */
		RGB565,

		/** ARGB4444 format. */
		ARGB4444,

		/** A8 format. */
		A8;
	}

	/**
	 * Represents a writer to an external resource.
	 *
	 * <p>
	 * A conversion step can be applied when writing or saving the external resource. For example, images can be
	 * converted into a specific format before they are stored, depending on how the user will use that resource in the
	 * application.
	 */
	interface Writer extends Closeable {

		/**
		 * Writes the given data to the external resource.
		 *
		 * @param src
		 *            the source array containing the data to write to the external resource.
		 * @param srcOffset
		 *            the offset in the source array of the first byte to write.
		 * @param length
		 *            the number of bytes to write.
		 * @param destOffset
		 *            the offset in the destination (i.e., external resource) at which to write the data.
		 * @throws IndexOutOfBoundsException
		 *             if the given srcOffset or the given length is out of the bounds of the source array.
		 * @throws IndexOutOfBoundsException
		 *             if the given destOffset or the given length is out of the bounds of the destination.
		 * @throws IllegalStateException
		 *             if this writer is closed.
		 * @throws IOException
		 *             if an error occurs while writing to the resource.
		 * @throws IOException
		 *             if an error occurs during the conversion phase.
		 */
		void write(byte[] src, int srcOffset, int length, int destOffset) throws IOException;

		/**
		 * Flushes all written data and finalizes the resource.
		 * <p>
		 * After calling this method, the resource is visible and accessible to the application. Any remaining buffered
		 * data is written, and the writer is closed, preventing further modifications.
		 *
		 * @throws IllegalStateException
		 *             if this writer is closed.
		 * @throws IOException
		 *             if an error occurs while committing the resource.
		 * @throws IOException
		 *             if an error occurred during the conversion phase.
		 */
		void commit() throws IOException;

		/**
		 * Closes this writer and clears all uncommitted native resources.
		 *
		 * <p>
		 * This method does nothing if this writer is already closed.
		 */
		@Override
		void close();
	}

	/**
	 * Creates an external resource at the given path.
	 *
	 * @param path
	 *            the path of the external resource (relative to the Module directory).
	 * @param size
	 *            the size of the data to be written to the external resource.
	 * @return a writer to the external resource.
	 * @throws IllegalArgumentException
	 *             if the path does not start with a "/".
	 * @throws IllegalArgumentException
	 *             if the size is negative.
	 * @throws IOException
	 *             if an external resource with the same path already exists.
	 * @throws IOException
	 *             if a writer to the same path is already opened.
	 * @throws IOException
	 *             if there is not enough memory in the external resource location to allocate the resource.
	 */
	Writer createExternalResource(String path, int size) throws IOException;

	/**
	 * Creates an external image at the given path, converting it to the given format.
	 *
	 * @param path
	 *            the path of the external resource (relative to the Module directory).
	 * @param size
	 *            the size of the data to be written to the external resource.
	 * @param imageFormat
	 *            the format to convert the image to.
	 * @return a writer to the external resource.
	 * @throws IllegalArgumentException
	 *             if the path does not start with a "/".
	 * @throws IllegalArgumentException
	 *             if the size is negative.
	 * @throws IOException
	 *             if an external resource with the same path already exists.
	 * @throws IOException
	 *             if a writer to the same path is already opened.
	 * @throws IOException
	 *             if there is not enough memory in the external resource location to allocate the resource.
	 */
	Writer createExternalImage(String path, int size, ImageFormat imageFormat) throws IOException;

	/**
	 * Deletes the external resource at the given path.
	 *
	 * @param path
	 *            the path of the external resource (relative to the Module directory).
	 * @return {@code true} if the resource was successfully deleted, {@code false} if no resource exists at the given
	 *         path.
	 * @throws IllegalArgumentException
	 *             if the path does not start with a "/".
	 * @throws IOException
	 *             if an error occurs while deleting the resource.
	 * @throws IOException
	 *             if a writer to the resource is currently open.
	 */
	boolean deleteExternalResource(String path) throws IOException;

	/**
	 * Returns the path to an external resource.
	 *
	 * @param path a path relative to the current context.
	 * @return the corresponding resource path.
	 */
	String getExternalResourcePath(String path);

	/**
	 * Decodes a file with the given format and returns the path of the decoded file.
	 *
	 * @param path
	 *            the path of the file to decode
	 * @param format
	 *            the format used to decode the image
	 * @return the path of the decoded file
	 * @throws IOException
	 *             if an error occurs while decoding the resource
	 */
	String decodeImage(String path, Format format) throws IOException;
}
