/*
 * Java
 *
 * Copyright 2016-2024 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 */
package ej.junit;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Base class used by generated wrappers. It is where the test execution happens. Implements the {@link TestWrapper}
 * interface so it can be called by another application.
 */
public abstract class AbstractTestWrapper implements TestWrapper {

	protected final Class<?> testClass;
	protected final String testClassName;

	/**
	 * Creates a wrapper instance for the specified test class.
	 * <p>
	 * Queries and store the class name, so clients of the wrappers can look them up by name if necessary.
	 *
	 * @param testClass
	 *            the test class to instantiate.
	 */
	public AbstractTestWrapper(Class<?> testClass) {
		this.testClass = testClass;
		this.testClassName = testClass.getName();
	}

	protected boolean failed = false;
	protected String testMethod;
	protected long testMethodStart;
	protected Object test;

	/* package */int runs;
	/* package */int failures;
	/* package */int errors;
	protected TestListener testListener;

	/**
	 * Called by generated wrappers to report an error.
	 * <p>
	 * If a listener has been passed to {@link #run(TestListener)}, then the
	 * {@link TestListener#testError(String, String, long, Throwable, String)} is called.
	 *
	 * @param error
	 *            the {@link Throwable} that caused the error.
	 * @see #run(TestListener)
	 */
	protected void reportError(Throwable error) {
		long duration = System.currentTimeMillis() - this.testMethodStart;
		this.errors++;
		if (this.testListener != null) {
			this.testListener.testError(this.testClassName, this.testMethod, duration, error, error.getMessage());
		}
	}

	/**
	 * Called by generated wrappers to report a failure.
	 * <p>
	 * If a listener has been passed to {@link #run(TestListener)}, then the
	 * {@link TestListener#testFailure(String, String, long, Throwable, String)} is called.
	 *
	 * @param error
	 *            the {@link AssertionError} that caused the failure.
	 * @see #run(TestListener)
	 */
	protected void reportFailure(AssertionError error) {
		long duration = System.currentTimeMillis() - this.testMethodStart;
		this.failures++;
		if (this.testListener != null) {
			this.testListener.testFailure(this.testClassName, this.testMethod, duration, error, error.getMessage());
		}
	}

	/**
	 * Called by generated wrappers to report a success.
	 * <p>
	 * If a listener has been passed to {@link #run(TestListener)}, then the
	 * {@link TestListener#testSuccess(String, String, long)} is called.
	 *
	 * @see #run(TestListener)
	 */
	protected void reportSuccess() {
		long duration = System.currentTimeMillis() - this.testMethodStart;
		if (this.testListener != null) {
			this.testListener.testSuccess(this.testClassName, this.testMethod, duration);
		}
	}

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link BeforeClass}
	 *
	 * @throws Exception
	 *             if one of the {@link BeforeClass} methods throws one.
	 */
	abstract protected void runBeforeClassMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link AfterClass}
	 *
	 * @throws Exception
	 *             if one of the {@link AfterClass} methods throws one.
	 */
	abstract protected void runAfterClassMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link Before}
	 *
	 * @throws Exception
	 *             if one of the {@link Before} methods throws one.
	 */
	abstract protected void runBeforeMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link After}
	 *
	 * @throws Exception
	 *             if one of the {@link BeforeClass} methods throws one.
	 */
	abstract protected void runAfterMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of test methods tagged with {@link Test}
	 */
	abstract protected void runTestMethods();

	/* package */ boolean wrapperRunBeforeClassMethod() {
		try {
			this.testMethod = null;
			this.testMethodStart = System.currentTimeMillis();
			runBeforeClassMethods();
			return true;
		} catch (Throwable t) {
			reportError(t);
		}
		return false;
	}

	/* package */void wrapperRunAfterClassMethod() {
		try {
			this.testMethod = null;
			this.testMethodStart = System.currentTimeMillis();
			runAfterClassMethods();
		} catch (Throwable t) {
			reportError(t);
		}
	}

	/**
	 * Called by generated subclasses to trigger test initialization, by instantiating the test class and calling the
	 * {@link Before} methods.
	 *
	 * @param testMethodName
	 *            the name of the test method to initialize.
	 * @return true if the operation succeeded and the test can continue, false otherwise.
	 */
	protected final boolean testInitialize(String testMethodName) {
		try {
			this.runs++;
			if (this.testListener != null) {
				this.testListener.testInstance(this.testClass.getName());
			}
			this.test = this.testClass.newInstance();
			this.testMethod = testMethodName;

			runBeforeMethods();

			// test will actually start right after that
			if (this.testListener != null) {
				this.testListener.testStarted(this.testClass.getName(), testMethodName);
			}
			return true;
		} catch (Throwable t) {
			reportError(t);
		}
		return false;
	}

	/**
	 * Called by generated subclasses to trigger test finalization, by reporting the test execution status and calling
	 * the After methods.
	 *
	 * @param success
	 *            if true, test success success will be reported
	 */
	protected final void testFinalize(boolean success) {
		try {
			if (this.test != null) {
				runAfterMethods();
			}
		} catch (Throwable t) {
			reportError(t);
		} finally {
			if (this.test != null) {
				if (success) {
					reportSuccess();
				}
			}
			this.testMethod = null;
		}
	}

	private synchronized void wrapperRunTestMethods() {
		try {
			try {
				boolean ok = wrapperRunBeforeClassMethod();
				if (ok) {
					runTestMethods();
				}
			} finally {
				wrapperRunAfterClassMethod();
			}
		} catch (Throwable t) {
			reportError(t);
		}
	}

	@Override
	public String getTestClass() {
		return this.testClassName;
	}

	@Override
	public synchronized boolean run(TestListener listener) {
		this.errors = this.failures = 0;
		this.testListener = listener;
		if (this.testListener != null) {
			this.testListener.testSuiteStarted(this.testClassName);
		}
		wrapperRunTestMethods();
		if (this.testListener != null) {
			this.testListener.testSuiteEnded(this.testClassName, this.runs, this.failures, this.errors);
		}
		return this.errors == 0 && this.failures == 0;
	}

}
