/*
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.android;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import com.microej.android.application.ApplicationDisplay;
import com.microej.android.application.ApplicationSupport;
import com.microej.android.internal.InternalDisplayListener;
import com.microej.android.internal.InternalSupport;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Rect;

/**
 * The MicroEJ support engine provides the glue methods required to run a MicroEJ Application on an Android Platform.
 */
public class MicroEJSupport {

	private static final String PROPERTIES_PATH = "/properties/merged.properties.list";

	private static Engine engine;

	private MicroEJSupport() {
		// private constructor
	}

	/**
	 * Returns an instance of the application interface of the MicroEJ support engine.
	 * <p>
	 * This instance can be used by Android Applications.
	 *
	 * @return an instance of the application interface.
	 */
	public static ApplicationSupport getApplicationSupport() {
		return getEngine();
	}

	/**
	 * Returns an instance of the internal interface of the MicroEJ support engine.
	 * <p>
	 * This instance can be used by Android Implementations of Foundation Libraries.
	 *
	 * @return an instance of the internal interface.
	 */
	public static InternalSupport getInternalSupport() {
		return getEngine();
	}

	private static Engine getEngine() {
		if (engine == null) {
			engine = new Engine();
		}
		return engine;
	}

	private static class Engine implements ApplicationSupport, InternalSupport {

		private boolean isStarted;
		private Context context;
		private ApplicationDisplay display;
		private InternalDisplayListener displayListener;

		// support API

		@Override
		public void startup() {
			// check if already started
			if (this.isStarted) {
				return;
			}
			this.isStarted = true;

			// set application properties
			try (InputStream inputStream = MicroEJSupport.class.getResourceAsStream(PROPERTIES_PATH)) {
				if (inputStream != null) {
					Properties properties = new Properties();
					properties.load(inputStream);
					System.setProperties(properties);
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		@Override
		public void setContext(Context context) {
			this.context = context;
		}

		@Override
		public void setDisplay(ApplicationDisplay display) {
			if (this.display != display) {
				this.display = display;
				onDisplayTearing(display);
			}
		}

		@Override
		public void unsetDisplay(ApplicationDisplay display) {
			if (this.display == display) {
				this.display = null;
			}
		}

		@Override
		public void onDisplayTearing(ApplicationDisplay display) {
			if (this.display == display) {
				InternalDisplayListener displayListener = this.displayListener;
				if (displayListener != null) {
					displayListener.onTearing();
				}
			}
		}

		@Override
		public void onPointerPress(ApplicationDisplay display, int x, int y) {
			if (this.display == display) {
				InternalDisplayListener displayListener = this.displayListener;
				if (displayListener != null) {
					displayListener.onPointerPress(x, y);
				}
			}
		}

		@Override
		public void onPointerRelease(ApplicationDisplay display) {
			if (this.display == display) {
				InternalDisplayListener displayListener = this.displayListener;
				if (displayListener != null) {
					displayListener.onPointerRelease();
				}
			}
		}

		@Override
		public void onPointerMove(ApplicationDisplay display, int x, int y) {
			if (this.display == display) {
				InternalDisplayListener displayListener = this.displayListener;
				if (displayListener != null) {
					displayListener.onPointerMove(x, y);
				}
			}
		}

		// internal API

		@Override
		public void checkStarted() {
			if (!this.isStarted) {
				throw new IllegalStateException("MicroEJ support engine not started");
			}
		}

		@Override
		public Context getContext() {
			if (this.context == null) {
				throw new IllegalStateException("Application context instance of MicroEJ support engine not set");
			}
			return this.context;
		}

		@Override
		public void flush(Bitmap bitmap, Rect rect) {
			if (this.display != null) {
				this.display.flush(bitmap, rect);
			}
		}

		@Override
		public void setDisplayListener(InternalDisplayListener displayListener) {
			this.displayListener = displayListener;

			if (this.display != null) {
				this.displayListener.onTearing();
			}
		}

		@Override
		public void unsetDisplayListener(InternalDisplayListener displayListener) {
			if (this.displayListener == displayListener) {
				this.displayListener = null;
			}
		}
	}
}
