/*
 * Java
 *
 * Copyright 2015-2022 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.util.concurrent;

import java.util.ArrayDeque;
import java.util.Collection;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.RunnableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import ej.annotation.Nullable;
import ej.bon.Util;

/**
 * An executor that uses a single worker thread operating off an unbounded queue.
 */
public class SingleThreadExecutor implements Runnable, ExecutorService {
	private final Thread thread;
	private final Queue<Runnable> tasks;
	private boolean terminated;

	/**
	 * Creates a single thread executor.
	 * <p>
	 * A new thread is created and started immediately.
	 */
	public SingleThreadExecutor() {
		this.thread = new Thread(this, SingleThreadExecutor.class.getName());
		this.tasks = new ArrayDeque<Runnable>();
		this.thread.start();
	}

	@Override
	public void run() {
		while (!isTerminated()) {
			Runnable active;
			synchronized (this.tasks) {
				active = this.tasks.poll();
				if (active == null) {
					try {
						this.tasks.wait();
					} catch (InterruptedException e) {
						// stop execution ?
					}
				}
			}
			if (active != null) {
				active.run();
			}
		}
	}

	@Override
	public void execute(final Runnable r) {
		synchronized (this.tasks) {
			this.tasks.offer(r);
			this.tasks.notifyAll();
		}
	}

	@Override
	public void shutdown() {
		this.terminated = true;
		synchronized (this.tasks) {
			this.tasks.notifyAll();
		}
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public List<Runnable> shutdownNow() {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean isShutdown() {
		return this.terminated;
	}

	@Override
	public boolean isTerminated() {
		return this.terminated;
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> Future<T> submit(Callable<T> task) {
		RunnableFuture<T> ftask = newTaskFor(task);
		execute(ftask);
		return ftask;
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public <T> Future<T> submit(Runnable task, T result) {
		throw new UnsupportedOperationException();
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public Future<?> submit(Runnable task) {
		throw new UnsupportedOperationException();
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
		throw new UnsupportedOperationException();
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
			throws InterruptedException {
		throw new UnsupportedOperationException();
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
		throw new UnsupportedOperationException();
	}

	/**
	 * This method is unsupported by this implementation.
	 * <p>
	 * Calling this method will throw an {@link UnsupportedOperationException}.
	 */
	@Override
	public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
			throws InterruptedException, ExecutionException, TimeoutException {
		throw new UnsupportedOperationException();
	}

	private <T> RunnableFuture<T> newTaskFor(final Callable<T> callable) {
		return new RunnableFuture<T>() {

			final Object lock = new Object();

			boolean isDone = false;
			@Nullable
			T result;
			@Nullable
			Exception exception;

			@Override
			public boolean cancel(boolean mayInterruptIfRunning) {
				throw new UnsupportedOperationException();
			}

			@Override
			public boolean isCancelled() {
				return false;
			}

			@Override
			public boolean isDone() {
				synchronized (this.lock) {
					return this.isDone;
				}
			}

			@Override
			public @Nullable T get() throws InterruptedException, ExecutionException {
				synchronized (this.lock) {
					while (!isDone()) {
						this.lock.wait();
					}
				}
				return getDone();
			}

			@Override
			public @Nullable T get(long timeout, TimeUnit unit)
					throws InterruptedException, ExecutionException, TimeoutException {
				long timeoutms = TimeUnit.MILLISECONDS.convert(timeout, unit);
				long start = Util.platformTimeMillis();
				synchronized (this.lock) {
					while (!isDone()) {
						long current = Util.platformTimeMillis();
						long toWait = start + timeoutms - current;
						if (toWait < 0) {
							throw new TimeoutException();
						}
						this.lock.wait(toWait);
					}
				}
				return getDone();
			}

			@Override
			public void run() {
				T result;
				Exception exception;
				try {
					result = callable.call();
					exception = null;
				} catch (Exception e) {
					result = null;
					exception = e;
				}
				synchronized (this.lock) {
					this.result = result;
					this.exception = exception;
					this.isDone = true;
					this.lock.notifyAll();
				}
			}

			private @Nullable T getDone() throws ExecutionException {
				Exception e = this.exception;
				if (e != null) {
					throw new ExecutionException(e);
				}
				return this.result;
			}

		};
	}

}