/*
 * Java
 *
 * Copyright 2019-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.library.iot.rcommand.bluetooth;

import java.io.IOException;
import java.util.concurrent.TimeoutException;

import ej.annotation.Nullable;
import ej.library.iot.rcommand.bluetooth.commands.AddServiceCommand;
import ej.library.iot.rcommand.bluetooth.commands.ConnectCommand;
import ej.library.iot.rcommand.bluetooth.commands.DisableCommand;
import ej.library.iot.rcommand.bluetooth.commands.DisconnectCommand;
import ej.library.iot.rcommand.bluetooth.commands.DiscoverServicesCommand;
import ej.library.iot.rcommand.bluetooth.commands.EnableCommand;
import ej.library.iot.rcommand.bluetooth.commands.PollEventCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendExecuteWriteResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendNotificationCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPairRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPairResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPasskeyResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPrepareWriteResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendReadRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendReadResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.StartAdvertisingCommand;
import ej.library.iot.rcommand.bluetooth.commands.StartScanningCommand;
import ej.library.iot.rcommand.bluetooth.commands.StopAdvertisingCommand;
import ej.library.iot.rcommand.bluetooth.commands.StopScanningCommand;
import ej.library.iot.rcommand.bluetooth.notifications.EventAvailableNotificationListener;
import ej.rcommand.synchronous.Command;
import ej.rcommand.synchronous.RemoteCommandClient;

public class BluetoothHost {

	private static final BluetoothHost INSTANCE = new BluetoothHost();

	private final Object eventMonitor;

	private @Nullable RemoteCommandClient rcommandClient;
	private long commandTimeout;
	private boolean enabled;
	private int numAvailableEvents;

	private BluetoothHost() {
		this.eventMonitor = new Object();
		this.rcommandClient = null;
		this.commandTimeout = 0;
		this.enabled = false;
		this.numAvailableEvents = 0;
	}

	public static BluetoothHost getInstance() {
		return INSTANCE;
	}

	public void setup(RemoteCommandClient rcommandClient, long commandTimeout) {
		rcommandClient.registerNotificationListener(new EventAvailableNotificationListener());

		this.rcommandClient = rcommandClient;
		this.commandTimeout = commandTimeout;
	}

	public void onEventAvailable() {
		synchronized (this.eventMonitor) {
			this.numAvailableEvents++;
			this.eventMonitor.notifyAll();
		}
	}

	public int waitEvent(byte[] buffer, int bufferLength) {
		while (true) {
			synchronized (this.eventMonitor) {
				while (this.enabled && this.numAvailableEvents == 0) {
					try {
						this.eventMonitor.wait();
					} catch (InterruptedException e) {
						Thread.currentThread().interrupt();
						return 0;
					}
				}
				if (!this.enabled) {
					return 0;
				}
				this.numAvailableEvents = Math.max(this.numAvailableEvents - 1, 0);
			}

			byte[] event = sendCommand(new PollEventCommand(), this.commandTimeout);
			if (event == null || event.length == 0) {
				// no event available, keep waiting for an event
			} else if (event.length > bufferLength) {
				System.out.println("WARNING: Bluetooth event discarded due to buffer being too small");
			} else {
				System.arraycopy(event, 0, buffer, 0, event.length);
				return event.length;
			}
		}
	}

	public boolean enable() {
		this.enabled = sendBooleanCommand(new EnableCommand());
		return this.enabled;
	}

	public void disable() {
		sendCommand(new DisableCommand(), this.commandTimeout);

		synchronized (this.eventMonitor) {
			this.enabled = false;
			this.numAvailableEvents = 0;
			this.eventMonitor.notifyAll();
		}
	}

	public boolean startScanning(byte filterAction, byte filterType, byte[] filterData, int filterDataSize) {
		return sendBooleanCommand(new StartScanningCommand(filterAction, filterType, filterData, filterDataSize));
	}

	public boolean stopScanning() {
		return sendBooleanCommand(new StopScanningCommand());
	}

	public boolean startAdvertising(byte[] advertisementData, int advertisementDataSize) {
		return sendBooleanCommand(new StartAdvertisingCommand(advertisementData, advertisementDataSize));
	}

	public boolean stopAdvertising() {
		return sendBooleanCommand(new StopAdvertisingCommand());
	}

	public boolean connect(byte[] deviceAddress) {
		return sendBooleanCommand(new ConnectCommand(deviceAddress));
	}

	public boolean disconnect(short connHandle) {
		return sendBooleanCommand(new DisconnectCommand(connHandle));
	}

	public boolean sendPairRequest(short connHandle) {
		return sendBooleanCommand(new SendPairRequestCommand(connHandle));
	}

	public boolean sendPairResponse(short connHandle, boolean accept) {
		return sendBooleanCommand(new SendPairResponseCommand(connHandle, accept));
	}

	public boolean sendPasskeyResponse(short connHandle, boolean accept, int passkey) {
		return sendBooleanCommand(new SendPasskeyResponseCommand(connHandle, accept, passkey));
	}

	public boolean discoverServices(short connHandle, @Nullable byte[] uuid) {
		return sendBooleanCommand(new DiscoverServicesCommand(connHandle, uuid));
	}

	public boolean addService(byte[] service, short[] handles) {
		int numAttr = service[16] + service[17];
		int serviceSize = 18 + numAttr * 20;
		return sendBooleanCommand(new AddServiceCommand(service, serviceSize, handles));
	}

	public boolean sendReadRequest(short connHandle, short attributeHandle) {
		return sendBooleanCommand(new SendReadRequestCommand(connHandle, attributeHandle));
	}

	public boolean sendWriteRequest(short connHandle, short attributeHandle, byte[] value, int valueSize,
			boolean noResponse) {
		return sendBooleanCommand(
				new SendWriteRequestCommand(connHandle, attributeHandle, value, valueSize, noResponse));
	}

	public boolean sendReadResponse(short connHandle, short attributeHandle, byte status, byte[] value, int valueSize) {
		return sendBooleanCommand(new SendReadResponseCommand(connHandle, attributeHandle, status, value, valueSize));
	}

	public boolean sendWriteResponse(short connHandle, short attributeHandle, byte status) {
		return sendBooleanCommand(new SendWriteResponseCommand(connHandle, attributeHandle, status));
	}

	public boolean sendPrepareWriteResponse(short connHandle, short attributeHandle, byte status, byte[] value,
			int valueSize, int offset) {
		return sendBooleanCommand(
				new SendPrepareWriteResponseCommand(connHandle, attributeHandle, status, value, valueSize, offset));
	}

	public boolean sendExecuteWriteResponse(short connHandle, short attributeHandle, byte status) {
		return sendBooleanCommand(new SendExecuteWriteResponseCommand(connHandle, attributeHandle, status));
	}

	public boolean sendNotification(short connHandle, short attributeHandle, byte[] value, int valueSize,
			boolean confirm) {
		return sendBooleanCommand(new SendNotificationCommand(connHandle, attributeHandle, value, valueSize, confirm));
	}

	private @Nullable <T> T sendCommand(Command<T> command, long timeout) {
		RemoteCommandClient rcommandClient = this.rcommandClient;
		if (rcommandClient == null) {
			return null;
		}

		try {
			return rcommandClient.execute(command, timeout);
		} catch (IOException | InterruptedException | TimeoutException e) {
			System.out.println("WARNING: Bluetooth operation failed due to disconnection with Bluetooth Controller");
			return null;
		}
	}

	private boolean sendBooleanCommand(Command<Boolean> command) {
		Boolean result = sendCommand(command, this.commandTimeout);
		return (result != null && result.booleanValue());
	}
}
