/*
 * Java
 *
 * Copyright 2018 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bluetooth.callbacks;

import ej.bluetooth.BluetoothAdapter;
import ej.bluetooth.BluetoothDevice;
import ej.bluetooth.callbacks.impl.DefaultConnectionCallbacks;

/**
 * The <code>ConnectionCallbacks</code> interface provides callbacks related to the connection with a device.
 * <p>
 * When {@link BluetoothDevice#connect connecting to a device} or when {@link BluetoothAdapter#startAdvertising starting
 * to advertise}, the application should give the connection callbacks in order to be notified of the events related to
 * the connection with the device.
 * <p>
 * This library provides a default implementation of this interface: {@link DefaultConnectionCallbacks}.
 */
public interface ConnectionCallbacks {

	/**
	 * Called when a connection attempt fails.
	 *
	 * @param device
	 *            the device to which the connection has failed.
	 * @see BluetoothDevice#connect(ConnectionCallbacks)
	 */
	void onConnectFailed(BluetoothDevice device);

	/**
	 * Called when a connection attempt is successful or when a remote device connects to the adapter.
	 *
	 * @param device
	 *            the device to which the connection was established.
	 * @see BluetoothDevice#connect(ConnectionCallbacks)
	 * @see BluetoothAdapter#startAdvertising(AdvertisementCallbacks, ConnectionCallbacks, byte[])
	 */
	void onConnected(BluetoothDevice device);

	/**
	 * Called when the connection with a device is closed.
	 *
	 * @param device
	 *            the device to which the connection was closed.
	 * @see BluetoothDevice#disconnect()
	 */
	void onDisconnected(BluetoothDevice device);

	/**
	 * Called when a pairing request from a device is received.
	 *
	 * @param device
	 *            the device from which the pairing request was received.
	 * @see BluetoothDevice#pairReply(boolean)
	 */
	void onPairRequest(BluetoothDevice device);

	/**
	 * Called when the pairing procedure with a device is completed.
	 *
	 * @param device
	 *            the device with which the pairing procedure was completed.
	 * @param success
	 *            true if the pairing procedure was successful, false otherwise.
	 * @see BluetoothDevice#pair()
	 * @see BluetoothDevice#pairReply(boolean)
	 */
	void onPairCompleted(BluetoothDevice device, boolean success);

	/**
	 * Called when a passkey request from a device is received.
	 *
	 * @param device
	 *            the device from which the passkey request was received.
	 * @see BluetoothDevice#passkeyReply(boolean, int)
	 */
	void onPasskeyRequest(BluetoothDevice device);

	/**
	 * Called when a passkey is generated to be displayed to the user.
	 *
	 * @param device
	 *            the device to which the passkey request was sent.
	 * @param passkey
	 *            the passkey which was generated.
	 * @see BluetoothDevice#passkeyReply(boolean, int)
	 */
	void onPasskeyGenerated(BluetoothDevice device, int passkey);

	/**
	 * Called when the services of the device are discovered.
	 *
	 * @param device
	 *            the device of which the services were discovered.
	 * @see BluetoothDevice#discoverServices()
	 */
	void onServicesDiscovered(BluetoothDevice device);
}
