/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.background;

import ej.annotation.Nullable;
import ej.bon.XMath;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;

/**
 * Draws a plain color on the background with a round corner.
 * <p>
 * Should be used in association with a {@link ej.mwt.style.outline.border.RoundedBorder} to:
 * <ul>
 * <li>anti-alias its edges by using the same color,</li>
 * <li>draw a colored border that fits its shape.</li>
 * </ul>
 * For a better rendering, the border radius must be a bit smaller than the background one.
 * <p>
 * The corner radius is stored as a <code>char</code> for heap optimization and therefore cannot exceed
 * <code>65535</code>.
 */
public class RoundedBackground implements Background {

	private final int color;
	private final char cornerRadius;

	/**
	 * Creates a rounded background.
	 * <p>
	 * The given corner radius is clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param color
	 *            the color to set.
	 * @param cornerRadius
	 *            the corner radius to set.
	 */
	public RoundedBackground(int color, int cornerRadius) {
		this.color = color;
		this.cornerRadius = (char) XMath.limit(cornerRadius, 0, Character.MAX_VALUE);
	}

	/**
	 * Gets the color.
	 *
	 * @return the color.
	 */
	public int getColor() {
		return this.color;
	}

	/**
	 * Gets the corner radius.
	 *
	 * @return the corner radius.
	 */
	public int getCornerRadius() {
		return this.cornerRadius;
	}

	@Override
	public boolean isTransparent(int width, int height) {
		// Is transparent because it is usually associated with an anti-aliased border
		// and because its corners are not drawn (if the widget is moved, its corners are not repainted).
		return true;
	}

	@Override
	public void apply(GraphicsContext g, int width, int height) {
		// Since draw ellipse arc is not available, both radius have the same size.
		int cornerRadius = Math.min(this.cornerRadius, (Math.min(width, height) - 1) >> 1);

		g.setColor(this.color);
		Painter.fillRoundedRectangle(g, 1, 1, width - 2, height - 2, cornerRadius << 1, cornerRadius << 1);

		// Set background color to optimize future anti-aliased drawings.
		g.setBackgroundColor(this.color);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj instanceof RoundedBackground) {
			RoundedBackground background = (RoundedBackground) obj;
			return this.color == background.color && this.cornerRadius == background.cornerRadius;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 17 * this.color + this.cornerRadius;
	}

}
