/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.buffer;

import ej.fp.FrontPanel;
import ej.fp.Image;
import ej.fp.Widget;
import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.Rectangle;

/**
 * One buffer in RAM and one buffer on the display.
 */
public class SingleBufferPolicy implements DisplayBufferPolicy {

	private Widget displayWidget;

	private Image backBuffer;
	private Image frontBuffer;

	@Override
	public void setDisplayProperties(Widget displayWidget, int width, int height, int initialColor) {
		this.displayWidget = displayWidget;
		this.backBuffer = FrontPanel.getFrontPanel().newImage(width, height, initialColor, false);
		this.frontBuffer = FrontPanel.getFrontPanel().newImage(width, height, initialColor, false);
	}

	@Override
	public int getBufferCount() {
		return 1;
	}

	@Override
	public boolean isDoubleBuffered() {
		return true;
	}

	@Override
	public void flush(DisplayBufferManager bufferManager, Rectangle[] rectangles) {
		// Transmit what has changed to the display.
		for (Rectangle rectangle : rectangles) {
			this.frontBuffer.drawImage(this.backBuffer, rectangle.getX1(), rectangle.getY1(), rectangle.getWidth(),
					rectangle.getHeight(), rectangle.getX1(), rectangle.getY1(), rectangle.getWidth(),
					rectangle.getHeight());
		}
		this.displayWidget.repaint();
		bufferManager.simulateFlushTime();
	}

	@Override
	public Image getBackBuffer() {
		return this.backBuffer;
	}

	@Override
	public Image getFrontBuffer() {
		return this.frontBuffer;
	}

	@Override
	public void dispose() {
		FrontPanel.getFrontPanel().disposeIfNotNull(this.backBuffer, this.frontBuffer);
	}

}
