/*
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.wear.framework.services;

import java.io.IOException;

/**
 * This service allows Features to create and delete external resources.
 */
public interface ExternalResourceService {

	/**
	 * Enumerates formats.
	 */
	public enum Format {

		/**
		 * RGB565 format.
		 */
		RGB565(0x04),

		/**
		 * ARGB4444 format.
		 */
		ARGB4444(0x06),

		/**
		 * A8 format.
		 */
		A8(0x08);

		private final byte value;

		private Format(int value) {
			this.value = (byte) value;
		}

		/**
		 * Returns the SNI value of this format.
		 *
		 * @return the SNI value.
		 */
		public byte getSNIContext() {
			return this.value;
		}
	}

	/**
	 * Creates an external resource at the given path with the given data.
	 *
	 * @param path
	 *            the path of the external resource.
	 * @param data
	 *            the data of the external resource.
	 * @throws IOException
	 *             if the external resource could not be created.
	 */
	void createExternalResource(String path, byte[] data) throws IOException;

	/**
	 * Creates an external image at the given path in the given format with the given compressed data.
	 *
	 * @param path
	 *            the path of the external image.
	 * @param format
	 *            the format of the external image.
	 * @param compressedData
	 *            the compressed data of the external image.
	 * @throws IOException
	 *             if the external image could not be created.
	 */
	void createExternalImage(String path, Format format, byte[] compressedData) throws IOException;

	/**
	 * Deletes the external resources contained in the folder at the given path.
	 *
	 * @param path
	 *            the path of the folder.
	 */
	void deleteExternalResources(String path);
}
