/*
 * Copyright (c) 1999, 2012, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2021 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package javax.net.ssl;

import java.net.HttpURLConnection;
import java.net.URL;

import ej.annotation.Nullable;

/**
 * <code>HttpsURLConnection</code> extends <code>HttpURLConnection</code> with support for https-specific features.
 * <P>
 * See <A HREF="http://www.w3.org/pub/WWW/Protocols/"> http://www.w3.org/pub/WWW/Protocols/</A> and
 * <A HREF="http://www.ietf.org/"> RFC 2818 </A> for more details on the https specification.
 * <P>
 * This class uses <code>HostnameVerifier</code> and <code>SSLSocketFactory</code>. There are default implementations
 * defined for both classes. However, the implementations can be replaced on a per-class (static) or per-instance basis.
 * All new <code>HttpsURLConnection</code>s instances will be assigned the "default" static values at instance creation,
 * but they can be overriden by calling the appropriate per-instance set method(s) before <code>connect</code>ing.
 *
 * @since 1.4
 */
abstract public class HttpsURLConnection extends HttpURLConnection {
	/**
	 * Creates an <code>HttpsURLConnection</code> using the URL specified.
	 *
	 * @param url
	 *            the URL
	 */
	protected HttpsURLConnection(URL url) {
		super(url);
	}

	@Nullable
	private static SSLSocketFactory defaultSSLSocketFactory = null;

	/**
	 * The <code>SSLSocketFactory</code> inherited when an instance of this class is created.
	 */
	private SSLSocketFactory sslSocketFactory = getDefaultSSLSocketFactory();

	/**
	 * Sets the default <code>SSLSocketFactory</code> inherited by new instances of this class.
	 * <P>
	 * The socket factories are used when creating sockets for secure https URL connections.
	 *
	 * @param sf
	 *            the default SSL socket factory
	 * @throws IllegalArgumentException
	 *             if the SSLSocketFactory parameter is null.
	 * @throws SecurityException
	 *             if a security manager exists and its <code>checkSetFactory</code> method does not allow a socket
	 *             factory to be specified.
	 * @see #getDefaultSSLSocketFactory()
	 */
	public static void setDefaultSSLSocketFactory(SSLSocketFactory sf) {
		if (sf == null) {
			throw new IllegalArgumentException("no default SSLSocketFactory specified");
		}

		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
			sm.checkPermission(new RuntimePermission("setFactory"));
		}
		defaultSSLSocketFactory = sf;
	}

	/**
	 * Gets the default static <code>SSLSocketFactory</code> that is inherited by new instances of this class.
	 * <P>
	 * The socket factories are used when creating sockets for secure https URL connections.
	 *
	 * @return the default <code>SSLSocketFactory</code>
	 * @see #setDefaultSSLSocketFactory(SSLSocketFactory)
	 */
	public static SSLSocketFactory getDefaultSSLSocketFactory() {
		SSLSocketFactory defaultSSLSocketFactory = HttpsURLConnection.defaultSSLSocketFactory;
		if (defaultSSLSocketFactory == null) {
			defaultSSLSocketFactory = (SSLSocketFactory) SSLSocketFactory.getDefault();
			HttpsURLConnection.defaultSSLSocketFactory = defaultSSLSocketFactory;
		}
		return defaultSSLSocketFactory;
	}

	/**
	 * Sets the <code>SSLSocketFactory</code> to be used when this instance creates sockets for secure https URL
	 * connections.
	 * <P>
	 * New instances of this class inherit the default static <code>SSLSocketFactory</code> set by
	 * {@link #setDefaultSSLSocketFactory(SSLSocketFactory) setDefaultSSLSocketFactory}. Calls to this method replace
	 * this object's <code>SSLSocketFactory</code>.
	 *
	 * @param sf
	 *            the SSL socket factory
	 * @throws IllegalArgumentException
	 *             if the <code>SSLSocketFactory</code> parameter is null.
	 * @throws SecurityException
	 *             if a security manager exists and its <code>checkSetFactory</code> method does not allow a socket
	 *             factory to be specified.
	 * @see #getSSLSocketFactory()
	 */
	public void setSSLSocketFactory(SSLSocketFactory sf) {
		if (sf == null) {
			throw new IllegalArgumentException("no SSLSocketFactory specified");
		}

		SecurityManager sm = System.getSecurityManager();
		if (sm != null) {
			sm.checkPermission(new RuntimePermission("setFactory"));
		}
		this.sslSocketFactory = sf;
	}

	/**
	 * Gets the SSL socket factory to be used when creating sockets for secure https URL connections.
	 *
	 * @return the <code>SSLSocketFactory</code>
	 * @see #setSSLSocketFactory(SSLSocketFactory)
	 */
	public SSLSocketFactory getSSLSocketFactory() {
		return this.sslSocketFactory;
	}
}
