/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */

package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests the API of {@link VectorFont} class.
 */
public class TestFontMeasures {

	private static final int PADDING = 1;
	private static final char CHAR_SQUARE = 'A';
	private static final char CHAR_HALF_WIDTH_HALF_HEIGHT = 'K';
	private static final char CHAR_QUARTER_WIDTH_TREE_QUARTERS_HEIGHT = 'L';
	private static final char CHAR_NO_GLYPH = 'Z';
	private static final String STRING_SQUARE_HALF_SQUARE = "ADA";
	private static final String STRING_SQUARE = "A";
	private static final String STRING_HALF_WIDTH_FULL_THIN = "KAE";
	private static final String STRING_OUTLEFT_FULL_OUTRIGHT = "MAN";
	private static final String STRING_LEFTINFLEXIONPOINT = "PA";
	private static final String STRING_LEFTSPACE = " A";
	private static final String STRING_RIGHTSPACE = "A ";
	private static final String STRING_SPACEINSIDE = "A A";
	private static final String STRING_SPACE = " ";

	// TODO refine the test precision value when comparing floats if needed (this is an arbitrary value)
	private static final float DELTA = TestUtilities.isOnAndroid() ? 0.04f : 0.01f;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test the measurement of the width of a string.
	 */
	@Test
	public void testMeasureStringWidth() {
		VectorFont font = getTestFont();

		int fontSizes[] = { 10, 18, 24, 28, 36, 50, 70, 96 };
		for (int fontSize : fontSizes) {
			float stringWidth = font.measureStringWidth(STRING_SQUARE_HALF_SQUARE, fontSize);
			Assert.assertEquals("string width", fontSize * 2.5f, stringWidth, DELTA);
		}
	}

	/**
	 * Test the measurement of the width of a string with space before first glyph and after last glyph.
	 */
	@Test
	public void testMeasureStringWidthSpaceOnStartEnd() {
		VectorFont font = getTestFont();

		int fontSizes[] = { 10, 18, 24, 28, 36, 50, 70, 96 };
		for (int fontSize : fontSizes) {
			float stringWidth = font.measureStringWidth(STRING_HALF_WIDTH_FULL_THIN, fontSize);
			float expected_width = (0.75f + 1f + 0.1f) * fontSize;
			Assert.assertEquals("string width", expected_width, stringWidth, DELTA);
		}
	}

	/**
	 * Test the measurement of the width of a string with glyphs outside bbox on left and right.
	 */
	@Test
	public void testMeasureStringWidthOutLeftRight() {
		VectorFont font = getTestFont();

		int fontSizes[] = { 10, 18, 24, 28, 36, 50, 70, 96 };
		for (int fontSize : fontSizes) {
			float stringWidth = font.measureStringWidth(STRING_OUTLEFT_FULL_OUTRIGHT, fontSize);
			float expected_width = (1.3f + 1f + 1.2f) * fontSize;
			Assert.assertEquals("string width", expected_width, stringWidth, DELTA);
		}
	}

	/**
	 * Test the measurement of the width of a string with letter spacing.
	 */
	@Test
	public void testMeasureStringWidthWithLetterSpacing() {
		VectorFont font = getTestFont();
		float fontSize = 100;

		// positive letter spacing
		float letterSpacing = 20;
		float stringWidth = font.measureStringWidth(STRING_SQUARE_HALF_SQUARE, fontSize, letterSpacing);
		Assert.assertEquals("string width with positive letter spacing", fontSize * 2.5f + 2 * letterSpacing,
				stringWidth, DELTA);

		// negative letter spacing
		letterSpacing = -20;
		stringWidth = font.measureStringWidth(STRING_SQUARE_HALF_SQUARE, fontSize, letterSpacing);
		Assert.assertEquals("string width with negative letter spacing", fontSize * 2.5f + 2 * letterSpacing,
				stringWidth, DELTA);

		// SpaceOnStartEnd
		letterSpacing = 10;
		stringWidth = font.measureStringWidth(STRING_HALF_WIDTH_FULL_THIN, fontSize, letterSpacing);
		Assert.assertEquals("string width with positive letter spacing(space on start and end)",
				fontSize * 1.85f + 2 * letterSpacing, stringWidth, DELTA);

		// OutLeftRight
		letterSpacing = 16;
		stringWidth = font.measureStringWidth(STRING_OUTLEFT_FULL_OUTRIGHT, fontSize, letterSpacing);
		Assert.assertEquals("string width with positive letter spacing(glyphs out on left and right)",
				fontSize * 3.5f + 2 * letterSpacing, stringWidth, DELTA);
	}

	/**
	 * Test the measurement of the width of a string when providing a font size lower or equal to 0.
	 */
	@Test
	public void testMeasureStringWidthSizeLowerOrEqualToZero() {
		VectorFont font = getTestFont();

		int fontSizes[] = { 0, -30 };
		for (int fontSize : fontSizes) {
			float stringWidth = font.measureStringWidth(STRING_SQUARE_HALF_SQUARE, fontSize);
			Assert.assertEquals("string width when size <= 0", 0, stringWidth, 0);
		}
	}

	/**
	 * Test the measurement of the width of a character with no glyph.
	 */
	@Test
	public void testMeasureNotDefGlyphWitdh() {

		// Disable on Android because the .notdef glyph from the is ignored and the character is rendered with a default
		// system font.
		// See https://youtrack.cross/issue/M0092MEJAUI-2600
		if (TestUtilities.isOnAndroid()) {
			return;
		}

		VectorFont font = getTestFont();

		// Unknown character glyph are replaced by .notdef glyph
		// .notdef glyph of the test font has a width which is 40% of font size
		Assert.assertEquals("char width of unknown character glyph", 40f,
				font.measureStringWidth(Character.toString(CHAR_NO_GLYPH), 100f), DELTA);
	}

	/**
	 * Test using multiple fonts.
	 */
	@Test
	public void testMeasureMultipleFonts() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = getTestFont();

		// Draw a character with the first font
		float fontSize = 100;
		String string = STRING_SQUARE;
		float glyphWidth = font.measureStringWidth(string, fontSize);

		Assert.assertEquals(fontSize, glyphWidth, DELTA);

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		int color = Colors.WHITE;
		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				0f);
		display.flush();

		// check inside the character
		TestUtilities.checkArea("first font", color, x, y, (int) glyphWidth, (int) fontSize, PADDING);
		// check outside the character (background color)
		TestUtilities.checkPeripheralArea("first font", Colors.BLACK, x, y, (int) glyphWidth, (int) fontSize, 50, 3);

		TestUtilities.clearScreen();

		// Load a new font
		VectorFont secondFont = VectorFont.loadFont("/fonts/secondfont.ttf");

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, secondFont, fontSize, matrix, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER, 0f);
		display.flush();

		Assert.assertEquals(fontSize * 0.2f, secondFont.measureStringWidth(string, fontSize), DELTA);

		float targetWidth = fontSize * 0.2f;
		float targetHeight = fontSize * 0.8f;

		int startY = (int) (y + secondFont.getBaselinePosition(fontSize) - targetHeight);

		// check inside the character
		TestUtilities.checkArea("second font", color, x, startY, (int) targetWidth, (int) targetHeight, PADDING);
		// check outside the character
		TestUtilities.checkPeripheralArea("second font", Colors.BLACK, x, startY, (int) targetWidth, (int) targetHeight,
				50, 3);
	}

	/**
	 * Test the glyph measurements.
	 */
	@Test
	public void testMeasureGlyphDimension() {
		VectorFont font = getTestFont();

		float fontSizes[] = { 12, 20, 24, 32, 64, 96 };
		for (float fontSize : fontSizes) {

			float glyphHeight = font.measureStringHeight(Character.toString(CHAR_HALF_WIDTH_HALF_HEIGHT), fontSize);
			Assert.assertEquals(fontSize / 2, glyphHeight, DELTA);

			float glyphWidth = font.measureStringWidth(Character.toString(CHAR_HALF_WIDTH_HALF_HEIGHT), fontSize);
			Assert.assertEquals(fontSize / 2, glyphWidth, DELTA);

			if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
				System.out.println("height:" + fontSize + " glyphHeight:" + glyphHeight + " glyphWidth:" + glyphWidth);
			}

			glyphHeight = font.measureStringHeight(Character.toString(CHAR_QUARTER_WIDTH_TREE_QUARTERS_HEIGHT),
					fontSize);
			Assert.assertEquals(fontSize / 4 * 3, glyphHeight, DELTA);

			glyphWidth = font.measureStringWidth(Character.toString(CHAR_QUARTER_WIDTH_TREE_QUARTERS_HEIGHT), fontSize);
			Assert.assertEquals(fontSize / 4, glyphWidth, DELTA);

			if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
				System.out.println("height:" + fontSize + " glyphHeight:" + glyphHeight + " glyphWidth:" + glyphWidth);
			}
		}
	}

	/**
	 * Test the glyph measurements when providing negative size.
	 */
	@Test
	public void testMeasureGlyphSizeLowerOrEqualToZero() {
		VectorFont font = getTestFont();

		float fontSizes[] = { 0, -30 };
		for (float fontSize : fontSizes) {

			float glyphHeight = font.measureStringHeight(Character.toString(CHAR_SQUARE), fontSize);
			Assert.assertEquals(0, glyphHeight, 0);

			float glyphWidth = font.measureStringWidth(Character.toString(CHAR_SQUARE), fontSize);
			Assert.assertEquals(0, glyphWidth, 0);

			if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
				System.out.println("height:" + fontSize + " glyphHeight:" + glyphHeight + " glyphWidth:" + glyphWidth);
			}
		}
	}

	/**
	 * Test that calling the {@link VectorFont} methods on a closed font throws an exception.
	 */
	@Test
	public void testMeasureWhenFontIsClosed() {
		VectorFont font = getTestFont();
		font.close();

		int errorCode = Integer.MAX_VALUE;
		try {
			font.getHeight(30);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("getHeight on closed font", VectorGraphicsException.RESOURCE_CLOSED, errorCode);

		try {
			font.measureStringHeight(STRING_SQUARE_HALF_SQUARE, 30);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("measureStringHeight on closed font", VectorGraphicsException.RESOURCE_CLOSED, errorCode);

		errorCode = Integer.MAX_VALUE;
		try {
			font.measureStringWidth(STRING_SQUARE_HALF_SQUARE, 30);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("measureStringWidth on closed font", VectorGraphicsException.RESOURCE_CLOSED, errorCode);

		errorCode = Integer.MAX_VALUE;
		try {
			font.measureStringWidth(STRING_SQUARE_HALF_SQUARE, 30, 3);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("measureStringWidth with letterSpacing on closed font", errorCode,
				VectorGraphicsException.RESOURCE_CLOSED);

		errorCode = Integer.MAX_VALUE;
		try {
			font.getBaselinePosition(30);
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}
		Assert.assertEquals("getBaselinePosition on closed font", VectorGraphicsException.RESOURCE_CLOSED, errorCode);
	}

	/**
	 * Test the measurement of the baseline position.
	 */
	@Test
	public void testMeasureBaselinePosition() {
		VectorFont font = getTestFont();
		Assert.assertEquals("baseline position, size > 0", 80, Math.round(font.getBaselinePosition(100)), DELTA);
		Assert.assertEquals("baseline position, size == 0", 0, font.getBaselinePosition(0), DELTA);
		Assert.assertEquals("baseline position, size < 0", 0, font.getBaselinePosition(-1), DELTA);

		int fontSize = 50;
		font = VectorFont.loadFont("/fonts/secondfont.ttf");
		Assert.assertEquals("secondfont.ttf baseline", fontSize * 1.200, font.getBaselinePosition(fontSize), 1);

		font = VectorFont.loadFont("/fonts/neuzeit.ttf");
		Assert.assertEquals("neuzeit.ttf baseline", fontSize * 0.75, font.getBaselinePosition(fontSize), 1);

		font = VectorFont.loadFont("/fonts/seguiemj_reduced.ttf");
		Assert.assertEquals("seguiemj_reduced.ttf baseline", fontSize * 0.73, font.getBaselinePosition(fontSize), 1);

		font = VectorFont.loadFont("/fonts/RobotoCondensed-Regular-reduced.ttf");
		Assert.assertEquals("RobotoCondensed-Regular-reduced baseline", fontSize * 0.93,
				font.getBaselinePosition(fontSize), 1);
	}

	/**
	 * Test that calling the {@link VectorFont} methods measuring method returns float values.
	 */
	@Test
	public void testMeasureFloat() {
		VectorFont font = getTestFont();

		float fontSize = 50.7f;
		float glyphHeight = font.measureStringHeight(Character.toString(CHAR_SQUARE), fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize, glyphHeight, DELTA);

		float glyphWidth = font.measureStringWidth(Character.toString(CHAR_SQUARE), fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("glyphWidth:" + glyphWidth);
		}
		Assert.assertEquals("glyph width", fontSize, glyphWidth, DELTA);

		float stringWidth = font.measureStringWidth(STRING_SQUARE_HALF_SQUARE, fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("stringWidth:" + glyphWidth);
		}
		Assert.assertEquals("string width", fontSize * 2.5f, stringWidth, DELTA);

		float baselinePosition = font.getBaselinePosition(fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("baselinePosition:" + baselinePosition + " <-> " + (fontSize * 0.8));
		}
		Assert.assertEquals("baseline position", fontSize * 0.8f, baselinePosition, DELTA * 10);

	}

	private static VectorFont getTestFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf");
	}

	/**
	 * Test the measurement of the height of different strings.
	 */
	@Test
	public void testMeasureStringHeight() {
		VectorFont font = getTestFont();

		float fontSize = 100f;

		float glyphHeight = font.measureStringHeight("K", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("K glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.5f, glyphHeight, DELTA);

		glyphHeight = font.measureStringHeight("I", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("K glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.6f, glyphHeight, DELTA);

		glyphHeight = font.measureStringHeight("KI", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("KI glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.875f, glyphHeight, DELTA);

		glyphHeight = font.measureStringHeight("O", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("O glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.304, glyphHeight, DELTA);

		glyphHeight = font.measureStringHeight("OOO", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("OOO glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.304, glyphHeight, DELTA);

		glyphHeight = font.measureStringHeight("OOOIO", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("OOOIO glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.6, glyphHeight, DELTA);

		glyphHeight = font.measureStringHeight("OKOOO", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("OKOOO glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.779, glyphHeight, DELTA);

		glyphHeight = font.measureStringHeight("OKOIO", fontSize);
		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			System.out.println("OKOIO glyphHeight:" + glyphHeight);
		}
		Assert.assertEquals("glyph height", fontSize * 0.875, glyphHeight, DELTA);

	}

	/**
	 * Test the measurement of the height of different fonts.
	 */
	@Test
	public void testMeasureFontHeight() {
		VectorFont font = null;

		float fontSizes[] = { 10, 12, 20, 24, 32, 64, 96 };
		for (float fontSize : fontSizes) {

			font = VectorFont.loadFont("/fonts/firstfont.ttf");
			Assert.assertEquals("firstfont.ttf height", fontSize * 1.090, font.getHeight(fontSize), DELTA * 5);

			font = VectorFont.loadFont("/fonts/secondfont.ttf");
			Assert.assertEquals("secondfont.ttf height", fontSize * 1.590, font.getHeight(fontSize), DELTA * 5);

			font = VectorFont.loadFont("/fonts/neuzeit.ttf");
			Assert.assertEquals("neuzeit.ttf height", fontSize * 1.1503, font.getHeight(fontSize), DELTA * 5);

			font = VectorFont.loadFont("/fonts/seguiemj_reduced.ttf");
			Assert.assertEquals("seguiemj_reduced.ttf height", fontSize * 1.0703, font.getHeight(fontSize), DELTA * 5);
		}
	}

	/**
	 * Test the measurement of the width of a string with a first glyph that has a left inflexion point before the first
	 * drawn pixel.
	 */
	@Test
	public void testMeasureStringWidthWithLeftInflexionPoint() {
		VectorFont font = getTestFont();
		float fontSize = 100;

		// positive letter spacing
		float stringWidth = font.measureStringWidth(STRING_LEFTINFLEXIONPOINT, fontSize, 0);
		Assert.assertEquals("string width with positive letter spacing", fontSize * 1.9f, stringWidth, DELTA);
	}

	/**
	 * Test the measurement of a string with an character displayed with an empty glyph.
	 */
	@Test
	public void testMeasureStringSpaceGlyph() {
		VectorFont font = getTestFont();
		float fontSize = 100;

		float stringWidth = font.measureStringWidth(STRING_SPACE, fontSize, 0);
		Assert.assertEquals("space string width", 0, stringWidth, DELTA);

		float stringHeight = font.measureStringWidth(STRING_SPACE, fontSize, 0);
		Assert.assertEquals("space string height", 0, stringHeight, DELTA);

		// space at start of string
		stringWidth = font.measureStringWidth(STRING_LEFTSPACE, fontSize, 0);
		Assert.assertEquals("space at start of string", fontSize * 1f, stringWidth, DELTA);

		// space at end of string
		stringWidth = font.measureStringWidth(STRING_RIGHTSPACE, fontSize, 0);
		Assert.assertEquals("space at end of string", fontSize * 1f, stringWidth, DELTA);

		// space inside the string
		stringWidth = font.measureStringWidth(STRING_SPACEINSIDE, fontSize, 0);
		Assert.assertEquals("space inside the string", fontSize * 2.4f, stringWidth, DELTA);

	}

}
