/*
 * Java
 *
 * Copyright 2022-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.BufferedVectorImage;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.VectorImage;

/**
 * Tests the drawing of MicroVG shapes into a BufferedVectorImage and using transparency.
 */
public class TestBufferedVectorImageTransparency {

	interface Drawer {

		/**
		 * Draws something in graphics context in the specified rectangle.
		 *
		 * @param g
		 *            the destination
		 * @param x
		 *            the rectangle top left X-coordinate
		 * @param y
		 *            the rectangle top left Y-coordinate
		 * @param w
		 *            the rectangle's width
		 * @param h
		 *            the rectangle's height
		 */
		void draw(GraphicsContext g, int x, int y, int w, int h);

	}

	interface AlphaDrawer {

		/**
		 * Draws something in graphics context in the specified rectangle ans opacity.
		 *
		 * @param g
		 *            the destination
		 * @param x
		 *            the rectangle top left X-coordinate
		 * @param y
		 *            the rectangle top left Y-coordinate
		 * @param w
		 *            the rectangle's width
		 * @param h
		 *            the rectangle's height
		 * @param alpha
		 *            the opacity
		 */
		void draw(GraphicsContext g, int x, int y, int w, int h, int alpha);

	}

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.reset();
		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: FillPath
	 */
	@Test
	public static void testVGPainterFillPath() {
		testDrawer("FillPath", new AlphaDrawer() { //$NON-NLS-1$
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				g.setColor(Colors.RED);
				VectorGraphicsPainter.fillPath(g, getPath(w, h), getTranslatedMatrix(x, y), FillType.WINDING, alpha,
						BlendMode.SRC_OVER);
			}
		}, 20, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: FillGradientPath
	 */
	@Test
	public static void testVGPainterFillGradientPath() {
		testDrawer("FillPath", new AlphaDrawer() { //$NON-NLS-1$
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				LinearGradient gradient = new LinearGradient(0, 0, w - 1, h - 1,
						new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
				VectorGraphicsPainter.fillGradientPath(g, getPath(w, h), getTranslatedMatrix(x, y), gradient,
						FillType.WINDING, alpha, BlendMode.SRC_OVER);
			}
		}, 20, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: DrawString
	 */
	@Test
	public static void testVGPainterDrawString() {
		testDrawer("DrawString", new AlphaDrawer() { //$NON-NLS-1$
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				g.setColor(Colors.WHITE);
				VectorGraphicsPainter.drawString(g, "ABC", getFont(), h, getTranslatedMatrix(x, y), alpha, //$NON-NLS-1$
						BlendMode.SRC_OVER, 0);
			}
		}, 60, 20, 0.01f);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: DrawStringGradient
	 */
	@Test
	public static void testVGPainterDrawStringGradient() {
		testDrawer("DrawStringGradient", new AlphaDrawer() { //$NON-NLS-1$
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				LinearGradient gradient = new LinearGradient(0, 0, w - 1, h - 1,
						new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
				VectorGraphicsPainter.drawGradientString(g, "ABC", getFont(), h, getTranslatedMatrix(x, y), gradient, //$NON-NLS-1$
						alpha, BlendMode.SRC_OVER, 0);
			}
		}, 60, 20, 0.01f);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawImage.
	 */
	@Test
	public static void testVGPainterDrawImage() {
		testDrawer("drawImage", new AlphaDrawer() { //$NON-NLS-1$
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				VectorImage image = getImage("red_rectangle"); //$NON-NLS-1$
				Matrix matrix = getTranslatedMatrix(x, y);
				matrix.preScale(w / image.getWidth(), h / image.getHeight());
				VectorGraphicsPainter.drawImage(g, image, matrix, alpha);
			}
		}, 50, 50, 0.01f);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawImage with several gradients.
	 */
	@Test
	public static void testVGPainterDrawGradientImage() {
		testDrawer("drawGradientImage", new AlphaDrawer() { //$NON-NLS-1$
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				VectorImage image = getImage("gradient"); //$NON-NLS-1$
				Matrix matrix = getTranslatedMatrix(x, y);
				matrix.preScale(w / image.getWidth(), h / image.getHeight());
				VectorGraphicsPainter.drawImage(g, image, matrix, alpha);
			}
		}, 50, 50, 0.01f);
	}

	private static void testDrawer(String label, AlphaDrawer drawer, int w, int h) {
		testDrawer(label, drawer, w, h, 0);
	}

	private static void testDrawer(String label, AlphaDrawer drawer, int w, int h, float tolerance) {
		try (BufferedVectorImage image = new BufferedVectorImage(w, h)) {

			Display display = Display.getDisplay();
			GraphicsContext g = display.getGraphicsContext();
			GraphicsContext ig = image.getGraphicsContext();

			// draw expected area on the left and virtual area on the right
			int padding = h;
			int x = g.getWidth() / 2 - w - padding;
			int y = g.getHeight() / 2 - h - padding;
			draw(g, x, y, w, h, 0x80, drawer);
			draw(ig, 0, 0, w, h, 0x80, drawer);

			// draw virtual area on destination
			Matrix m = new Matrix();
			m.setTranslate(x + w + padding, y);
			VectorGraphicsPainter.drawImage(g, image, m, 0xff);

			// draw without opacity in image but opacity in destination
			image.clear();
			draw(ig, 0, 0, w, h, 0xff, drawer);
			m.setTranslate(x, y + h + padding);
			VectorGraphicsPainter.drawImage(g, image, m, 0x80);

			// visualize the result
			display.flush();

			// compare areas
			TestUtilities.compareAreas(label, x, y, x + w + padding, y, w, h, tolerance);
			TestUtilities.compareAreas(label, x, y, x, y + h + padding, w, h, tolerance);
		}
	}

	private static void draw(GraphicsContext g, int x, int y, int w, int h, int alpha, AlphaDrawer drawer) {
		g.reset();
		drawer.draw(g, x, y, w, h, alpha);
	}

	private static Matrix getTranslatedMatrix(int x, int y) {
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		return matrix;
	}

	private static Path getPath(int w, int h) {
		Path path = new Path();
		path.lineTo(w, 0);
		path.lineTo(w, h);
		path.lineTo(0, h);
		path.close();
		return path;
	}

	private static VectorFont getFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf"); //$NON-NLS-1$
	}

	private static VectorImage getImage(String name) {
		return VectorImage.getImage("/com/microej/microvg/test/" + name + ".xml"); //$NON-NLS-1$//$NON-NLS-2$
	}
}
