/*
 * Copyright 2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.render;

import ej.mwt.Desktop;
import ej.mwt.Widget;

/**
 * A render policy is responsible of rendering the widgets of a desktop on the display.
 * <p>
 * A render policy is associated to a single desktop.
 * <p>
 * A render policy provides a method to render the desktop on the display. This method is used by the desktop in order
 * to perform the initial rendering of the complete widget hierarchy.
 * <p>
 * A render policy also provides a method to handle render requests targeting specific widgets of the desktop. The goal
 * of this method is to render only a part of the widget hierarchy considering that an old representation of the desktop
 * is already visible on the display. The most naive implementation would be to render the whole hierarchy of the
 * desktop.
 */
public abstract class RenderPolicy {

	private final Desktop desktop;

	/**
	 * Creates a render policy.
	 *
	 * @param desktop
	 *            the desktop to render.
	 */
	public RenderPolicy(Desktop desktop) {
		this.desktop = desktop;
	}

	/**
	 * Returns the desktop managed by this render policy.
	 *
	 * @return the desktop managed by this render policy.
	 */
	public Desktop getDesktop() {
		return this.desktop;
	}

	/**
	 * Renders the desktop on the display. This method is used by the desktop in order to perform the initial rendering
	 * of the complete widget hierarchy.
	 * <p>
	 * This method should be called in the MicroUI thread as the rendering of the widget is performed synchronously.
	 */
	public abstract void renderDesktop();

	/**
	 * Requests a rendering of the given widget on the display. The goal of this method is to render only a part of the
	 * widget hierarchy considering that an old representation of the desktop is already visible on the display.
	 * <p>
	 * This method returns immediately and the rendering of the widget is performed asynchronously in the MicroUI
	 * thread.
	 * <p>
	 * The given bounds are relative to the widget.
	 *
	 * @param widget
	 *            the widget to render.
	 * @param x
	 *            the x coordinate of the area to render.
	 * @param y
	 *            the y coordinate of the area to render.
	 * @param width
	 *            the width of the area to render.
	 * @param height
	 *            the height of the area to render.
	 */
	public abstract void requestRender(Widget widget, int x, int y, int width, int height);
}
