/*
 * Java
 *
 * Copyright 2012-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.event;

/**
 * This class provides some methods to create internal MicroUI events which target the MicroUI
 * <code>EventGenerator</code> event generator implementation.
 * <p>
 * Several methods takes in argument a string <code>generatorID</code>. This ID must refer to a MicroUI events generator
 * defined during the MicroEJ plaform build in <code>microui/microui.xml</code> file (in platform configuration
 * project).
 * <p>
 * In designer mode (plugin mode), this ID is unknown so all methods throw an {@link IllegalArgumentException}. The
 * caller should catch this exception and prints in the standard output stream a description of this event.
 * <p>
 * In MicroEJ application mode, this exception should not occur (the MicroUI events generator ID should exist). However,
 * when exception occurs, the trace allows to the user to <i>note</i> there is an issue according the MicroUI events
 * generator ID.
 */
@SuppressWarnings("nls")
public class LLUIInput {

	// MicroUI events representation
	private static final int EVENT_PUMP = 0x07000000; // 0x07 is MicroUI event ID for "native input event"
	private static final int EVENT_ID_MASK = 0xff;
	private static final int EVENT_ID_SHIFT = 16;
	private static final int EVENT_DATA_MASK = 0xffff;
	private static final int EVENT_DATA_SHIFT = 0;

	private LLUIInput() {
		// hide implicit constructor
	}

	/**
	 * Returns the type (i.e. the event generator id) of the event.
	 *
	 * @param event
	 *            the event to decode.
	 * @return the event's type.
	 */
	public static int getType(int event) {
		return (event >> EVENT_ID_SHIFT) & EVENT_ID_MASK;
	}

	/**
	 * Returns the data component of the event.
	 *
	 * @param event
	 *            the event to decode.
	 * @return the data component of the event.
	 */
	public static int getData(int event) {
		return (event >> EVENT_DATA_SHIFT) & EVENT_DATA_MASK;
	}

	/**
	 * Constructs and returns an event.
	 *
	 * @param generatorId
	 *            the required generator id (i.e. type) of the event.
	 * @param data
	 *            the data component of the event.
	 * @return the constructed event.
	 * @throws IllegalArgumentException
	 *             whether the generatorID refers to an unknown MicroUI generator (see class comment).
	 */
	public static int buildEvent(String generatorId, int data) {
		return buildEvent(getEventGeneratorID(generatorId), data);
	}

	/**
	 * Constructs and returns an event.
	 *
	 * @param generatorId
	 *            the required generator id (i.e. type) of the event.
	 * @param data
	 *            the data component of the event.
	 * @return the constructed event.
	 */
	public static int buildEvent(int generatorId, int data) {
		return EVENT_PUMP | ((generatorId & EVENT_ID_MASK) << EVENT_ID_SHIFT)
				| ((data & EVENT_DATA_MASK) << EVENT_DATA_SHIFT);
	}

	/**
	 * Sends an event in the queue. This event must have been build using {@link #buildEvent(int, int)} or
	 * {@link #buildEvent(String, int)}.
	 *
	 * @param event
	 *            the event to send.
	 */
	public static void sendEvent(int event) {
		getEventQueue().enqueudEvent(event);
	}

	/**
	 * Sends several events in the queue. The first event in the array must have been build using
	 * {@link #buildEvent(int, int)} or {@link #buildEvent(String, int)}.
	 *
	 * @param events
	 *            the events to send.
	 */
	public static void sendEvents(int[] events) {
		getEventQueue().enqueudEvents(events);
	}

	/**
	 * Constructs an event and sends it to the simulator input queue. Exactly equivalent to
	 * {@code sendEvent(buildEvent(generatorId, data))}.
	 *
	 * @param generatorId
	 *            the required generator id (i.e. type) of the event.
	 * @param data
	 *            the data component of the event.
	 * @throws IllegalArgumentException
	 *             whether the generatorID refers to an unknown MicroUI generator (see class comment).
	 */
	public static void sendEvent(String generatorId, int data) {
		sendEvent(buildEvent(generatorId, data));
	}

	/**
	 * Constructs an event and sends it to the simulator input queue. Exactly equivalent to
	 * {@code sendEvent(buildEvent(generatorId, data))}.
	 *
	 * @param generatorId
	 *            the required generator id (i.e. type) of the event.
	 * @param data
	 *            the data component of the event.
	 */
	public static void sendEvent(int generatorId, int data) {
		sendEvent(buildEvent(generatorId, data));
	}

	private static int getEventGeneratorID(String eventGenerator) {
		try {
			String eventGeneratorID = System
					.getProperty("com.is2t.microbsp.microui.io.eventgenerator." + eventGenerator);
			return Integer.parseInt(eventGeneratorID);
		} catch (Exception e) {
			// missing or invalid ID: not specified ID in designer mode, or error in MicroEJ application mode
			throw new IllegalArgumentException("MicroUI events generator ID '" + eventGenerator + "' is unknown.");
		}
	}

	private static EventQueue getEventQueue() {
		return EventQueue.get();
	}
}
