/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

/**
 * Represents a MicroUI GraphicsContext.
 * <p>
 * This interface is used by all drawing functions to target the destination. It can be mapped on byte array given as
 * parameter in MicroUI natives. This byte array is retrieved in MicroEJ application using the method
 * GraphicsContext.getSNIContext().
 *
 * @see LLUIDisplay#mapMicroUIGraphicsContext(byte[])
 * @see LLUIDisplay#newMicroUIGraphicsContext(byte[])
 */
public interface MicroUIGraphicsContext extends MicroUIImage {

	/**
	 * Returns the current MicroUI drawing color to use to perform drawings.
	 *
	 * @return a color in 0xAARRGGBB format (where alpha level is always 0xff == fully opaque).
	 */
	int getMicroUIColor();

	/**
	 * Sets the current MicroUI drawing color to use to perform drawings.
	 * <p>
	 * Beware that calling this method alters the color in the GraphicsContext instance manipulated by the Java user.
	 * The caller must save the color before modifying it and restore it afterwards.
	 *
	 * @param color
	 *            in 0xAARRGGBB format (where alpha level is always 0xff == fully opaque).
	 */
	void setMicroUIColor(int color);

	/**
	 * Returns the color to use to render the drawings. This color takes in consideration the embedded display pixel
	 * representation constraints.
	 * <p>
	 * Equivalent of <code>LLUIDisplay.Instance.convertARGBColorToColorToDraw(getMicroUIColor());</code>
	 *
	 * @return the color to use to render the drawings in 0xAARRGGBB format (where alpha level is always 0xff == fully
	 *         opaque).
	 * @see LLUIDisplay#convertARGBColorToColorToDraw(int)
	 */
	int getRenderingColor();

	/**
	 * @return top-left X coordinate of current clip.
	 */
	int getClipX1();

	/**
	 * @return top-left Y coordinate of current clip.
	 */
	int getClipY1();

	/**
	 * @return bottom-right X coordinate of current clip.
	 */
	int getClipX2();

	/**
	 * @return bottom-right Y coordinate of current clip.
	 */
	int getClipY2();

	/**
	 * Sets the clip.
	 * <p>
	 * Make sure that the clip is enabled for this call to be efficient.
	 * <p>
	 * Beware that calling this method alters the clip in the GraphicsContext instance manipulated by the Java user. The
	 * caller must save the clip before modifying it and restore it afterwards.
	 *
	 * @param x
	 *            top-left X coordinate of the clip.
	 * @param y
	 *            top-left Y coordinate of the clip.
	 * @param width
	 *            width of the clip.
	 * @param height
	 *            height of the clip.
	 *
	 * @see #configureClip(boolean)
	 * @see #isClipEnabled()
	 */
	void setClip(int x, int y, int width, int height);

	/**
	 * Combines the given clipping rectangle with the clipping area of this graphics context.
	 * <p>
	 * This method sets the clipping area to be the intersection of the given rectangle with the current clipping area.
	 * <p>
	 * Make sure that the clip is enabled for this call to be efficient.
	 * <p>
	 * Beware that calling this method alters the clip in the GraphicsContext instance manipulated by the Java user. The
	 * caller must save the clip before modifying it and restore it afterwards.
	 *
	 * @param x
	 *            top-left X coordinate of the clip.
	 * @param y
	 *            top-left Y coordinate of the clip.
	 * @param width
	 *            width of the clip.
	 * @param height
	 *            height of the clip.
	 *
	 * @see #configureClip(boolean)
	 * @see #isClipEnabled()
	 */
	void intersectClip(int x, int y, int width, int height);

	/**
	 * Enables or disables the graphics context clip. Useful to ignore clip during a drawing. The clip is automatically
	 * re-enabled when calling requestDrawing().
	 *
	 * @param enable
	 *            false to disable the clip.
	 */
	void configureClip(boolean enable);

	/**
	 * Tells if the clip is enabled or not. This call should be performed at the beginning of a drawing function in
	 * order to prevent to make some useless operations when the clip is disabled.
	 * <p>
	 * When it is disabled, that means the caller considers the drawing is fitting the clip. In this case checking the
	 * clip bounds is useless.
	 *
	 * @return true when the clip is enabled (the clip must be checked).
	 */
	boolean isClipEnabled();

	/**
	 * Tells if given point fits the clip or not.
	 *
	 * @param x
	 *            the pixel X coordinate.
	 * @param y
	 *            the pixel Y coordinate.
	 *
	 * @return false when the point is outside the clip.
	 */
	boolean isPointInClip(int x, int y);

	/**
	 * Tells if given horizontal line fully fits the clip or not.
	 *
	 * @param x1
	 *            the first pixel line X coordinate.
	 * @param x2
	 *            the last pixel line X coordinate.
	 * @param y
	 *            the both pixels line Y coordinate.
	 *
	 * @return false when the line is fully or partially outside the clip, true when line fully fits the clip (clip can
	 *         be disabled).
	 */
	boolean isHorizontalLineInClip(int x1, int x2, int y);

	/**
	 * Tells if given vertical line fully fits the clip or not.
	 *
	 * @param x
	 *            the both pixels line X coordinate.
	 * @param y1
	 *            the first pixel line Y coordinate.
	 * @param y2
	 *            the last pixel line Y coordinate.
	 *
	 * @return false when the line is fully or partially outside the clip, true when line fully fits the clip (clip can
	 *         be disabled).
	 */
	boolean isVerticalLineInClip(int y1, int y2, int x);

	/**
	 * Tells if given rectangle fully fits the clip or not.
	 *
	 * @param x1
	 *            the top-left pixel X coordinate.
	 * @param y1
	 *            the top-left pixel Y coordinate.
	 * @param x2
	 *            the bottom-right pixel X coordinate.
	 * @param y2
	 *            the top-right pixel Y coordinate.
	 *
	 * @return false when the rectangle is fully or partially outside the clip, true when rectangle fully fits the clip
	 *         (clip can be disabled).
	 */
	boolean isRectangleInClip(int x1, int y1, int x2, int y2);

	/**
	 * Tells if given region (from x,y to x+width-1,y+height-1) fully fits the clip or not.
	 *
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the region width.
	 * @param height
	 *            the region height.
	 *
	 * @return false when the region is fully or partially outside the clip, true when region fully fits the clip (clip
	 *         can be disabled).
	 */
	boolean isRegionInClip(int x, int y, int width, int height);

	/**
	 * Tells if source and destination share a region.
	 *
	 * Source and destination can share a region if and only if source and destination target the same MicroUI Image. In
	 * that case, this function checks if there is an intersection of source and destination regions. This function is
	 * useful when drawing an image on same image.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target of destination.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to check.
	 * @param regionY
	 *            the y coordinate of the upper-left corner of the region to check.
	 * @param width
	 *            the width of the region to check.
	 * @param height
	 *            the height of the region to check.
	 * @param destX
	 *            the x coordinate of the top-left point in the destination.
	 * @param destY
	 *            the y coordinate of the top-left point in the destination.
	 *
	 * @return true when source and destination are same image and when destination region intersects source region.
	 */
	boolean regionsOverlap(MicroUIImage img, int regionX, int regionY, int width, int height, int destX, int destY);

	/**
	 * Tells if the ellipsis is enabled or not. Returns 0 when ellipsis is disabled, a positive value otherwise. This
	 * value is the maximum string width in pixels. If string width is higher than this value, an ellipsis will be drawn
	 * to crop the string.
	 *
	 * @return the ellipsis width or 0.
	 */
	int getEllipsisWidth();

	/**
	 * Gets and clears the drawing log flags from a graphics context.
	 *
	 * The flags are reset to their success value after retrieving their value.
	 *
	 * @param sd
	 *            the graphics context
	 * @return the drawing log flags
	 */
	int getAndClearDrawingLogFlags();

	/**
	 * Reports non-critical incidents occurring during a drawing operation.
	 *
	 * This will set flags into the drawing log flags.
	 *
	 * @param context
	 *            the graphics context in which the incidents occurred
	 * @param flags
	 *            the flags to set
	 */
	void reportWarning(int flags);

	/**
	 * Reports errors occurring during a drawing operation.
	 *
	 * This will set flags into the drawing log flags. This will additionally set the DRAWING_LOG_ERROR flag, causing an
	 * exception to be thrown in the application when checking the flags.
	 *
	 * @param context
	 *            the graphics context in which the errors occurred
	 * @param flags
	 *            the flags to set
	 */
	void reportError(int flags);

	/**
	 * Requests a drawing on the graphics context.
	 *
	 * @return true if drawing can be performed, false when the drawing does not need to be done for any reason.
	 */
	boolean requestDrawing();
}
