/*
 * Copyright (c) 1996, 2015, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2017-2022 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. The Copyright holder(s) designates
 * this particular file as subject to the "Classpath" exception as provided
 * by the Copyright holder(s) in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact the Copyright holder(s) if you need additional information
 * or have any questions.
 * (if Oracle: 500 Oracle Parkway, Redwood Shores, CA 94065 USA or visit www.oracle.com)
 */

package java.security;

import java.security.cert.Certificate;
import java.security.spec.AlgorithmParameterSpec;

/**
 * The Signature class is used to provide applications the functionality of a digital signature algorithm. Digital
 * signatures are used for authentication and integrity assurance of digital data.
 *
 * <p>
 * The signature algorithm can be, among others, the NIST standard DSA, using DSA and SHA-1. The DSA algorithm using the
 * SHA-1 message digest algorithm can be specified as {@code SHA1withDSA}. In the case of RSA, there are multiple
 * choices for the message digest algorithm, so the signing algorithm could be specified as, for example,
 * {@code MD2withRSA}, {@code MD5withRSA}, or {@code SHA1withRSA}. The algorithm name must be specified, as there is no
 * default.
 *
 * <p>
 * A Signature object can be used to generate and verify digital signatures.
 *
 * <p>
 * There are three phases to the use of a Signature object for either signing data or verifying a signature:
 * <ol>
 *
 * <li>Initialization, with either
 *
 * <ul>
 *
 * <li>a public key, which initializes the signature for verification (see {@link #initVerify(PublicKey) initVerify}),
 * or
 *
 * <li>a private key (and optionally a Secure Random Number Generator), which initializes the signature for signing (see
 * {@link #initSign(PrivateKey)} and {@link #initSign(PrivateKey, SecureRandom)}).
 *
 * </ul>
 *
 * <li>Updating
 *
 * <p>
 * Depending on the type of initialization, this will update the bytes to be signed or verified. See the
 * {@link #update(byte) update} methods.
 *
 * <li>Signing or Verifying a signature on all updated bytes. See the {@link #sign() sign} methods and the
 * {@link #verify(byte[]) verify} method.
 *
 * </ol>
 *
 * <p>
 * Note that this class is abstract and extends from {@code SignatureSpi} for historical reasons. Application developers
 * should only take notice of the methods defined in this {@code Signature} class; all the methods in the superclass are
 * intended for cryptographic service providers who wish to supply their own implementations of digital signature
 * algorithms.
 *
 * <p>
 * Every implementation of the Java platform is required to support the following standard {@code Signature} algorithms:
 * <ul>
 * <li>{@code SHA1withDSA}</li>
 * <li>{@code SHA1withRSA}</li>
 * <li>{@code SHA256withRSA}</li>
 * </ul>
 * These algorithms are described in the
 * <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#Signature"> Signature section</a> of the Java
 * Cryptography Architecture Standard Algorithm Name Documentation. Consult the release documentation for your
 * implementation to see if any other algorithms are supported.
 *
 * @author Benjamin Renaud
 *
 */

public abstract class Signature extends SignatureSpi {

	/**
	 * Possible {@link #state} value, signifying that this signature object has not yet been initialized.
	 */
	protected final static int UNINITIALIZED = 0;

	/**
	 * Possible {@link #state} value, signifying that this signature object has been initialized for signing.
	 */
	protected final static int SIGN = 2;

	/**
	 * Possible {@link #state} value, signifying that this signature object has been initialized for verification.
	 */
	protected final static int VERIFY = 3;

	/**
	 * Current state of this signature object.
	 */
	protected int state = UNINITIALIZED;

	/**
	 * Creates a Signature object for the specified algorithm.
	 *
	 * @param algorithm
	 *            the standard string name of the algorithm. See the Signature section in the
	 *            <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#Signature"> Java Cryptography
	 *            Architecture Standard Algorithm Name Documentation</a> for information about standard algorithm names.
	 */
	protected Signature(String algorithm) {
		throw new RuntimeException();
	}

	/**
	 * Returns a Signature object that implements the specified signature algorithm.
	 *
	 * <p>
	 * This method traverses the list of registered security Providers, starting with the most preferred Provider. A new
	 * Signature object encapsulating the SignatureSpi implementation from the first Provider that supports the
	 * specified algorithm is returned.
	 *
	 * @param algorithm
	 *            the standard name of the algorithm requested. See the Signature section in the
	 *            <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#Signature"> Java Cryptography
	 *            Architecture Standard Algorithm Name Documentation</a> for information about standard algorithm names.
	 *
	 * @return the new Signature object.
	 *
	 * @exception NoSuchAlgorithmException
	 *                if no Provider supports a Signature implementation for the specified algorithm.
	 */
	public static Signature getInstance(String algorithm) throws NoSuchAlgorithmException {
		throw new RuntimeException();
	}

	/**
	 * Returns a Signature object that implements the specified signature algorithm.
	 *
	 * <p>
	 * A new Signature object encapsulating the SignatureSpi implementation from the specified provider is returned. The
	 * specified provider must be registered in the security provider list.
	 *
	 * @param algorithm
	 *            the name of the algorithm requested. See the Signature section in the
	 *            <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#Signature"> Java Cryptography
	 *            Architecture Standard Algorithm Name Documentation</a> for information about standard algorithm names.
	 *
	 * @param provider
	 *            the name of the provider.
	 *
	 * @return the new Signature object.
	 *
	 * @exception NoSuchAlgorithmException
	 *                if a SignatureSpi implementation for the specified algorithm is not available from the specified
	 *                provider.
	 *
	 * @exception NoSuchProviderException
	 *                if the specified provider is not registered in the security provider list.
	 *
	 * @exception IllegalArgumentException
	 *                if the provider name is null or empty.
	 */
	public static Signature getInstance(String algorithm, String provider)
			throws NoSuchAlgorithmException, NoSuchProviderException {
		throw new RuntimeException();
	}

	/**
	 * Initializes this object for verification. If this method is called again with a different argument, it negates
	 * the effect of this call.
	 *
	 * @param publicKey
	 *            the public key of the identity whose signature is going to be verified.
	 *
	 * @exception InvalidKeyException
	 *                if the key is invalid.
	 */
	public final void initVerify(PublicKey publicKey) throws InvalidKeyException {
		throw new RuntimeException();
	}

	/**
	 * Initializes this object for verification, using the public key from the given certificate.
	 * <p>
	 * If the certificate is of type X.509 and has a <i>key usage</i> extension field marked as critical, and the value
	 * of the <i>key usage</i> extension field implies that the public key in the certificate and its corresponding
	 * private key are not supposed to be used for digital signatures, an {@code InvalidKeyException} is thrown.
	 *
	 * @param certificate
	 *            the certificate of the identity whose signature is going to be verified.
	 *
	 * @exception InvalidKeyException
	 *                if the public key in the certificate is not encoded properly or does not include required
	 *                parameter information or cannot be used for digital signature purposes.
	 * @since 1.3
	 */
	public final void initVerify(Certificate certificate) throws InvalidKeyException {
		throw new RuntimeException();
	}

	/**
	 * Initialize this object for signing. If this method is called again with a different argument, it negates the
	 * effect of this call.
	 *
	 * @param privateKey
	 *            the private key of the identity whose signature is going to be generated.
	 *
	 * @exception InvalidKeyException
	 *                if the key is invalid.
	 */
	public final void initSign(PrivateKey privateKey) throws InvalidKeyException {
		throw new RuntimeException();
	}

	/**
	 * Initialize this object for signing. If this method is called again with a different argument, it negates the
	 * effect of this call.
	 *
	 * @param privateKey
	 *            the private key of the identity whose signature is going to be generated.
	 *
	 * @param random
	 *            the source of randomness for this signature.
	 *
	 * @exception InvalidKeyException
	 *                if the key is invalid.
	 */
	public final void initSign(PrivateKey privateKey, SecureRandom random) throws InvalidKeyException {
		throw new RuntimeException();
	}

	/**
	 * Returns the signature bytes of all the data updated. The format of the signature depends on the underlying
	 * signature scheme.
	 *
	 * <p>
	 * A call to this method resets this signature object to the state it was in when previously initialized for signing
	 * via a call to {@code initSign(PrivateKey)}. That is, the object is reset and available to generate another
	 * signature from the same signer, if desired, via new calls to {@code update} and {@code sign}.
	 *
	 * @return the signature bytes of the signing operation's result.
	 *
	 * @exception SignatureException
	 *                if this signature object is not initialized properly or if this signature algorithm is unable to
	 *                process the input data provided.
	 */
	public final byte[] sign() throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Finishes the signature operation and stores the resulting signature bytes in the provided buffer {@code outbuf},
	 * starting at {@code offset}. The format of the signature depends on the underlying signature scheme.
	 *
	 * <p>
	 * This signature object is reset to its initial state (the state it was in after a call to one of the
	 * {@code initSign} methods) and can be reused to generate further signatures with the same private key.
	 *
	 * @param outbuf
	 *            buffer for the signature result.
	 *
	 * @param offset
	 *            offset into {@code outbuf} where the signature is stored.
	 *
	 * @param len
	 *            number of bytes within {@code outbuf} allotted for the signature.
	 *
	 * @return the number of bytes placed into {@code outbuf}.
	 *
	 * @exception SignatureException
	 *                if this signature object is not initialized properly, if this signature algorithm is unable to
	 *                process the input data provided, or if {@code len} is less than the actual signature length.
	 *
	 * @since 1.2
	 */
	public final int sign(byte[] outbuf, int offset, int len) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Verifies the passed-in signature.
	 *
	 * <p>
	 * A call to this method resets this signature object to the state it was in when previously initialized for
	 * verification via a call to {@code initVerify(PublicKey)}. That is, the object is reset and available to verify
	 * another signature from the identity whose public key was specified in the call to {@code initVerify}.
	 *
	 * @param signature
	 *            the signature bytes to be verified.
	 *
	 * @return true if the signature was verified, false if not.
	 *
	 * @exception SignatureException
	 *                if this signature object is not initialized properly, the passed-in signature is improperly
	 *                encoded or of the wrong type, if this signature algorithm is unable to process the input data
	 *                provided, etc.
	 */
	public final boolean verify(byte[] signature) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Verifies the passed-in signature in the specified array of bytes, starting at the specified offset.
	 *
	 * <p>
	 * A call to this method resets this signature object to the state it was in when previously initialized for
	 * verification via a call to {@code initVerify(PublicKey)}. That is, the object is reset and available to verify
	 * another signature from the identity whose public key was specified in the call to {@code initVerify}.
	 *
	 *
	 * @param signature
	 *            the signature bytes to be verified.
	 * @param offset
	 *            the offset to start from in the array of bytes.
	 * @param length
	 *            the number of bytes to use, starting at offset.
	 *
	 * @return true if the signature was verified, false if not.
	 *
	 * @exception SignatureException
	 *                if this signature object is not initialized properly, the passed-in signature is improperly
	 *                encoded or of the wrong type, if this signature algorithm is unable to process the input data
	 *                provided, etc.
	 * @exception IllegalArgumentException
	 *                if the {@code signature} byte array is null, or the {@code offset} or {@code length} is less than
	 *                0, or the sum of the {@code offset} and {@code length} is greater than the length of the
	 *                {@code signature} byte array.
	 * @since 1.4
	 */
	public final boolean verify(byte[] signature, int offset, int length) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Updates the data to be signed or verified by a byte.
	 *
	 * @param b
	 *            the byte to use for the update.
	 *
	 * @exception SignatureException
	 *                if this signature object is not initialized properly.
	 */
	public final void update(byte b) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Updates the data to be signed or verified, using the specified array of bytes.
	 *
	 * @param data
	 *            the byte array to use for the update.
	 *
	 * @exception SignatureException
	 *                if this signature object is not initialized properly.
	 */
	public final void update(byte[] data) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Updates the data to be signed or verified, using the specified array of bytes, starting at the specified offset.
	 *
	 * @param data
	 *            the array of bytes.
	 * @param off
	 *            the offset to start from in the array of bytes.
	 * @param len
	 *            the number of bytes to use, starting at offset.
	 *
	 * @exception SignatureException
	 *                if this signature object is not initialized properly.
	 */
	public final void update(byte[] data, int off, int len) throws SignatureException {
		throw new RuntimeException();
	}

	/**
	 * Returns the name of the algorithm for this signature object.
	 *
	 * @return the name of the algorithm for this signature object.
	 */
	public final String getAlgorithm() {
		throw new RuntimeException();
	}

	/**
	 * Returns a string representation of this signature object, providing information that includes the state of the
	 * object and the name of the algorithm used.
	 *
	 * @return a string representation of this signature object.
	 */
	@Override
	public String toString() {
		throw new RuntimeException();
	}

	/**
	 * Initializes this signature engine with the specified parameter set.
	 *
	 * @param params
	 *            the parameters
	 *
	 * @exception InvalidAlgorithmParameterException
	 *                if the given parameters are inappropriate for this signature engine
	 */
	public final void setParameter(AlgorithmParameterSpec params) throws InvalidAlgorithmParameterException {
		throw new RuntimeException();
	}

	/**
	 * Returns a clone if the implementation is cloneable.
	 *
	 * @return a clone if the implementation is cloneable.
	 *
	 * @exception CloneNotSupportedException
	 *                if this is called on an implementation that does not support {@code Cloneable}.
	 */
	@Override
	public Object clone() throws CloneNotSupportedException {
		if (this instanceof Cloneable) {
			return super.clone();
		} else {
			throw new CloneNotSupportedException();
		}
	}
}
