/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import ej.bon.Constants;
import ej.microui.MicroUI;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Image;
import ej.microui.display.Painter;
import ej.microui.display.ResourceImage;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Provides utility methods for tests.
 */
/* package */ class TestUtilities {

	/* package */ static final String DEBUG_CONSTANT = "com.microej.microvg.test.debug";
	private static final int MAX_COLOR_COMPONENT_VALUE = 0xff;
	private static final int ALPHA_SHIFT = 24;
	private static final int RED_SHIFT = 16;
	private static final int GREEN_SHIFT = 8;
	private static final int BLUE_SHIFT = 0;
	private static final int RED_MASK = MAX_COLOR_COMPONENT_VALUE << RED_SHIFT;
	private static final int GREEN_MASK = MAX_COLOR_COMPONENT_VALUE << GREEN_SHIFT;
	private static final int BLUE_MASK = MAX_COLOR_COMPONENT_VALUE << BLUE_SHIFT;
	private static final int MAX_OPAQUE_COLOR_VALUE = 0xffffff;

	private TestUtilities() {
		// Prevent instantiation.
	}

	/**
	 * Checks that the pixel at given (x,y) coordinates has the expected color.
	 *
	 * <p>
	 * This method compares two colors component-by-component (R,G,B). A tolerance of 8 is used when comparing
	 * respective components (see {@link #compareComponent(int, int)}).
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x-coordinate of the pixel to test.
	 * @param y
	 *            the y-coordinate of the pixel to test.
	 * @param g
	 *            the graphics context.
	 * @param expectedColor
	 *            the color to test the display color against.
	 */
	/* package */ static void check(String label, int x, int y, GraphicsContext g, int expectedColor) {
		int displayColor = Display.getDisplay().getDisplayColor(expectedColor);
		int readColor = g.readPixel(x, y);
		if (!compare(displayColor, readColor)) {
			System.out.println("TestUtilities.check() 0x" + Integer.toHexString(displayColor) + " 0x"
					+ Integer.toHexString(readColor));
		}

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, x, y, 3, 3);
			Display.getDisplay().flush();
		}
		assertTrue(label, compare(displayColor, readColor));
	}

	/**
	 * Checks that the pixel at given (x,y) coordinates has not the specified color.
	 *
	 * <p>
	 * This method compares two colors component-by-component (R,G,B). A tolerance of 8 is used when comparing
	 * respective components (see {@link #compareComponent(int, int)}).
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x-coordinate of the pixel to test.
	 * @param y
	 *            the y-coordinate of the pixel to test.
	 * @param g
	 *            the graphics context.
	 * @param expectedColor
	 *            the color to test the display color against.
	 */
	/* package */ static void checkNot(String label, int x, int y, GraphicsContext g, int expectedColor) {
		int displayColor = Display.getDisplay().getDisplayColor(expectedColor);
		int readColor = g.readPixel(x, y);
		assertFalse(label, compare(displayColor, readColor));
	}

	/**
	 * Checks that all the pixels in the specified area have the expected color.
	 *
	 * <p>
	 * The padding is the amount of space inside the specified area (along the border) to be excluded from the test.
	 * This can be convenient to avoid anti-aliasing issues on the edges when testing pixels.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param color
	 *            the expected color
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 * @param padding
	 *            the space inside the area, along the border of this area, to be excluded from the test.
	 */
	/* package */ static void checkArea(String label, int color, int x, int y, int width, int height, int padding) {
		checkArea(label, color, x + padding, y + padding, width - 2 * padding, height - 2 * padding);
	}

	/**
	 * Checks that all the pixels in the specified area have the expected color.
	 *
	 * <p>
	 * This method uses a padding of 0, that means that no pixels are excluded from the test in the specified area.
	 * Equivalent to calling {@link #checkArea(String, int, int, int, int, int, int)} with padding equals 0.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param color
	 *            the expected color
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 */
	/* package */ static void checkArea(String label, int color, int x, int y, int width, int height) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		int expectedColor = display.getDisplayColor(color);
		boolean result = true;
		int translationX = g.getTranslationX();
		int translationY = g.getTranslationY();

		for (int i = 0; i < width; i++) {
			int xi = x + i;
			if ((xi + translationX < 0) || (xi + translationX >= display.getWidth())) {
				continue;
			}
			for (int j = 0; j < height; j++) {
				int yj = y + j;
				if ((yj + translationY < 0) || (yj + translationY >= display.getHeight())) {
					continue;
				}

				int readColor = g.readPixel(xi, yj) & MAX_OPAQUE_COLOR_VALUE;
				boolean compare = compare(expectedColor, readColor);
				if (!compare && Constants.getBoolean(DEBUG_CONSTANT)) {
					System.out.println("TestUtilities.checkArea() 0x" + Integer.toHexString(expectedColor) + " 0x"
							+ Integer.toHexString(readColor) + " x=" + xi + " y=" + yj);
				}
				result &= compare;
			}
		}

		if (Constants.getBoolean(DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, x, y, width, height);
			display.flush();
		}

		assertTrue(label, result);
	}

	/**
	 * Checks the expected color in the surrounding periphery of the specified area.
	 *
	 * <p>
	 * The periphery is made of 4 rectangular areas located on the edges of the target area: top, bottom, left and
	 * right. The specified thickness defines the size of these areas.
	 *
	 * <p>
	 * The padding is the amount of space inside the test areas (along the border) to be excluded from the test. This
	 * can be convenient to avoid anti-aliasing issues on the edges when testing pixels.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param color
	 *            the expected color in the peripheral area
	 * @param x
	 *            the x origin of the target area (top left corner)
	 * @param y
	 *            the y origin of the target area (top left corner)
	 * @param width
	 *            the width of the target area
	 * @param height
	 *            the height of the target area
	 * @param thickness
	 *            the thickness of the peripheral area (i.e., test area)
	 * @param padding
	 *            the padding of the peripheral area (i.e., test area)
	 */
	/* package */ static int checkPeripheralArea(String label, int color, int x, int y, int width, int height,
			int thickness, int padding, boolean assertOnError) {
		int errors = 0;
		try {
			checkArea(label + " (outside top)", color, x, y - thickness, width, thickness, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}
		try {
			checkArea(label + " (outside bottom)", color, x, y + height, width, thickness, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}
		try {
			checkArea(label + " (outside left)", color, x - thickness, y, thickness, height, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}
		try {
			checkArea(label + " (outside right)", color, x + width, y, thickness, height, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}

		if (assertOnError) {
			assertEquals(0, errors);
		}

		return errors;
	}

	static void checkPeripheralArea(String label, int color, int x, int y, int width, int height, int thickness,
			int padding) {
		checkPeripheralArea(label, color, x, y, width, height, thickness, padding, true);
	}

	/* package */ static boolean compare(int displayColor, int readColor) {
		return compareComponent(getRed(displayColor), getRed(readColor))
				&& compareComponent(getGreen(displayColor), getGreen(readColor))
				&& compareComponent(getBlue(displayColor), getBlue(readColor));
	}

	/* package */ static boolean compareComponent(int displayComponent, int readComponent) {
		int diff = Math.abs(displayComponent - readComponent);
		return diff <= 0x08;
	}

	/**
	 * Gets a color red component.
	 *
	 * @param color
	 *            the color.
	 * @return the red component.
	 */
	/* package */ static int getRed(int color) {
		return (color & RED_MASK) >>> RED_SHIFT;
	}

	/**
	 * Gets a color green component.
	 *
	 * @param color
	 *            the color.
	 * @return the green component.
	 */
	/* package */ static int getGreen(int color) {
		return (color & GREEN_MASK) >>> GREEN_SHIFT;
	}

	/**
	 * Gets a color blue component.
	 *
	 * @param color
	 *            the color.
	 * @return the blue component.
	 */
	/* package */ static int getBlue(int color) {
		return (color & BLUE_MASK) >>> BLUE_SHIFT;
	}

	/**
	 * Gets a color with specified alpha applied.
	 *
	 * @param color
	 *            the color.
	 * @param alpha
	 *            the alpha, between {@link GraphicsContext#TRANSPARENT} and {@link GraphicsContext#OPAQUE}.
	 * @return the color with alpha.
	 */
	/* package */ static int getColorWithAlpha(int color, int alpha) {
		return (color & MAX_OPAQUE_COLOR_VALUE) | (alpha << ALPHA_SHIFT);
	}

	/* package */ static int blendColors(int color, int background, int alpha) {
		int red = (TestUtilities.getRed(color) * alpha
				+ TestUtilities.getRed(background) * (MAX_COLOR_COMPONENT_VALUE - alpha)) / MAX_COLOR_COMPONENT_VALUE;
		int green = (TestUtilities.getGreen(color) * alpha
				+ TestUtilities.getGreen(background) * (MAX_COLOR_COMPONENT_VALUE - alpha)) / MAX_COLOR_COMPONENT_VALUE;
		int blue = (TestUtilities.getBlue(color) * alpha
				+ TestUtilities.getBlue(background) * (MAX_COLOR_COMPONENT_VALUE - alpha)) / MAX_COLOR_COMPONENT_VALUE;

		return (red << RED_SHIFT) | (green << GREEN_SHIFT) | blue;
	}

	/**
	 * Clears the screen with black.
	 */
	/* package */ static void clearScreen() {
		clearScreen(Colors.BLACK);
	}

	/**
	 * Clears the screen with the specified color.
	 */
	/* package */ static void clearScreen(int color) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.setColor(color);
		Painter.fillRectangle(g, -g.getTranslationX(), -g.getTranslationY(), display.getWidth(), display.getHeight());
	}

	/* package */ static VectorFont getTestFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf");
	}

	/* package */ static void drawStringOnCircleGrid(VectorFont font, float fontSize, float radius, int x, int y,
			Direction direction) {

		// Clockwise direction
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		float bottomCircleRadius = topCircleRadius - font.getHeight(fontSize);

		// If counter-Clockwise direction
		if (direction == Direction.COUNTER_CLOCKWISE) {
			topCircleRadius = radius - font.getBaselinePosition(fontSize);
			bottomCircleRadius = topCircleRadius + fontSize;
		}

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.setColor(Colors.GREEN);
		Painter.drawCircle(g, (int) (x - topCircleRadius), (int) (y - topCircleRadius), (int) (topCircleRadius * 2));

		g.setColor(Colors.LIME);
		Painter.drawCircle(g, (int) (x - bottomCircleRadius), (int) (y - bottomCircleRadius),
				(int) (bottomCircleRadius * 2));

		g.setColor(Colors.YELLOW);
		Painter.drawCircle(g, (int) (x - radius), (int) (y - radius), (int) (radius * 2));

		g.setColor(Colors.RED);
		Painter.drawLine(g, x, y - 10, x, y + 10);
		Painter.drawLine(g, x - 10, y, x + 10, y);

		display.flush();
	}

	/**
	 * Return whether the testsuite is running on simulator.
	 *
	 * On Android, the constant `com.microej.library.microui.onS3` is set to true to use the Platform simulation mode
	 * but it is not considered as the simulator.
	 *
	 * @return whether the testsuite is running on simulator.
	 */
	static public boolean isOnSimulator() {
		return Constants.getBoolean("com.microej.library.microui.onS3") && !isOnAndroid();
	}

	/**
	 * Return whether the testsuite is running on Android.
	 *
	 * @return whether the testsuite is running on Android.
	 */
	public static boolean isOnAndroid() {
		return Constants.getBoolean("com.microej.microvg.test.android");
	}

	/*
	 * Measure character bbox width
	 *
	 * A character box is splitted in 3 parts a|b|c
	 *
	 * a is the padding space between the start of the bbox and the start of the glyph b is the glyph c is the padding
	 * space between the end of the glyph and the end of the bbox
	 *
	 * font.measureStringWidth of a 1 character string returns "b". If the string as 2 identical characters it returns:
	 * "b + c + a + b = 2b + a + c"
	 *
	 * Thus the bbox width is equal to (measureStringWidth(2*character) - measureStringWidth(character)
	 *
	 */
	static public float measureCharWidth(String s, VectorFont font, float fontSize) {
		return (font.measureStringWidth(s + s, fontSize) - font.measureStringWidth(s, fontSize));
	}

	static public void compareDisplay(String filename, int backgroundColor, int x, int y, int width, int height,
			float maxDiff) {
		GraphicsContext g = Display.getDisplay().getGraphicsContext();

		// flushDisplayNative(g.getSNIContext(), filename.toCharArray(), x, y, width, height);

		String imagePath = "/images/" + filename + ".png";
		Image ref = isOnAndroid() ? ResourceImage.loadImage(imagePath) : Image.getImage(imagePath);

		int widthRef = ref.getWidth();
		int heightRef = ref.getHeight();

		if (widthRef != width || heightRef != height) {
			System.err.println("Incorrect screen size: screenshot = " + widthRef + "x" + heightRef + ", screen = "
					+ width + "x" + height);
		}

		int incorrectPixels = 0;
		int nbPixel = 0;
		for (int i = 0; i < widthRef; i++) {
			for (int j = 0; j < heightRef; j++) {
				int k = x + i;
				int l = y + j;

				int colorRef = ref.readPixel(i, j);

				int colorDisp = g.readPixel(k, l);

				if (((colorRef & MAX_OPAQUE_COLOR_VALUE) == backgroundColor)
						&& ((colorDisp & MAX_OPAQUE_COLOR_VALUE) == backgroundColor)) {
					continue;
				}

				int pixelDiff = 0;
				pixelDiff += Math.abs(getRed(colorRef) - getRed(colorDisp));
				pixelDiff += Math.abs(getGreen(colorRef) - getGreen(colorDisp));
				pixelDiff += Math.abs(getBlue(colorRef) - getBlue(colorDisp));

				incorrectPixels += pixelDiff;
				nbPixel += 1;
			}
		}

		double percentIncorrectPixels = (double) incorrectPixels / (nbPixel * 255 * 3);

		String label = "compareDisplay_" + filename + " - percentIncorrectPixels=" + percentIncorrectPixels;
		System.out.println(label);
		assertFalse(label, percentIncorrectPixels > maxDiff);
	}

	/**
	 * Returns the correct reference image name for the simulator or embedded implementation.
	 *
	 * This is needed as simulator and embedded implementation don't provide the same rendering for complex layout
	 * tests.
	 *
	 * @param name
	 *            prefix of imageRefName
	 * @return image name
	 */
	static public String getImageContextualName(String name) {
		if (TestUtilities.isOnSimulator()) {
			return name + "_sim";
		} else {
			return name + "_emb";
		}
	}

	/**
	 * Compares both matrices and throws an error when different or equal.
	 *
	 * @param a
	 *            the first matrix
	 * @param b
	 *            the second matrix
	 * @param errorIfDifferent
	 *            true to throw an error when different, false to throw an error when equal.
	 */
	public static void checkMatrix(Matrix a, Matrix b, boolean errorIfDifferent) {
		float[] ma = a.getSNIContext();
		float[] mb = b.getSNIContext();
		checkMatrix(ma, mb, errorIfDifferent);
	}

	/**
	 * Compares both matrices and throws an error when different or equal. Both matrices must have the same size.
	 *
	 * @param a
	 *            the first matrix
	 * @param b
	 *            the second matrix
	 * @param errorIfDifferent
	 *            true to throw an error when different, false to throw an error when equal.
	 */
	public static void checkMatrix(float[] a, float[] b, boolean errorIfDifferent) {

		assertEquals("matrices sizes", a.length, b.length);

		boolean same = true;
		for (int i = 0; i < a.length; i++) {
			if (a[i] != b[i]) {
				same = false;
			}
		}

		if (errorIfDifferent) {
			assertTrue("matrices compare", same);
		} else {
			assertFalse("matrices are identical", same);
		}
	}

	// static native void flushDisplayNative(byte[] gcArray, char[] filename, int x, int y, int width, int height);

	static public void startMicroUI() {
		if (!MicroUI.isStarted() || !isOnAndroid()) {
			MicroUI.start();
		} else {
			// Reset graphics context
			GraphicsContext g = Display.getDisplay().getGraphicsContext();
			g.reset();
		}
	}

	static public void stopMicroUI() {
		if (!isOnAndroid()) {
			MicroUI.stop();
		}
	}

	/**
	 * Compares all the pixels in the specified areas. Both areas must fit the display area (with or without a
	 * translation).
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param ex
	 *            the x origin of the expected area (top-left)
	 * @param ey
	 *            the y origin of the expected area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 */
	/* package */ static void compareAreas(String label, int x, int y, int ex, int ey, int width, int height) {
		compareAreas(label, x, y, ex, ey, width, height, 0);
	}

	/**
	 * Compares all the pixels in the specified areas. Both areas must fit the display area (with or without a
	 * translation). A tolerance (a percentage less than 10%) is allowed.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param ex
	 *            the x origin of the expected area (top-left)
	 * @param ey
	 *            the y origin of the expected area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 * @param tolerance
	 *            maximum number of pixels (in percantage) that can be different (max 0.1 for 10%)
	 */
	/* package */ static void compareAreas(String label, int x, int y, int ex, int ey, int width, int height,
			float tolerance) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		boolean result = true;
		int counter = 0;

		try {

			for (int i = 0; i < width; i++) {
				for (int j = 0; j < height; j++) {
					int readColor = g.readPixel(x + i, y + j);
					int expectedColor = g.readPixel(ex + i, ey + j);
					boolean compare = TestUtilities.compare(expectedColor, readColor);
					if (!compare) {
						++counter;
						if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
							System.out.print("TestUtilities2.compareArea() at ");
							System.out.print(i);
							System.out.print(",");
							System.out.print(j);
							System.out.println(" 0x" + Integer.toHexString(expectedColor) + " 0x"
									+ Integer.toHexString(readColor));
						}
					}
					result &= compare;
				}
			}
		} catch (Exception e) {
			// out of bounds
			result = false;
		}

		float tr = ((float) counter) / (width * height);
		if (tr > 0) {
			System.out.println("Invalid pixels ratio: " + tr);
			assertTrue(label + " tolerance", tr < tolerance);
		}

		if (tolerance > 0) {
			assertTrue(label + " to big tolerance (higher than 0.1)", tolerance <= 0.1f); // 10%
		} else {
			assertTrue(label, result);
		}
	}
}
