/*
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.wear.services;

/**
 * This service allows Features to get time information.
 */
public interface TimeService {

	/**
	 * Returns the current time.
	 *
	 * @return the current time (in milliseconds, measured from the Java Epoch)
	 */
	long getCurrentTime();

	/**
	 * Returns the current offset from UTC in seconds.
	 *
	 * <p>
	 * The offset reflects the time zone and daylight saving time, applicable at the current instant.
	 *
	 * @return the current offset from UTC in seconds
	 */

	int getTimeZoneOffset();

	/**
	 * Returns the current time zone ID.
	 *
	 * <p>
	 * The zone ID is a string that represents either a region-based time zone or a fixed offset from UTC.
	 *
	 * @return the current time zone ID
	 */
	String getTimeZoneId();

	/**
	 * Sets the time configuration to use.
	 * 
	 * @param currentTime
	 *            the new time (in milliseconds, measured from the Java Epoch)
	 * @param zoneId
	 *            the new time zone ID
	 * @param zoneOffset
	 *            the new offset from UTC
	 */
	void setTimeConfiguration(long currentTime, String zoneId, int zoneOffset);

	/**
	 * Adds the given {@link ConfigurationListener} to the list of listeners that are notified when the time
	 * configuration of the device has changed.
	 *
	 * @param listener
	 *            the new listener to add
	 */
	void addConfigurationListener(ConfigurationListener listener);

	/**
	 * Removes the given {@link ConfigurationListener} to the list of listeners that are notified when the time
	 * configuration of the device has changed.
	 *
	 * <p>
	 * Does nothing if the listener is not registered.
	 *
	 * @param listener
	 *            the listener to be removed
	 */
	void removeConfigurationListener(ConfigurationListener listener);

	/**
	 * The listener interface for receiving notifications when the time configuration of the device has changed (date,
	 * time, time zone).
	 */
	interface ConfigurationListener {

		/**
		 * Called when the time configuration of the device has changed (date, time, time zone).
		 *
		 * @param currentTime
		 *            the new time (in milliseconds, measured from the Java Epoch)
		 * @param zoneId
		 *            the new time zone ID
		 * @param zoneOffset
		 *            the new offset from UTC
		 */
		void onTimeConfigurationChanged(long currentTime, String zoneId, int zoneOffset);
	}
}
