/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import java.lang.ref.WeakReference;

import com.is2t.tools.BitFieldHelper;
import com.is2t.tools.GeometryTools;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.MicroUI;
import ej.microui.display.GraphicsContext;
import ej.microui.event.generator.Command;
import ej.microui.util.EventHandler;

/**
 * A panel is a {@link Renderable} object that can be added on a {@link Desktop}. It can contain a {@link Widget}.
 */
public class Panel /* extends Displayable */ implements Renderable {

	// Panel state.
	private static final int IS_VALIDATING_SHIFT = 0;
	private static final int IS_VALIDATING_MASK = 0x1 << IS_VALIDATING_SHIFT;
	private static final int IS_VALID_SHIFT = 1;
	private static final int IS_VALID_MASK = 0x1 << IS_VALID_SHIFT;
	private static final int IS_PACKED_SHIFT = 2;
	private static final int IS_PACKED_MASK = 0x1 << IS_PACKED_SHIFT;
	private static final int WAIT_VALIDATE_SHIFT = 3;
	private static final int WAIT_VALIDATE_MASK = 0x1 << WAIT_VALIDATE_SHIFT;

	private byte flags;
	// private boolean isValidating;
	// private boolean isValid;
	// private boolean isPacked;
	// private boolean waitValidate;

	// widget that own the focus on the panel
	@NonNull
	private WeakReference<Widget> focusedWidget;

	// preferred size
	private short preferredWidth;
	private short preferredHeight;

	// Relative position.
	/* package */ short x;
	/* package */ short y;
	/* package */ short width;
	/* package */ short height;

	// panel controller
	@Nullable
	private EventHandler eventHandler;

	@Nullable
	private Desktop desktop;

	// XXX from MicroUI Viewable
	@Nullable
	private Widget widget;

	/**
	 * Creates a new panel.
	 * <p>
	 * By default:
	 * <ul>
	 * <li>its bounds are <code>0</code>,</li>
	 * <li>it is packed.</li>
	 * </ul>
	 */
	public Panel() {
		super();
		this.focusedWidget = new WeakReference<>(null);
		setPacked(true);
	}

	/**
	 * Attach the specified widget to this panel.
	 * <p>
	 * If there is already a widget on this panel, the former is detached from the latter.<br>
	 * If the specified widget is <code>null</code>, the panel does not hold a widget anymore.<br>
	 * <p>
	 * The panel is also ask to be revalidated if shown.
	 *
	 * @param widget
	 *            the widget to set.
	 * @throws IllegalArgumentException
	 *             if the specified widget is already in a hierarchy (already contained in a composite or panel).
	 * @see #revalidate()
	 * @see Composite#add(Widget)
	 */
	public void setWidget(@Nullable Widget widget) throws IllegalArgumentException {
		// /!\ don't repaint the panel at the end of this method

		if (this.widget != null) {
			// remove the current widget from panel
			this.widget.setPanel(null);
			setFocus(null);
		}

		// views can be null if the application wants to deactivate all views
		if (widget != null) {
			if (widget.parent != null || widget.panel != null) {
				throw new IllegalArgumentException();
			}
			// store the panel on widgets
			widget.setPanel(this);
		}

		// set the new views
		this.widget = widget;
		// the panel needs to be laid out
		invalidate();

		if (isShown()) {
			// ask for a new lay out
			revalidate();
		}
	}

	/**
	 * Gets the widget attached to this panel.
	 *
	 * @return the widget attached with this panel or <code>null</code>.
	 */
	@Nullable
	public Widget getWidget() {
		return this.widget;
	}

	/**
	 * Sets the location of this panel.
	 *
	 * @param x
	 *            the x coordinate to set.
	 * @param y
	 *            the y coordinate to set.
	 */
	public void setLocation(int x, int y) {
		setBounds(x, y, this.width, this.height);
	}

	/**
	 * Sets the size of this panel.
	 *
	 * @param width
	 *            the width to set.
	 * @param height
	 *            the height to set.
	 */
	public void setSize(int width, int height) {
		setBounds(this.x, this.y, width, height);
	}

	/**
	 * Sets the bounds of this panel.
	 *
	 * @param x
	 *            the x coordinate to set.
	 * @param y
	 *            the y coordinate to set.
	 * @param width
	 *            the width to set.
	 * @param height
	 *            the height to set.
	 */
	public void setBounds(int x, int y, int width, int height) {
		boolean resized = this.width != width || this.height != height;
		setBoundsInternal(x, y, width, height);
		if (resized) {
			invalidate();
		}
	}

	private void setBoundsInternal(int x, int y, int width, int height) {
		this.x = (short) x;
		this.y = (short) y;
		this.width = (short) width;
		this.height = (short) height;
	}

	/**
	 * Returns the preferred width of the panel. The result returned is meaningful only if {@link #isValid()} is
	 * <code>true</code> or if {@link #setPreferredSize(int, int)} has been called explicitly.
	 *
	 * @return the preferred width of the panel.
	 */
	public int getPreferredWidth() {
		return this.preferredWidth;
	}

	/**
	 * Returns the preferred height of the panel. The result returned is meaningful only if {@link #isValid()} is
	 * <code>true</code> or if {@link #setPreferredSize(int, int)} has been called explicitly.
	 *
	 * @return the preferred height of the panel.
	 */
	public int getPreferredHeight() {
		return this.preferredHeight;
	}

	/**
	 * Sets the preferred size of the panel.
	 *
	 * @param preferredWidth
	 *            the width to set.
	 * @param preferredHeight
	 *            the height to set.
	 */
	public void setPreferredSize(int preferredWidth, int preferredHeight) {
		this.preferredWidth = (short) preferredWidth;
		this.preferredHeight = (short) preferredHeight;
	}

	@Override
	public int getX() {
		return this.x;
	}

	@Override
	public int getY() {
		return this.y;
	}

	@Override
	public int getWidth() {
		return this.width;
	}

	@Override
	public int getHeight() {
		return this.height;
	}

	/**
	 * Requests the panel to be shown on the specified desktop.
	 * <p>
	 * Identical to calling {@link #show(Desktop, boolean)} without filling the desktop.
	 *
	 * @param desktop
	 *            the desktop
	 * @throws NullPointerException
	 *             if <code>desktop</code> is <code>null</code>.
	 * @see #show(Desktop, boolean)
	 * @see #hide()
	 * @see #isShown()
	 * @see #revalidate()
	 * @deprecated Use {@link #showAdjustingToChild(Desktop)} instead.
	 */
	@Deprecated
	public void show(Desktop desktop) throws NullPointerException {
		show(desktop, false);
	}

	/**
	 * Requests the panel to be shown on the specified desktop.
	 * <p>
	 * If <code>fill</code> is <code>true</code>, it is fitted to desktop size and it will be declared as not be packed.
	 * <br>
	 * If <code>fill</code> is <code>false</code> and no size has been set, it will be declared as packed.<br>
	 * If the desktop is shown, the panel is automatically validated.<br>
	 * The panel is added to the list of panels known by the desktop.
	 * <p>
	 * Special cases:
	 * <ul>
	 * <li>If the panel is already shown on this desktop, nothing is changed.</li>
	 * <li>If the panel is already shown on another desktop, it is hidden on this desktop before being shown on the new
	 * one.</li>
	 * </ul>
	 *
	 * @param desktop
	 *            the desktop.
	 * @param fill
	 *            <code>true</code> to fit the panel to the desktop size, <code>false</code> otherwise.
	 * @throws NullPointerException
	 *             if <code>desktop</code> is <code>null</code>.
	 * @see #hide()
	 * @see #isShown()
	 * @see #revalidate()
	 * @deprecated Use {@link #showAdjustingToChild(Desktop)} or {@link #showFullScreen(Desktop)} instead.
	 */
	@Deprecated
	public void show(Desktop desktop, boolean fill) throws NullPointerException {
		if (fill) {
			setPacked(false);
			setBounds(0, 0, desktop.getWidth(), desktop.getHeight());
		} else if (this.width == 0 || this.height == 0) {
			setPacked(true);
		}

		showInternal(desktop);
	}

	/**
	 * Requests the panel to be shown on the specified desktop.
	 * <p>
	 * The panel is fitted to desktop size.
	 * <p>
	 * If the desktop is shown, the panel is automatically validated.
	 * <p>
	 * The panel is added to the list of panels known by the desktop.
	 * <p>
	 * If the panel is already shown on another desktop, it is hidden on this desktop before being shown on the new
	 *
	 * @param desktop
	 *            the desktop
	 * @throws NullPointerException
	 *             if the given desktop is <code>null</code>.
	 * @see #hide()
	 * @see #isShown()
	 * @see #revalidate()
	 * @since 2.2
	 */
	public void showFullScreen(@NonNull Desktop desktop) throws NullPointerException {
		setBounds(0, 0, desktop.getWidth(), desktop.getHeight());
		invalidate();
		this.flags = (byte) BitFieldHelper.unsetBooleanProperty(this.flags, IS_PACKED_MASK);

		showInternal(desktop);
	}

	/**
	 * Requests the panel to be shown on the specified desktop.
	 * <p>
	 * The panel is fitted to its child widget size. If a location has been set, it is kept.
	 * <p>
	 * If the desktop is shown, the panel is automatically validated.
	 * <p>
	 * The panel is added to the list of panels known by the desktop.
	 * <p>
	 * If the panel is already shown on another desktop, it is hidden on this desktop before being shown on the new
	 *
	 * @param desktop
	 *            the desktop
	 * @throws NullPointerException
	 *             if the given desktop is <code>null</code>.
	 * @see #setLocation(int, int)
	 * @see #hide()
	 * @see #isShown()
	 * @see #revalidate()
	 * @since 2.2
	 */
	public void showAdjustingToChild(@NonNull Desktop desktop) throws NullPointerException {
		invalidate();
		this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, IS_PACKED_MASK);

		showInternal(desktop);
	}

	/**
	 * Requests the panel to be shown on the specified desktop.
	 * <p>
	 * The bounds of the panel remains unchanged and must be set with {@link #setBounds(int, int, int, int)} before.
	 * <p>
	 * If the desktop is shown, the panel is automatically validated.
	 * <p>
	 * The panel is added to the list of panels known by the desktop.
	 * <p>
	 * If the panel is already shown on another desktop, it is hidden on this desktop before being shown on the new
	 *
	 * @param desktop
	 *            the desktop
	 * @throws NullPointerException
	 *             if the given desktop is <code>null</code>.
	 * @see #setBounds(int, int, int, int)
	 * @see #hide()
	 * @see #isShown()
	 * @see #revalidate()
	 * @since 2.2
	 */
	public void showUsingBounds(@NonNull Desktop desktop) throws NullPointerException {
		// If bounds is not changed or the panel not invalidated, it is just repaint.
		this.flags = (byte) BitFieldHelper.unsetBooleanProperty(this.flags, IS_PACKED_MASK);

		showInternal(desktop);
	}

	/**
	 * Adds the panel to the desktop and validate it asynchronously.
	 *
	 * @param desktop
	 *            the desktop to show the panel on.
	 */
	protected void showInternal(@NonNull Desktop desktop) {
		Desktop previousDesktop = this.desktop;
		// check same desktop
		if (previousDesktop != desktop) {
			this.desktop = desktop;

			if (previousDesktop != null) {
				// remove the panel from its old desktop and repaint the latter
				hideFromDesktop(previousDesktop);
				previousDesktop.repaint();
			}

			showToDesktop(desktop); // check null
		}

		revalidate();
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * Notifies its child widget that it is shown.
	 */
	@Override
	public void showNotify() {
		if (this.widget != null) {
			this.widget.showNotify();
		}
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * Notifies its child widget that it is hidden.
	 */
	@Override
	public void hideNotify() {
		if (this.widget != null) {
			this.widget.hideNotify();
		}
	}

	/* package */void showToDesktop(@NonNull Desktop desktop) {
		desktop.addPanel(this);
	}

	/* package */void hideFromDesktop(@NonNull Desktop desktop) {
		desktop.removePanel(this);
		detachFromDesktop();
	}

	/* package */ void detachFromDesktop() {
		this.desktop = null;
	}

	/**
	 * Requests the panel be hidden.
	 * <p>
	 * The panel is invalidated.
	 * <p>
	 * The panel is removed from the list of panels known by the desktop.
	 *
	 * @see #showFullScreen(Desktop)
	 * @see #showAdjustingToChild(Desktop)
	 * @see #showUsingBounds(Desktop)
	 * @see #isShown()
	 */
	public void hide() {
		// FIXME synchro ?
		Desktop desktop = this.desktop;
		if (desktop != null) {
			hideFromDesktop(desktop);
			desktop.repaint();
		}
	}

	/**
	 * Gets whether or not the panel is shown on a shown desktop.
	 *
	 * @return <code>true</code> if the panel is shown, <code>false</code> otherwise.
	 * @see #getDesktop()
	 * @see Desktop#isShown()
	 */
	@Override
	public boolean isShown() {
		Desktop desktop = this.desktop;
		return desktop != null && desktop.isShown();
	}

	/**
	 * Gets whether or not the panel is the active one on its desktop. If the panel is not shown on a desktop, return
	 * <code>false</code>.
	 *
	 * @return <code>true</code> if the panel is the active one, <code>false</code> otherwise.
	 * @see #isShown()
	 */
	public boolean isActive() {
		if (this.desktop != null) {
			return this.desktop.getActivePanel() == this;
		}
		// not shown
		return false;
	}

	/**
	 * Notifies the panel that it is now the active panel of its desktop. The subclasses can override this method to add
	 * behavior.
	 */
	public void becameActive() {
		// Do nothing by default.
	}

	/**
	 * Notifies the panel that it is no longer the active panel of its desktop. The subclasses can override this method
	 * to add behavior.
	 */
	public void becameInactive() {
		// Do nothing by default.
	}

	/**
	 * Gets the desktop on which the panel is shown. Returns <code>null</code> if the panel is not shown.
	 *
	 * @return the desktop on which the panel is shown or <code>null</code>.
	 * @see #isShown()
	 */
	@Nullable
	public Desktop getDesktop() {
		return this.desktop;
	}

	/**
	 * Requests a repaint of this entire panel. This method returns immediately, repainting of the panel is performed
	 * asynchronously.
	 * <p>
	 * If the panel is not shown, nothing is done.
	 */
	@Override
	public void repaint() {
		repaintInternal(0, 0, this.width, this.height);
	}

	/**
	 * Requests a repaint of a zone of this panel. This method returns immediately, repainting of the panel is performed
	 * asynchronously.
	 * <p>
	 * If the panel is not shown, nothing is done.
	 *
	 * @param x
	 *            the relative x coordinate of the area to repaint.
	 * @param y
	 *            the relative y coordinate of the area to repaint.
	 * @param width
	 *            the width of the area to repaint.
	 * @param height
	 *            the height of the area to repaint.
	 */
	@Override
	public void repaint(int x, int y, int width, int height) {
		// copied in Widget.repaint(int, int, int, int)
		// crop to widget size
		// must be done in this order: location before size
		if (x < 0) {
			width += x;
			x = 0;
		}
		if (y < 0) {
			height += y;
			y = 0;
		}
		width = Math.min(this.width - x, width);
		height = Math.min(this.height - y, height);
		if (width <= 0 || height <= 0) {
			return; // nothing to repaint
		}

		repaintInternal(x, y, width, height);
	}

	@Override
	public void render(GraphicsContext g) {
		// does nothing by default
	}

	/**
	 * Gets whether this panel is transparent or not.
	 * <p>
	 * By default, a panel is transparent.
	 * <p>
	 * A transparent panel means that it will not repaint ALL the rectangular zone defined by its bounds. Then each time
	 * it needs to be repainted, its parent (recursively if also transparent) will be repainted within the bounds of the
	 * panel.
	 * <p>
	 * Each time a non-transparent panel needs to be repainted, it is the only one to be repainted.
	 *
	 * @return <code>true</code> if this panel is transparent, <code>false</code> otherwise.
	 * @see #contains(int, int)
	 */
	public boolean isTransparent() {
		return true;
	}

	/**
	 * Gets whether or not a location (x,y) is in the panel's bounds.
	 * <p>
	 * The given location is considered here as a relative location to the desktop.
	 *
	 * @param x
	 *            x coordinate.
	 * @param y
	 *            y coordinate.
	 * @return <code>true</code> if the (x,y) location is in the panel bounds, <code>false</code> otherwise.
	 * @see #isTransparent()
	 */
	public boolean contains(int x, int y) {
		// copied in widget
		return GeometryTools.contains(this.x, this.y, this.width, this.height, x, y);
	}

	/**
	 * Returns the child widget at the specified location.
	 * <p>
	 * If this panel does not <code>contains(x, y)</code>, <code>null</code> is returned. The location is considered
	 * here as a relative location to the desktop.
	 *
	 * @param x
	 *            x coordinate.
	 * @param y
	 *            y coordinate.
	 * @return the widget at the location, <code>null</code> if no widget is found in this panel hierarchy.
	 * @see Widget#getWidgetAt(int, int)
	 */
	@Nullable
	public Widget getWidgetAt(int x, int y) {
		if (contains(x, y) && this.widget != null) {
			return this.widget.getWidgetAt(x - this.x, y - this.y);
		}
		return null;
	}

	/**
	 * Sets the specified widget as the current focus owner of this panel.
	 * <p>
	 * If the widget is not enabled, nothing is done.
	 * <p>
	 * If the widget already own the focus, nothing is done.
	 * <p>
	 * Throws an {@link IllegalArgumentException} if the specified widget is not in the panel.
	 *
	 * @param widget
	 *            the widget to focus.
	 * @throws IllegalArgumentException
	 *             if the specified widget is not in the panel.
	 */
	public void setFocus(@Nullable Widget widget) throws IllegalArgumentException {
		// FIXME synchro ?
		if (widget != null) {
			if (widget.panel != this) {
				throw new IllegalArgumentException();
			}
			if (!widget.isEnabled()) {
				return;
			}
		}
		Widget oldFocusedWidget = getFocusedWidget();
		if (oldFocusedWidget == widget) {
			return;
		}
		setFocusedWidget(widget);
		if (oldFocusedWidget != null) {
			try {
				oldFocusedWidget.lostFocus();
			} catch (Throwable e) {
				// FIXME use logger
				MicroUI.MicroUI.errorLog(e);
			}
		}
		if (widget != null) {
			try {
				widget.gainFocus();
			} catch (Throwable e) {
				// FIXME use logger
				MicroUI.MicroUI.errorLog(e);
			}
		}
	}

	/**
	 * Gets the widget that is the focus owner of this panel.
	 *
	 * @return the widget that own the focus on this panel or <code>null</code>.
	 */
	public Widget getFocus() {
		return getFocusedWidget();
	}

	/**
	 * Sets this panel as packed or not.
	 * <p>
	 * This property is used when the panel is validated: if it is packed (specified boolean is <code>true</code>), the
	 * panel is resized to the preferred size of its widgets, otherwise its widgets fill its size.
	 *
	 * @param packed
	 *            <code>true</code> to pack the panel, <code>false</code> otherwise.
	 * @see #revalidate()
	 * @see #setSize(int, int)
	 * @since 1.0
	 */
	@Deprecated
	public void setPacked(boolean packed) {
		this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, packed, IS_PACKED_MASK);
		invalidate();
	}

	/**
	 * Gets whether this panel is packed or not.
	 *
	 * @return <code>true</code> if this panel is packed, <code>false</code> otherwise.
	 * @since 1.0
	 */
	@Deprecated
	public boolean isPacked() {
		return BitFieldHelper.getBooleanProperty(this.flags, IS_PACKED_MASK);
	}

	/**
	 * Gets whether this panel is adjusted to its child or not.
	 *
	 * @return <code>true</code> if this panel is adjusted to its child, <code>false</code> otherwise.
	 * @see #showAdjustingToChild(Desktop)
	 * @since 2.2
	 */
	public boolean isAdjustedToChild() {
		return BitFieldHelper.getBooleanProperty(this.flags, IS_PACKED_MASK);
	}

	/**
	 * Declares that this panel needs to be laid out.
	 *
	 * @see #revalidate()
	 * @see #validate()
	 */
	public void invalidate() {
		this.flags = (byte) BitFieldHelper.unsetBooleanProperty(this.flags, IS_VALID_MASK);
	}

	/**
	 * Gets whether this panel is valid.
	 * <p>
	 * A panel is valid if its contents are correctly laid out.
	 *
	 * @return <code>true</code> if this panel is valid, <code>false</code> otherwise.
	 * @see #invalidate()
	 * @see #validate()
	 */
	public boolean isValid() {
		return BitFieldHelper.getBooleanProperty(this.flags, IS_VALID_MASK);
	}

	/**
	 * Lays out all the hierarchy of this panel.
	 * <p>
	 * It performs the method {@link #validate()} asynchronously. Therefore this method does not block until the
	 * validation of the hierarchy is done.
	 * <p>
	 * Nothing is done if the panel is already valid.
	 * <p>
	 * The panel is not validated if it is not shown or if its desktop is not shown.
	 *
	 * @see #invalidate()
	 * @since 1.0
	 */
	public void revalidate() {
		if (!isValid() && !BitFieldHelper.getBooleanProperty(this.flags, WAIT_VALIDATE_MASK)) {
			Desktop desktop = this.desktop;
			if (desktop != null && desktop.isShown()) {
				this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, WAIT_VALIDATE_MASK);
				// send event in display pump
				desktop.getDisplay().callSerially(new Runnable() {
					@Override
					public void run() {
						validate();
					}
				});
				// repaint cannot be done here, the size of the panel is not yet computed.
			}
		}
	}

	/**
	 * Lays out all the hierarchy of this panel.
	 * <p>
	 * Nothing is done if the panel is already valid.
	 * <p>
	 * The panel is repainted if it is shown on a desktop.
	 * <p>
	 * If the panel is declared as packed, it is resized to the preferred size of its widgets, otherwise its widgets
	 * fill its size.
	 *
	 * @see #isPacked()
	 * @see #isValid()
	 */
	public void validate() {
		// FIXME check already validating ?
		this.flags = (byte) BitFieldHelper.unsetBooleanProperty(this.flags, WAIT_VALIDATE_MASK);
		if (!isValid()) {
			// use local (avoid synchro)
			boolean pack = isPacked();

			// validate hierarchy
			startValidating();
			int widthHint;
			int heightHint;
			if (pack) {
				widthHint = MWT.NONE;
				heightHint = MWT.NONE;
			} else {
				widthHint = this.width;
				heightHint = this.height;
			}
			validate(widthHint, heightHint);
			if (pack) {
				setSize(this.preferredWidth, this.preferredHeight);
			}
			stopValidating();

			// update flags (is valid)
			this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, IS_VALID_MASK);

			Desktop desktop = this.desktop; // avoid synchro
			if (desktop != null) {
				if (!desktop.isValidating()) {
					// repaint is done here because size is just computed above (see repaintInternal())
					repaint();
				}
			}
		}
	}

	// The "validating" flag can be shared with widgets as long as the validation is done in the display pump.
	/* package */ void startValidating() {
		this.flags = (byte) BitFieldHelper.setBooleanProperty(this.flags, IS_VALIDATING_MASK);
	}

	/* package */ void stopValidating() {
		this.flags = (byte) BitFieldHelper.unsetBooleanProperty(this.flags, IS_VALIDATING_MASK);
	}

	/**
	 * Lays out all the hierarchy of this panel.
	 * <p>
	 * After this call the preferred size will have been established. The parameters defines the maximum size available
	 * for this panel, or {@link MWT#NONE} if there is no constraint.
	 *
	 * @param widthHint
	 *            the width available for this panel or {@link MWT#NONE}.
	 * @param heightHint
	 *            the height available for this panel or {@link MWT#NONE}.
	 */
	public void validate(int widthHint, int heightHint) {
		Widget widget = this.widget;
		if (widget != null) {
			widget.validate(widthHint, heightHint);

			if (widthHint == MWT.NONE) {
				widthHint = widget.getPreferredWidth();
			}
			if (heightHint == MWT.NONE) {
				heightHint = widget.getPreferredHeight();
			}

			widget.setBounds(0, 0, widthHint, heightHint);

			this.preferredWidth = (short) widthHint;
			this.preferredHeight = (short) heightHint;
		}
	}

	@Override
	public void setEventHandler(@Nullable EventHandler eventHandler) {
		this.eventHandler = eventHandler;
	}

	@Override
	@Nullable
	public EventHandler getEventHandler() {
		return this.eventHandler;
	}

	/**
	 * Called by the system if this is the active panel and if no widget in the hierarchy of the focused widget has
	 * consumed the event. Panels handles {@link Command#UP}, {@link Command#DOWN}, {@link Command#LEFT}, and
	 * {@link Command#RIGHT} commands to manage navigation.
	 *
	 * @param event
	 *            the event to handle.
	 * @return <code>true</code> if the panel has consumed the event, <code>false</code> otherwise.
	 */
	@Override
	public boolean handleEvent(int event) {
		return RenderableHelper.handleEvent(event, this.eventHandler) || handleDirection(event);
	}

	private boolean handleDirection(int event) {
		Widget widget = this.widget;
		if (widget != null) {
			int direction = RenderableHelper.getDirection(event);
			if (direction != MWT.NONE) {
				return widget.requestFocus(direction);
			}
		}
		return false;
	}

	/* NOT IN API */

	/**
	 * @return the focused widget.
	 */
	@Nullable
	private Widget getFocusedWidget() {
		return this.focusedWidget.get();
	}

	/**
	 * @param focusedWidget
	 *            the widget to set.
	 */
	/* package */void setFocusedWidget(@Nullable Widget focusedWidget) {
		this.focusedWidget = new WeakReference<>(focusedWidget);
	}

	/**
	 * Checks whether or not the panel is validating in order to avoid useless {@link #computeAbsoluteXY()}.
	 */
	/* package */boolean isValidating() {
		Desktop desktop = this.desktop; // Avoid synchro.
		if (desktop != null && desktop.isValidating()) {
			return true;
		}
		return BitFieldHelper.getBooleanProperty(this.flags, IS_VALIDATING_MASK);
	}

	/* package */void paint(@NonNull GraphicsContext g, int x, int y, int width, int height) {
		Desktop desktop = this.desktop;
		Widget widget = this.widget;
		beforePaint(g, x, y, width, height);
		// get clip before calling panel rendering (avoid potential side effects)
		int translateX = g.getTranslateX();
		int translateY = g.getTranslateY();
		int clipX = g.getClipX();
		int clipY = g.getClipY();
		int clipWidth = g.getClipWidth();
		int clipHeight = g.getClipHeight();
		RenderableHelper.paintRenderable(g, desktop, this);
		if (widget != null) {
			widget.paint(g, translateX, translateY, clipX, clipY, clipWidth, clipHeight);
		}
	}

	/**
	 * Initialize the graphics context before calling the component view paint() method (clip & translation), to paint
	 * only in the wanted area. x, y coordinates must be relatives.
	 */
	/* package */void beforePaint(@NonNull GraphicsContext g, int x, int y, int width, int height) {
		// set default parameters
		g.reset();

		// set the graphics attributes in relation with the view
		g.setSystemClip(this.x, this.y, this.width, this.height);

		// set cview translate
		g.translate(this.x, this.y);

		// update views' clip
		g.setClipAndUpdateSystemClip(x, y, width, height);
	}

	/**
	 * Repaints the widgets.<br>
	 * <ol>
	 * <li>If it is transparent, repaints all the desktop.</li>
	 * <li>If it is active (on the top of the stacking order), repaints this panel only.</li>
	 * <li>Otherwise, see {@link Desktop#repaintFrom(Renderable, int, int, int, int)}.</li>
	 * </ol>
	 */
	private void repaintInternal(int x, int y, int width, int height) {
		Desktop desktop = this.desktop; // avoid synchronizing
		if (desktop != null && desktop.isShown()) {
			if (isTransparent()) {
				// transparent
				desktop.repaint(this.x + x, this.y + y, width, height);
			} else if (desktop.getActivePanel() == this) {
				// on the top
				desktop.getDisplay().callSerially(new RepaintRenderable(desktop, x, y, width, height) {
					@Override
					protected void paint(@NonNull GraphicsContext g) {
						Panel.this.paint(g, this.x, this.y, this.width, this.height);
					}
				});
			} else {
				// not active, must repaint all renderables above
				// optim: do not need to use getAbsoluteX/Y since panel is on a desktop (that is the reference for 0, 0)
				desktop.repaintFrom(this, this.x + x, this.y + y, width, height);
			}
		}
	}

}
