/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import com.is2t.tools.ArrayTools;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.microui.display.Display;
import ej.microui.display.Displayable;
import ej.microui.display.GraphicsContext;
import ej.microui.util.EventHandler;

/**
 * A desktop is the top-level object that can be displayed on a {@link Display}.<br>
 * A desktop is built for a specific {@link Display}, and that relationship cannot be modified.<br>
 * A desktop may be shown or hidden, but at most one desktop is shown per {@link Display}.<br>
 * <p>
 * A desktop can contains several {@link Panel} instances.<br>
 * These panels are stored in a list. The order of the list defines the front-to-back stacking order of the panels
 * within the desktop. The first panel in the list is at the back of the stacking order.
 *
 * @see Display
 * @see Panel
 */
public class Desktop extends Displayable implements Renderable {

	private boolean isValidating;

	// FIXME use Displayables!
	@NonNull
	private Panel[] panels;

	// stack of the dialogs
	@NonNull
	private Dialog[] dialogsStack;

	// view from which the "paint from" must start
	@Nullable
	private Renderable renderableFrom;

	@NonNull
	private final EventHandler eventManagementPolicy;

	// desktop controller
	@Nullable
	private EventHandler eventHandler;

	/**
	 * Creates a new desktop on the default display.<br>
	 * Identical to <code>new Desktop({@link Display#getDefaultDisplay()})</code>.
	 */
	public Desktop() {
		this(Display.getDefaultDisplay());
	}

	/**
	 * Creates a new desktop on the specified display. The newly created desktop is hidden.
	 *
	 * @param display
	 *            the display for which the desktop is created.
	 * @throws NullPointerException
	 *             if <code>display</code> is <code>null</code>.
	 */
	public Desktop(@NonNull Display display) throws NullPointerException {
		super(display);
		this.eventManagementPolicy = new DesktopEventManagementPolicy(this);
		this.isValidating = false;
		this.panels = new Panel[0];
		this.dialogsStack = new Dialog[0];
	}

	/**
	 * Returns the x coordinate of this desktop, which is always 0.
	 *
	 * @return the x coordinate of this desktop.
	 */
	@Override
	public int getX() {
		return 0;
	}

	/**
	 * Returns the y coordinate of this desktop, which is always 0.
	 *
	 * @return the y coordinate of this desktop.
	 */
	@Override
	public int getY() {
		return 0;
	}

	/**
	 * Returns the width of this desktop: it is equal to the width of its associated display.
	 *
	 * @return the width of this desktop.
	 */
	@Override
	public int getWidth() {
		return getDisplay().getWidth();
	}

	/**
	 * Returns the height of this desktop: it is equal to the height of its associated display.
	 *
	 * @return the height of this desktop.
	 */
	@Override
	public int getHeight() {
		return getDisplay().getHeight();
	}

	/**
	 * Requests a repaint of a zone of this desktop.
	 * <p>
	 * This method returns immediately, the repaint of the desktop is performed asynchronously.<br>
	 * If the desktop is not shown, nothing is done.
	 *
	 * @param x
	 *            the relative x coordinate of the area to repaint.
	 * @param y
	 *            the relative y coordinate of the area to repaint.
	 * @param width
	 *            the width of the area to repaint.
	 * @param height
	 *            the height of the area to repaint.
	 */
	@Override
	public void repaint(final int x, final int y, final int width, final int height) {
		if (isShown()) {
			getDisplay().callSerially(new RepaintRenderable(this, x, y, width, height) {
				@Override
				protected void paint(GraphicsContext g) {
					Desktop.this.paint(g, this.x, this.y, this.width, this.height);
				}
			});
		}
	}

	@Override
	public void render(@NonNull GraphicsContext g) {
		// does nothing by default
	}

	// /**
	// * MICROWT-API Gets the style of this widget. The style is used to get the best match renderer to associate with
	// * this desktop.
	// *
	// * @return the style.
	// * @since 1.0
	// */
	// @Override
	// public int getStyle() {
	// return 0;
	// }

	/**
	 * Gets this desktop's active panel. The active panel is the last in the panel's list.
	 *
	 * @return the desktop's active panel, or <code>null</code> if none.
	 */
	@Nullable
	public Panel getActivePanel() {
		Panel[] panels = this.panels;
		return panels.length == 0 ? null : (Panel) panels[panels.length - 1];
	}

	/**
	 * Sets the specified panel as the active one on this desktop.
	 *
	 * @param panel
	 *            the panel to set active
	 * @see #getActivePanel()
	 * @throws NullPointerException
	 *             if the specified panel is null.
	 * @throws IllegalArgumentException
	 *             if the specified panel is not on this desktop.
	 */
	public void setActivePanel(@NonNull Panel panel) throws NullPointerException, IllegalArgumentException {
		if (!setActivePanelInternal(panel)) {
			throw new IllegalArgumentException();
		}
	}

	/**
	 * Does not check if the panel is shown.
	 *
	 * @param panel
	 *            the panel to set active.
	 * @return <code>true</code> if the panel is in the desktop, <code>false</code> otherwise.
	 */
	/* package */boolean setActivePanelInternal(@Nullable Panel panel) {
		if (this.dialogsStack.length != 0) {
			return contains(panel);
		}
		Panel[] panels = this.panels;
		int panelsLength = panels.length;
		if (panelsLength != 0) {
			Panel previousActivePanel = panels[panelsLength - 1];
			if (previousActivePanel == panel) {
				// already on front
				return true;
			}
			for (int i = panelsLength; --i >= 0;) {
				@NonNull
				Panel panelCandidate = panels[i];
				if (panelCandidate == panel) {
					if (i != panelsLength - 1) {
						System.arraycopy(panels, i + 1, panels, i, panelsLength - i - 1);
						panels[panelsLength - 1] = panel;
						// notify panels AFTER updating the array (avoid recursive calls and other problems)
						previousActivePanel.becameInactive();
						panel.becameActive();
						panel.repaint();
					}
					return true;
				}
			}
		}
		// the panel is not in the desktop
		return false;
	}

	/**
	 * Gets the list of the panels being shown on this desktop.
	 * <p>
	 * A panel is added to this list when a panel is shown on this desktop and removed when a panel is hidden from this
	 * desktop.
	 *
	 * @return the list of the panels on this desktop.
	 * @see Panel#showFullScreen(Desktop)
	 * @see Panel#showAdjustingToChild(Desktop)
	 * @see Panel#showUsingBounds(Desktop)
	 * @see Panel#hide()
	 */
	@NonNull
	public Panel[] getPanels() {
		return (Panel[]) ArrayTools.copy(this.panels, Panel[].class);
	}

	/**
	 * Lays out all the hierarchy of this desktop.
	 * <p>
	 * It performs the method {@link #validate()} asynchronously. Therefore this method does not block until the
	 * validation of the hierarchy is done.
	 * <p>
	 * Nothing is done if it is not shown.
	 *
	 * @see #validate()
	 * @since 1.0
	 */
	public void revalidate() {
		// send event in display pump
		if (isShown()) {
			getDisplay().callSerially(new Runnable() {
				@Override
				public void run() {
					validate();
				}
			});
			repaint();
		}
	}

	/**
	 * Lays out all the hierarchy of this desktop.
	 *
	 * @see Panel#validate()
	 * @since 1.0
	 */
	public void validate() {
		this.isValidating = true;
		// validate all panels
		Panel[] panels = this.panels;
		int length = panels.length;
		for (int i = length; --i >= 0;) {
			Panel panel = panels[i];
			panel.validate();
		}
		this.isValidating = false;
	}

	/**
	 * The desktop is visible on its display.
	 * <p>
	 * Notifies its children panels that they are shown.
	 * <p>
	 * Ask for a revalidation of the entire desktop.
	 *
	 * @see #revalidate()
	 * @since 1.0
	 */
	@Override
	public void showNotify() {
		for (Panel panel : this.panels) {
			panel.showNotify();
		}
		revalidate();
	}

	/**
	 * The desktop is hidden from its display.
	 * <p>
	 * Notifies its children panels that they are hidden.
	 *
	 * @since 1.0
	 */
	@Override
	public void hideNotify() {
		for (Panel panel : this.panels) {
			panel.hideNotify();
		}
	}

	@Override
	public void becomeCurrent() {
		showNotify();
	}

	@Override
	@NonNull
	public EventHandler getController() {
		return this.eventManagementPolicy;
	}

	@Override
	public void setEventHandler(@Nullable EventHandler eventHandler) {
		this.eventHandler = eventHandler;
	}

	@Override
	@Nullable
	public EventHandler getEventHandler() {
		return this.eventHandler;
	}

	/**
	 * Called by the system if no widget nor panel in the focused hierarchy has consumed the event.
	 *
	 * @param event
	 *            the event to handle
	 * @return <code>true</code> if the desktop has consumed the event, <code>false</code> otherwise
	 * @since 0.9
	 */
	@Override
	public boolean handleEvent(int event) {
		return RenderableHelper.handleEvent(event, this.eventHandler);
	}

	/**
	 * Repaints all the hierarchy of the desktop that is over (in the drawing order) the given renderable.
	 *
	 * @param renderable
	 *            the renderable to start rendering from.
	 * @param x
	 *            the relative x coordinate of the area to paint.
	 * @param y
	 *            the relative y coordinate of the area to paint.
	 * @param width
	 *            the width of the area to paint.
	 * @param height
	 *            the height of the area to paint.
	 */
	/* package */void repaintFrom(@NonNull final Renderable renderable, final int x, final int y, final int width,
			final int height) {
		if (isShown()) {
			getDisplay().callSerially(new RepaintRenderable(this, x, y, width, height) {

				@Override
				protected void paint(GraphicsContext g) {
					Desktop.this.renderableFrom = renderable;
					Desktop.this.paint(g, this.x, this.y, this.width, this.height);
					Desktop.this.renderableFrom = null;
				}
			});
		}
	}

	@Override
	public final void paint(GraphicsContext g) {
		paint(g, 0, 0, getWidth(), getHeight());
	}

	/* package */void paint(@NonNull GraphicsContext g, int x, int y, int width, int height) {
		beforePaint(g, x, y, width, height);
		RenderableHelper.paintRenderable(g, this, this);
		Panel[] panels = this.panels; // Avoid synchro.
		int panelsLength = panels.length;
		// from last to first
		for (int i = -1; ++i < panelsLength;) {
			Panel panel = panels[i];
			panel.paint(g, x - panel.x, y - panel.y, width, height);
		}
	}

	/**
	 * Initialize the graphics context before calling the component view paint() method (clip & translation), to paint
	 * only in the wanted area. x, y coordinates must be relatives.
	 */
	/* package */void beforePaint(@NonNull GraphicsContext g, int x, int y, int width, int height) {
		// set default parameters
		g.reset();

		// set the graphics attributes in relation with the view
		g.setSystemClip(0, 0, getWidth(), getHeight());

		// set cview translate
		g.translate(0, 0);

		// update views' clip
		g.setClipAndUpdateSystemClip(x, y, width, height);
	}

	/* package */boolean checkPaintFrom(@NonNull Renderable renderable) {
		if (this.renderableFrom != null) { // the from view is not yet found
			if (this.renderableFrom == renderable) {
				// start rendering
				this.renderableFrom = null;
			} else {
				return false;
			}
		}
		return true;
	}

	/**
	 * Checks whether or not the desktop is validating in order to avoid useless {@link #computeAbsoluteXY()}.
	 */
	/* package */boolean isValidating() {
		return this.isValidating;
	}

	private boolean contains(@Nullable Panel panel) {
		for (Panel panelCandidate : this.panels) {
			if (panelCandidate == panel) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Adds a panel.
	 * <p>
	 * The panel became the active one, the old one became inactive except if there is at least one dialog open.
	 */
	/* package */void addPanel(@NonNull Panel panel) {
		Panel[] panels = this.panels;
		int panelsLength = panels.length;
		Panel previousActivePanel;
		if (panelsLength != 0) {
			previousActivePanel = panels[panels.length - 1];
		} else {
			previousActivePanel = null;
		}
		panels = (Panel[]) ArrayTools.add(panels, panel);

		Dialog[] dialogsStack = this.dialogsStack;
		int dialogsStackLength = dialogsStack.length;
		if (dialogsStackLength != 0) {
			// Do not hide dialogs: move the added panel before the dialogs in the array.
			System.arraycopy(panels, panelsLength - dialogsStackLength, panels, panelsLength - dialogsStackLength + 1,
					dialogsStackLength);
			panels[panelsLength - dialogsStackLength] = panel;
		}
		this.panels = panels;
		if (isShown()) {
			panel.showNotify();
		}
		if (dialogsStackLength == 0) {
			if (previousActivePanel != null) {
				previousActivePanel.becameInactive();
			}
			panel.becameActive();
		}
	}

	/**
	 * Adds a dialog.
	 * <p>
	 * The panel became the active one, the old one became inactive.
	 */
	/* package */void addDialog(@NonNull Dialog dialog) {
		Panel[] panels = this.panels;
		int panelsLength = panels.length;
		if (panelsLength != 0) {
			Panel previousActivePanel = panels[panelsLength - 1];
			previousActivePanel.becameInactive();
		}
		this.panels = (Panel[]) ArrayTools.add(panels, dialog);
		this.dialogsStack = (Dialog[]) ArrayTools.add(this.dialogsStack, dialog);
		if (isShown()) {
			dialog.showNotify();
		}
		dialog.becameActive();
	}

	/**
	 * Removes a panel.<br>
	 * If the panel was active, it became inactive before, the new one became active.<br>
	 * If it is a dialog, remove it from the dialog stack.<br>
	 */
	/* package */void removePanel(@NonNull Panel panel) {
		Panel[] panels = this.panels;
		int panelsLength = panels.length;
		boolean wasActive = panelsLength == 0 ? false : panels[panelsLength - 1] == panel;
		this.panels = (Panel[]) ArrayTools.remove(panels, panel);
		if (wasActive) {
			panel.becameInactive();
		}
		if (isShown()) {
			panel.hideNotify();
		}
		if (wasActive && panelsLength > 1) {
			this.panels[panelsLength - 2].becameActive();
		}
	}

	/* package */void removeDialog(@NonNull Dialog dialog) {
		removePanel(dialog);
		this.dialogsStack = (Dialog[]) ArrayTools.remove(this.dialogsStack, dialog);
	}

	/**
	 * Gets the panel at the specified position.
	 *
	 * @param x
	 *            the x coordinate to search at.
	 * @param y
	 *            the y coordinate to search at.
	 * @return the panel at the specified location or <code>null</code> if none.
	 *
	 * @since 2.0
	 */
	@Nullable
	public Panel panelAt(int x, int y) {
		Panel[] panels = this.panels;
		int length = panels.length;
		// must lookup from end to start (stacking order, see addPanel and setActivePanel)
		for (int i = length; --i >= 0;) {
			Panel panel = panels[i];
			if (panel.contains(x, y)) {
				return panel;
			}
		}
		return null;
	}

	/**
	 * Checks whether or not a panel is accessible to the user.
	 * <p>
	 * A panel is accessible if:
	 * <ol>
	 * <li>there is no dialog,</li>
	 * <li>or the panel is the dialog on the top.</li>
	 * </ol>
	 *
	 * @param panel
	 *            the panel to check.
	 * @return <code>true</code> if the panel is accessible, <code>false</code> otherwise.
	 *
	 * @since 2.0
	 */
	public boolean isAccessible(@NonNull Panel panel) {
		Dialog[] dialogsStack = this.dialogsStack;
		int dialogsStackLength = dialogsStack.length;
		return (dialogsStackLength == 0 // there is no dialog
				|| panel == dialogsStack[dialogsStackLength - 1]); // the panel is the top dialog
		// FIXME use getActivePanel() instead of dialogsStack[dialogsStackLength - 1]?
		// -> avoid exposing this method
	}

}
