/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util.server;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.bluetooth.BluetoothAdapter;
import ej.bluetooth.BluetoothDevice;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.callbacks.AdvertisementCallbacks;
import ej.bluetooth.callbacks.ConnectionCallbacks;
import ej.bluetooth.util.BluetoothPayloadBuilder;
import ej.bluetooth.util.BluetoothPayloadUtil;

/**
 * A bluetooth server.
 */
public class BasicBluetoothServer implements ConnectionCallbacks, AdvertisementCallbacks {

	private final BluetoothAdapter adapter;
	private final BluetoothPayloadBuilder bluetoothPayloadBuilder;
	private AdvertisementCallbacks advertisment;
	private ConnectionCallbacks connectionCallbacks;
	private int advertisingThreshold = 1;

	/**
	 * Instantiates a new {@link BasicBluetoothServer}.
	 *
	 * @param adapter
	 *            the adaptater to use.
	 * @param bluetoothPayloadBuilder
	 *            the payload to add.
	 */
	public BasicBluetoothServer(@NonNull BluetoothAdapter adapter,
			@NonNull BluetoothPayloadBuilder bluetoothPayloadBuilder) {
		this.adapter = adapter;
		this.bluetoothPayloadBuilder = bluetoothPayloadBuilder;
	}

	/**
	 * Instantiates a new {@link BasicBluetoothServer}.
	 *
	 * @param adapter
	 *            the adapter to use.
	 * @param localName
	 *            the local name to advertise.
	 */
	public BasicBluetoothServer(@NonNull BluetoothAdapter adapter, @NonNull String localName) {
		this(adapter, new BluetoothPayloadBuilder(BluetoothPayloadUtil.COMPLETE_LOCAL_NAME, localName));
	}

	/**
	 * Instantiates a new {@link BasicBluetoothServer} with the default {@link BluetoothAdapter}.
	 *
	 * @param localName
	 *            the local name to advertise.
	 * @see BluetoothAdapter#getDefaultAdapter()
	 */
	public BasicBluetoothServer(@NonNull String localName) {
		this(BluetoothAdapter.getDefaultAdapter(), localName);
	}

	/**
	 * Starts the server.
	 */
	public void start() {
		startAdvertising();
	}

	/**
	 * Start advertising.
	 */
	public void startAdvertising() {
		byte[] payload = this.bluetoothPayloadBuilder.getPayload();
		if (BluetoothPayloadUtil.getByte(BluetoothPayloadUtil.FLAGS, payload) == null) {
			this.bluetoothPayloadBuilder.append(BluetoothPayloadUtil.FLAGS,
					BluetoothPayloadUtil.FLAGS_LE_ONLY_GENERAL_DISC_MODE);
			payload = this.bluetoothPayloadBuilder.getPayload();
		}
		this.adapter.startAdvertising(this, this, payload);
	}

	/**
	 * Starts the server.
	 *
	 * @param advertisment
	 *            the advertisment callback.
	 * @param connectionCallbacks
	 *            the connection callback.
	 */
	public void start(@Nullable AdvertisementCallbacks advertisment,
			@Nullable ConnectionCallbacks connectionCallbacks) {
		this.advertisment = advertisment;
		this.connectionCallbacks = connectionCallbacks;
		start();
	}

	/**
	 * Stops the server.
	 */
	public void stop() {
		stopAdvertising();
	}

	/**
	 * Stops the advertising.
	 *
	 * @see BluetoothAdapter#stopAdvertising()
	 */
	public void stopAdvertising() {
		this.adapter.stopAdvertising();
	}

	@Override
	public void onConnectFailed(BluetoothDevice device) {
		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onConnectFailed(device);
		}
	}

	@Override
	public void onConnected(BluetoothDevice device) {
		device.discoverServices();

		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onConnected(device);
		}

		BluetoothDevice[] devices = getDevices();
		if (this.advertisingThreshold >= 0 && devices.length >= this.advertisingThreshold) {
			stopAdvertising();
		}
	}

	@Override
	public void onDisconnected(BluetoothDevice device) {
		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onDisconnected(device);
		}
		ServerStorage.remove(device);

		BluetoothDevice[] devices = getDevices();
		if (this.advertisingThreshold >= 0 && devices.length < this.advertisingThreshold) {
			startAdvertising();
		}
	}

	@Override
	public void onPairRequest(BluetoothDevice device) {
		device.pairReply(true);
		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onPairRequest(device);
		}
	}

	@Override
	public void onPairCompleted(BluetoothDevice device, boolean success) {
		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onPairCompleted(device, success);
		}
	}

	@Override
	public void onPasskeyRequest(BluetoothDevice device) {
		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onPasskeyRequest(device);
		}
	}

	@Override
	public void onPasskeyGenerated(BluetoothDevice device, int passkey) {
		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onPasskeyGenerated(device, passkey);
		}
	}

	@Override
	public void onServicesDiscovered(BluetoothDevice device) {
		if (this.connectionCallbacks != null) {
			this.connectionCallbacks.onServicesDiscovered(device);
		}
	}

	/**
	 * Gets the devices.
	 *
	 * @return the devices.
	 * @see BluetoothAdapter#getDevices()
	 */
	public BluetoothDevice[] getDevices() {
		return this.adapter.getConnectedDevices();
	}

	/**
	 * Notifies the devices that are register to a characteristics.
	 *
	 * @param characteristic
	 *            the characteristic to notify.
	 */
	public void notifyDevices(BasicServerCharacteristic characteristic) {
		characteristic.notify(getDevices());
	}

	@Override
	public void onAdvertisementCompleted(BluetoothAdapter adapter) {
		if (this.advertisment != null) {
			this.advertisment.onAdvertisementCompleted(adapter);
		}
	}

	/**
	 * Add a service to the server.
	 *
	 * @param service
	 *            the service to add.
	 * @see BluetoothAdapter#addService(ej.bluetooth.BluetoothService)
	 */
	public void addService(BluetoothService service) {
		this.adapter.addService(service);
	}

	/**
	 * Sets the advertisingThreshold. This value is the threshold of the number of device to auto start and stop the
	 * advertising.
	 *
	 * @param advertisingThreshold
	 *            the advertisingThreshold to set. 0 to not using it.
	 */
	public void setAdvertisingThreshold(int advertisingThreshold) {
		this.advertisingThreshold = advertisingThreshold;
	}

	/**
	 * Gets the advertisingThreshold.
	 *
	 * @return the advertisingThreshold.
	 */
	public int getAdvertisingThreshold() {
		return this.advertisingThreshold;
	}
}
