/*
 * Java
 *
 * Copyright 2019-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */

package com.microej.kf.util.storage;

import ej.kf.Kernel;
import ej.kf.Module;
import ej.storage.fs.StorageFs;

import java.io.IOException;

/**
 * Extends the storage on file system to add sandboxing for the applications.
 */
public class SandboxedStorage extends StorageFs {

	private static final String KERNEL_PARENT_PREFIX = "kernel";
	private static final String FEATURES_PARENT_PREFIX = "apps";

	/**
	 * Creates a sandboxed storage on file system.
	 * <p>
	 * The root folder is determined dynamically depending on the KF context. Each module has a different one.
	 *
	 * @throws IOException
	 * 		if the root cannot be created.
	 * @see Kernel#getContextOwner()
	 * @see StorageFs#StorageFs()
	 */
	public SandboxedStorage() throws IOException {
		this(getRootFolderPropertyOrDefault());
	}

	/**
	 * Creates a storage on file system specifying the root folder where the root folder will be created.
	 * <p>
	 * The root folder is determined dynamically depending on the KF context. Each module has a different one.
	 *
	 * @param root
	 * 		the root folder.
	 * @throws IllegalArgumentException
	 * 		if the root already exists and is not a directory.
	 * @throws IOException
	 * 		if the root cannot be created.
	 */
	public SandboxedStorage(String root) throws IOException {
		super(root + '/' + getContextName(Kernel.getContextOwner()));
	}

	/**
	 * Gets the root folder of the storage service for a given module.
	 *
	 * @param module
	 * 		the module.
	 * @return the root folder.
	 */
	public String getRootFolder(Module module) {
		return getRootFolderPropertyOrDefault() + '/' + getContextName(module);
	}

	private static String getContextName(Module module) {
		if (module == Kernel.getInstance()) {
			return KERNEL_PARENT_PREFIX;
		} else {
			return FEATURES_PARENT_PREFIX + "/" + module.getName();
		}
	}

}