/*
	Copyright (c) 2002 JSON.org
	Copyright 2016 IS2T. This file has been modified by IS2T S.A.

	Permission is hereby granted, free of charge, to any person obtaining a copy
	of this software and associated documentation files (the "Software"), to deal
	in the Software without restriction, including without limitation the rights
	to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
	copies of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in all
	copies or substantial portions of the Software.

	The Software shall be used for Good, not Evil.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
	SOFTWARE.
 */
package ej.json;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

/**
 * A JSONReader takes a source {@link InputStream} and extracts characters and tokens from it.
 * @author IS2T
 * @version 1
 */
public class JSONReader {

	/**
	 * It is sometimes more convenient and less ambiguous to have a <code>NULL</code> object than to use Java's
	 * <code>null</code> value. <code>NULL.equals(null)</code> returns <code>true</code>.
	 * <code>NULL.toString()</code> returns <code>"null"</code>.
	 */
	public static final Object NULL = new Null();

	private static final int ERROR_MISSING_VALUE 			= 0;
	private static final int ERROR_UNTERMINATED_STRING 		= 1;
	private static final int ERROR_UNCLOSED_COMMENT 		= 2;
	private static final int ERROR_WRONG_KEY_SEP 			= 3;
	private static final int ERROR_UNEXPECTED_TOKEN 		= 4;
	private static final int ERROR_MISSING_KEY 				= 5;

	private static final int NOT_SET = -1;

	private final InputStreamReader is;
	/**
	 * -1: not set
	 * 0: EOF
	 * >0: the current char
	 */
	private int current = NOT_SET;


	/**
	 * Construct a JSONReader from an {@link InputStream}.
	 *
	 * @param is     the InputStream.
	 */
	public JSONReader(InputStream is) {
		this.is = new InputStreamReader(is);
	}

	/**
	 * Get the first character of the next token.
	 * @return the first character or <code>0</code> on EOF.
	 * @throws JSONException on I/O error
	 */
	public char peek() throws JSONException{
		char c = nextClean();
		if(c == ','){
			c = nextClean();
		}
		back(c);
		return c;
	}

	/**
	 * Back up one character. This provides a sort of lookahead capability,
	 * so that you can test for a digit or letter before attempting to parse
	 * the next number or identifier.
	 * @param c the character to back up
	 */
	private void back(char c) {
		current = c;
	}

	/**
	 * Get the next character in the source string.
	 *
	 * @return The next character, or 0 if past the end of the source string.
	 * @throws JSONException on I/O error
	 */
	private char next() throws JSONException{
		int i;
		if((i=current) != NOT_SET){
			current = NOT_SET;
			return (char)i;
		}
		try {
			i = is.read();
			if(i != -1){
				return (char)i;
			}
			else{
				return 0;
			}
		} catch (IOException e) {
			throw new JSONException(e);
		}
	}

	/**
	 * Get the next n characters.
	 *
	 * @param n     The number of characters to take.
	 * @return      A string of n characters.
	 * @throws JSONException
	 *   Substring bounds error if there are not
	 *   n characters remaining in the source string.
	 */
	private String next(int n) throws JSONException {
		StringBuilder sb = new StringBuilder();
		for(int i=0; i<n;++i){
			sb.append(next());
		}
		return sb.toString();
	}


	/**
	 * Get the next char in the string, skipping whitespace
	 * and comments (slashslash, slashstar, and hash).
	 * @throws JSONException
	 * @return  A character, or 0 if there are no more characters.
	 */
	private char nextClean() throws JSONException {
		for (;;) {
			char c = next();
			if (c == '/') {
				switch (c = next()) {
				case '/':
					do {
						c = next();
					} while (c != '\n' && c != '\r' && c != 0);
					break;
				case '*':
					for (;;) {
						c = next();
						if (c == 0) {
							throw syntaxError(ERROR_UNCLOSED_COMMENT);
						}
						if (c == '*') {
							if ((c = next()) == '/') {
								break;
							}
							back(c);
						}
					}
					break;
				default:
					back(c);
					return '/';
				}
			} else if (c == '#') {
				do {
					c = next();
				} while (c != '\n' && c != '\r' && c != 0);
			} else if (c == 0 || c > ' ') {
				return c;
			}
		}
	}


	/**
	 * Return the characters up to the next close quote character.
	 * Backslash processing is done. The formal JSON format does not
	 * allow strings in single quotes, but an implementation is allowed to
	 * accept them.
	 * @param quote The quoting character, either
	 *      <code>"</code>&nbsp;<small>(double quote)</small> or
	 *      <code>'</code>&nbsp;<small>(single quote)</small>.
	 * @return      A String.
	 * @throws JSONException Unterminated string.
	 */
	private String nextString(char quote) throws JSONException {
		char c;
		StringBuffer sb = new StringBuffer();
		for (;;) {
			c = next();
			switch (c) {
			case 0:
			case '\n':
			case '\r':
				throw syntaxError(ERROR_UNTERMINATED_STRING);
			case '\\':
				c = next();
				switch (c) {
				case 'b':
					sb.append('\b');
					break;
				case 't':
					sb.append('\t');
					break;
				case 'n':
					sb.append('\n');
					break;
				case 'f':
					sb.append('\f');
					break;
				case 'r':
					sb.append('\r');
					break;
				case 'u':
					sb.append((char)Integer.parseInt(next(4), 16));
					break;
				case 'x' :
					sb.append((char) Integer.parseInt(next(2), 16));
					break;
				default:
					sb.append(c);
				}
				break;
			default:
				if (c == quote) {
					return sb.toString();
				}
				sb.append(c);
			}
		}
	}


	/**
	 * Get the next key as String.
	 * @throws JSONException If syntax error.
	 *
	 * @return A String
	 */
	public String nextKey() throws JSONException{
		Object o = nextValue();
		if(o == null){
			throw syntaxError(ERROR_MISSING_KEY);
		}
		String result;
		try{
			result = (String)o;
		}
		catch(ClassCastException e){
			throw syntaxError(ERROR_MISSING_KEY);
		}

		/*
		 * The key is followed by ':'. We will also tolerate '=' or '=>'.
		 */
		char c = nextClean();
		if (c == '=') {
			if ((c=next()) != '>') {
				back(c);
			}
		} else if (c != ':') {
			throw syntaxError(ERROR_WRONG_KEY_SEP);
		}

		return result;
	}

	/**
	 * Start an array.
	 * @throws JSONException If syntax error.
	 */
	public void arrayStart() throws JSONException {
		char c = nextClean();
		if(c == ','){
			// skip array value separator
			c = nextClean();
		}
		if(c != '['){
			throw syntaxError(ERROR_UNEXPECTED_TOKEN);
		}
	}

	/**
	 * End an array.
	 * @throws JSONException If syntax error.
	 */
	public void arrayEnd() throws JSONException {
		if(nextClean() != ']'){
			throw syntaxError(ERROR_UNEXPECTED_TOKEN);
		}
	}

	/**
	 * Start a map.
	 * @throws JSONException If syntax error.
	 */
	public void mapStart() throws JSONException {
		char c = nextClean();
		if(c == ','){
			// skip array value separator
			c = nextClean();
		}
		if(c != '{'){
			throw syntaxError(ERROR_UNEXPECTED_TOKEN);
		}
	}

	/**
	 * End a map.
	 * @throws JSONException If syntax error.
	 */
	public void mapEnd() throws JSONException {
		if(nextClean() != '}'){
			throw syntaxError(ERROR_UNEXPECTED_TOKEN);
		}
	}

	/**
	 * Get the next value. The value can be a Boolean, Double, Integer, Long, or String, or the {@link Null} object.
	 * @throws JSONException If syntax error.
	 *
	 * @return An object or null if the beginning of array or a map is reached.
	 */
	public Object nextValue() throws JSONException {
		char c = nextClean();

		if(c == ','){
			// skip array value separator
			c = nextClean();
		}

		String s;

		switch (c) {
		case '"':
		case '\'':
			return nextString(c);
		case '{':
			back(c);
			return null;
		case '[':
			back(c);
			return null;
		}

		/*
		 * Handle unquoted text. This could be the values true, false, or
		 * null, or it can be a number. An implementation (such as this one)
		 * is allowed to also accept non-standard forms.
		 *
		 * Accumulate characters until we reach the end of the text or a
		 * formatting character.
		 */

		StringBuffer sb = new StringBuffer();
		char b = c;
		while (c >= ' ' && ",:]}/\\\"[{;=#".indexOf(c) < 0) {
			sb.append(c);
			c = next();
		}
		back(c);

		/*
		 * If it is true, false, or null, return the proper value.
		 */

		s = sb.toString().trim();
		if (s.equals("")) {
			throw syntaxError(ERROR_MISSING_VALUE);
		}
		if (s.toLowerCase().equals("true")) {
			return Boolean.TRUE;
		}
		if (s.toLowerCase().equals("false")) {
			return Boolean.FALSE;
		}
		if (s.toLowerCase().equals("null")) {
			return NULL;
		}

		/*
		 * If it might be a number, try converting it. We support the 0- and 0x-
		 * conventions. If a number cannot be produced, then the value will just
		 * be a string. Note that the 0-, 0x-, plus, and implied string
		 * conventions are non-standard. A JSON parser is free to accept
		 * non-JSON forms as long as it accepts all correct JSON forms.
		 */

		if ((b >= '0' && b <= '9') || b == '.' || b == '-' || b == '+') {
			if (b == '0') {
				if (s.length() > 2 &&
						(s.charAt(1) == 'x' || s.charAt(1) == 'X')) {
					try {
						return new Integer(Integer.parseInt(s.substring(2),
								16));
					} catch (Exception e) {
						/* Ignore the error */
					}
				} else {
					try {
						return new Integer(Integer.parseInt(s, 8));
					} catch (Exception e) {
						/* Ignore the error */
					}
				}
			}
			try {
				return Integer.valueOf(s);
			} catch (Exception e) {
				try {
					return new Long(Long.parseLong(s));
				} catch (Exception f) {
					try {
						return Double.valueOf(s);
					}  catch (Exception g) {
						return s;
					}
				}
			}
		}
		return s;
	}


	/**
	 * Make a JSONException to signal a syntax error.
	 *
	 * @param code The error code.
	 * @return  A JSONException object, suitable for throwing
	 */
	private JSONException syntaxError(int code) {
		return new JSONException("E="+code);
	}

	/**
	 * {@link Null} is equivalent to the value that JavaScript calls null, while Java's null is equivalent to the
	 * value that JavaScript calls undefined.
	 */
	private static final class Null {

		/**
		 * There is only intended to be a single instance of the NULL object, so the clone method returns itself.
		 *
		 * @return NULL.
		 */
		@Override
		protected final Object clone() {
			return this;
		}

		/**
		 * A Null object is equal to the null value and to itself.
		 *
		 * @param object
		 *            An object to test for nullness.
		 * @return true if the object parameter is the {@link Null} object or null.
		 */
		@Override
		public boolean equals(Object object) {
			return object == null || object == this;
		}

		/**
		 * Get the "null" string value.
		 *
		 * @return The string "null".
		 */
		@Override
		public String toString() {
			return "null";
		}
	}
}