/*
 * Copyright (c) 2000, 2015, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2013-2020 MicroEJ Corp. - EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.util.logging;

import ej.annotation.Nullable;

/**
 * LogRecord objects are used to pass logging requests between the logging framework and individual log Handlers.
 * <p>
 * When a LogRecord is passed into the logging framework it logically belongs to the framework and should no longer be
 * used or updated by the client application.
 * <p>
 * Note that if the client application has not specified an explicit source method name and source class name, then the
 * LogRecord class will infer them automatically when they are first accessed (due to a call on getSourceMethodName or
 * getSourceClassName) by analyzing the call stack. Therefore, if a logging Handler wants to pass off a LogRecord to
 * another thread, or to transmit it over RMI, and if it wishes to subsequently obtain method name or class name
 * information it should call one of getSourceClassName or getSourceMethodName to force the values to be filled in.
 */
public class LogRecord {

	// Current sequence number that will be assigned to the next LogRecord
	private static long CurrentSequenceNumber = 0;

	// Properties of this log record
	private final long sequenceNumber;
	private long millis;

	private Level level;
	@Nullable
	private String message;

	@Nullable
	private Throwable thrown;
	@Nullable
	private String loggerName;

	/**
	 * Construct a LogRecord with the given level and message.
	 *
	 * @param level
	 *            a logging level value.
	 * @param msg
	 *            the logging message (may be {@code null}).
	 */
	public LogRecord(Level level, @Nullable String msg) {
		this.level = level;
		this.message = msg;

		// Generated fields
		this.sequenceNumber = CurrentSequenceNumber++;
		// FIXME thread safety issue
		this.millis = System.currentTimeMillis();

		// Initialize other properties to "null"
		this.thrown = null;
		this.loggerName = null;
	}

	/**
	 * Get the name of the logger that logged this LogRecord.
	 *
	 * @return source logger name (may be {@code null}).
	 */
	@Nullable
	public String getLoggerName() {
		return this.loggerName;
	}

	/**
	 * Set the source logger's name.
	 *
	 * @param name
	 *            the source logger's name (may be {@code null}).
	 */
	public void setLoggerName(@Nullable String name) {
		this.loggerName = name;
	}

	/**
	 * Get the logging message level.
	 *
	 * @return the logging message level.
	 */
	public Level getLevel() {
		return this.level;
	}

	/**
	 * Set the logging message level.
	 *
	 * @param level
	 *            the logging message level.
	 */
	public void setLevel(Level level) {
		this.level = level;
	}

	/**
	 * Get the sequence number of the LogRecord. This number is normally a unique identifier assigned by the
	 * constructor.
	 *
	 * @return the sequence number of the LogRecord.
	 */
	public long getSequenceNumber() {
		return this.sequenceNumber;
	}

	/**
	 * Get the raw log message.
	 *
	 * @return the message string (may be {@code null}).
	 */
	@Nullable
	public String getMessage() {
		return this.message;
	}

	/**
	 * Set the raw log message.
	 *
	 * @param message
	 *            the message to set (may be {@code null}).
	 */
	public void setMessage(@Nullable String message) {
		this.message = message;
	}

	/**
	 * Get event time in milliseconds since 1970. This is the creation time of the LogRecord.
	 *
	 * @return event time in milliseconds.
	 */
	public long getMillis() {
		return this.millis;
	}

	/**
	 * Set event time of the LogRecord.
	 *
	 * @param millis
	 *            event time.
	 */
	public void setMillis(long millis) {
		this.millis = millis;
	}

	/**
	 * Get the throwable associated with this log record, if any.
	 *
	 * @return a throwable (may be {@code null}).
	 */
	@Nullable
	public Throwable getThrown() {
		return this.thrown;
	}

	/**
	 * Set the throwable associated with this log record.
	 *
	 * @param thrown
	 *            a throwable (may be {@code null})
	 */
	public void setThrown(@Nullable Throwable thrown) {
		this.thrown = thrown;
	}

}
