/*
 * Java
 *
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.drawing.ShapePainter;
import ej.drawing.ShapePainter.Cap;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BufferedVectorImage;
import ej.microvg.Matrix;
import ej.microvg.VectorGraphicsPainter;

/**
 * Tests the drawing of MicroUI shapes into a BufferedVectorImage.
 *
 * If a drawing is not implemented on a BufferedVectorImage, the test is automatically ignored thanks to the flag
 * {@link GraphicsContext#DRAWING_LOG_NOT_IMPLEMENTED}.
 *
 * This test requires to use the GPU to draw the simple aliased shapes (otherwise the difference between software
 * algorithms and BufferedVectorImlage drawings are too important).
 */
@SuppressWarnings("nls")
public class TestBufferedVectorImageMicroUIShapes {

	interface Drawer {

		/**
		 * Draws something in graphics context in the specified rectangle.
		 *
		 * @param g
		 *            the destination
		 * @param x
		 *            the rectangle top left X-coordinate
		 * @param y
		 *            the rectangle top left Y-coordinate
		 * @param w
		 *            the rectangle's width
		 * @param h
		 *            the rectangle's height
		 */
		void draw(GraphicsContext g, int x, int y, int w, int h);

	}

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.reset();
		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	}

	/**
	 * Tests some {@link Painter} algorithms: draw a line
	 */
	@Test
	public static void testUIPainterDrawLine() {
		testDrawer("DrawLine", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawLine(g, x, y, x + w, y + h);
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: draw an horizontal line
	 */
	@Test
	public static void testUIPainterDrawHorizontalLine() {
		testDrawer("DrawHorizontalLine", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawHorizontalLine(g, x, y + h / 2, w);
			}
		}, 20, 20, 0);
	}

	/**
	 * Tests some {@link Painter} algorithms: draw a vertical line
	 */
	@Test
	public static void testUIPainterDrawVerticalLine() {
		testDrawer("DrawVerticalLine", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawVerticalLine(g, x + w / 2, y, h);
			}
		}, 20, 20, 0);
	}

	/**
	 * Tests some {@link Painter} algorithms: fill a rectangle
	 */
	@Test
	public static void testUIPainterFillRectangle() {
		testDrawer("FillRectangle", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.fillRectangle(g, x, y, w, h);
			}
		}, 20, 20, 0);
	}

	/**
	 * Tests some {@link Painter} algorithms: draw a rounded rectangle
	 */
	@Test
	public static void testUIPainterDrawRoundedRectangle() {
		testDrawer("DrawRoundedRectangle", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawRoundedRectangle(g, x, y, w, h, 10, 9);
			}
		}, 40, 40, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: fill a rounded rectangle
	 */
	@Test
	public static void testUIPainterFillRoundedRectangle() {
		testDrawer("FillRoundedRectangle", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.fillRoundedRectangle(g, x, y, w, h, 10, 9);
			}
		}, 40, 40, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: draw a circle arc
	 */
	@Test
	public static void testUIPainterDrawCircleArc() {
		testDrawer("DrawCircleArc", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawCircleArc(g, x, y, Math.min(w, h), 45, 223);
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: fill a circle arc
	 */
	@Test
	public static void testUIPainterFillCircleArc() {
		testDrawer("FillCircleArc", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.fillCircleArc(g, x, y, Math.min(w, h), 68, 223);
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: draw an ellipse arc
	 */
	@Test
	public static void testUIPainterDrawEllipseArc() {
		testDrawer("DrawEllipseArc", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawEllipseArc(g, x, y, w, h, 45, 223);
			}
		}, 20, 30, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: fill an ellipse arc
	 */
	@Test
	public static void testUIPainterFillEllipseArc() {
		testDrawer("FillEllipseArc", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.fillEllipseArc(g, x, y, w, h, 68, 223);
			}
		}, 30, 20, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: draw a circle
	 */
	@Test
	public static void testUIPainterDrawCircle() {
		testDrawer("DrawCircle", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawCircle(g, x, y, Math.min(w, h));
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: fill a circle
	 */
	@Test
	public static void testUIPainterFillCircle() {
		testDrawer("FillCircle", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.fillCircle(g, x, y, Math.min(w, h));
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: draw an ellipse
	 */
	@Test
	public static void testUIPainterDrawEllipse() {
		testDrawer("DrawEllipse", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.drawEllipse(g, x, y, w, h);
			}
		}, 20, 30, 0.1f);
	}

	/**
	 * Tests some {@link Painter} algorithms: fill an ellipse
	 */
	@Test
	public static void testUIPainterFillEllipse() {
		testDrawer("FillEllipse", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				Painter.fillEllipse(g, x, y, w, h);
			}
		}, 30, 20, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: draw ThickFadedPoint
	 */
	@Test
	public static void testUIPainterDrawThickFadedPoint() {
		testDrawer("DrawThickFadedPoint", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickFadedPoint(g, x + w / 2, y + h / 2, w / 2, 1);
			}
		}, 40, 40, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: drawThickFadedLine
	 */
	@Test
	public static void testUIPainterDrawThickFadedLine() {
		testDrawer("DrawThickFadedLine", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickFadedLine(g, x + w - 5, y + 5, x + 5, y + h - 5, 5, 1, Cap.NONE, Cap.ROUNDED);
			}
		}, 40, 50, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: drawThickFadedCircle
	 */
	@Test
	public static void testUIPainterDrawThickFadedCircle() {
		testDrawer("DrawThickFadedCircle", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickFadedCircle(g, x + 2, y + 2, 15, 3, 1);
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: draw a circle arc
	 */
	@Test
	public static void testUIPainterDrawThickFadedCircleArc() {
		testDrawer("DrawThickFadedCircleArc", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickFadedCircleArc(g, x + 2, y + 2, 35, 45, 223, 3, 1, Cap.ROUNDED, Cap.NONE);
			}
		}, 50, 50, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: draw an ellipse
	 */
	@Test
	public static void testUIPainterDrawThickFadedEllipse() {
		testDrawer("DrawThickFadedEllipse", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickFadedEllipse(g, x + 2, y + 2, w - 5, h - 5, 3, 1);
			}
		}, 40, 50, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: drawThickLine
	 */
	@Test
	public static void testUIPainterDrawThickLine() {
		testDrawer("DrawThickLine", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickLine(g, x + 2, y + 2, x + w - 5, y + h - 5, 3);
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: drawThickCircle
	 */
	@Test
	public static void testUIPainterDrawThickCircle() {
		testDrawer("DrawThickCircle", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickCircle(g, x + 2, y + 2, 15, 3);
			}
		}, 20, 20, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: draw an ellipse
	 */
	@Test
	public static void testUIPainterDrawThickEllipse() {
		testDrawer("DrawThickEllipse", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickEllipse(g, x + 2, y + 2, w - 5, h - 5, 3);
			}
		}, 40, 50, 0.1f);
	}

	/**
	 * Tests some {@link ShapePainter} algorithms: draw a circle arc
	 */
	@Test
	public static void testUIPainterDrawThickCircleArc() {
		testDrawer("DrawThickCircleArc", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				ShapePainter.drawThickCircleArc(g, x + 2, y + 2, 15, 45, 223, 3);
			}
		}, 20, 20, 0.1f);
	}

	private static void testDrawer(String label, Drawer drawer, int w, int h, float tolerance) {
		try (BufferedVectorImage image = new BufferedVectorImage(w + 2, h + 3)) {

			Display display = Display.getDisplay();
			GraphicsContext g = display.getGraphicsContext();
			GraphicsContext ig = image.getGraphicsContext();

			// draw expected area on the left and virtual area on the right
			int padding = h;
			int x = g.getWidth() / 2 - w - padding;
			int y = g.getHeight() / 2;
			final int color = Colors.RED; // useful to test BGR pixel format

			g.setColor(color);
			drawer.draw(g, x, y, w, h);
			ig.setColor(color);
			drawer.draw(ig, 1, 2, w, h); // (1,2) to test the translation

			if (GraphicsContext.DRAWING_LOG_NOT_IMPLEMENTED != (ig.getAndClearDrawingLogFlags()
					& GraphicsContext.DRAWING_LOG_NOT_IMPLEMENTED)) {
				// draw virtual area on destination
				g.reset();

				Matrix m = new Matrix();
				m.setTranslate(x + w + padding - 1, y - 2);
				VectorGraphicsPainter.drawImage(g, image, m);

				// visualize the result
				display.flush();

				// compare areas
				TestUtilities.compareAreas(label, x, y, x + w + padding, y, w, h, tolerance);
			}
			// else: drawing cannot be performed in a BVI: drop the test
		}
	}
}
