/*
 * Copyright 2014-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.container;

import ej.mwt.Container;
import ej.mwt.Widget;
import ej.mwt.util.Size;

/**
 * Lays out any number of children horizontally or vertically.
 * <p>
 * All the children are laid out on the same horizontal or vertical line.
 * <p>
 * In a horizontal list, all children will have the height of the available space, and their width will be determined
 * proportionally. In a vertical list, all children will have the width of the available space, and their height will be
 * determined proportionally.
 * <p>
 * Horizontal:
 * <p>
 * <img src="doc-files/list_horizontal.png" alt="Horizontal list.">
 * <p>
 * Vertical:
 * <p>
 * <img src="doc-files/list_vertical.png" alt="Vertical list.">
 */
public class List extends Container {

	private boolean orientation;

	private int totalLength;

	/**
	 * Creates a list specifying its orientation.
	 *
	 * @param orientation
	 *            the orientation of the list (see {@link LayoutOrientation}).
	 */
	public List(boolean orientation) {
		this.orientation = orientation;
	}

	/**
	 * Sets the orientation of this list.
	 *
	 * @param orientation
	 *            the orientation to set (see {@link LayoutOrientation}).
	 */
	public void setOrientation(boolean orientation) {
		this.orientation = orientation;
	}

	/**
	 * Gets the orientation of this list.
	 *
	 * @return the orientation of this list (see {@link LayoutOrientation}).
	 */
	public boolean getOrientation() {
		return this.orientation;
	}

	@Override
	public void addChild(Widget child) {
		super.addChild(child);
	}

	@Override
	public void removeChild(Widget child) {
		super.removeChild(child);
	}

	@Override
	public void insertChild(Widget child, int index) {
		super.insertChild(child, index);
	}

	@Override
	public void replaceChild(int index, Widget child) {
		super.replaceChild(index, child);
	}

	@Override
	public void removeAllChildren() {
		super.removeAllChildren();
	}

	@Override
	protected void computeContentOptimalSize(Size size) {
		// perform quick optimization if there is no child
		int childrenCount = getChildrenCount();
		if (childrenCount == 0) {
			size.setSize(0, 0);
			return;
		}

		// compute size hint for children
		boolean isHorizontal = (this.orientation == LayoutOrientation.HORIZONTAL);
		int childWidthHint = size.getWidth();
		int childHeightHint = size.getHeight();

		// compute total children size
		int totalWidth = 0;
		int totalHeight = 0;

		for (Widget widget : getChildren()) {
			assert (widget != null);

			// compute child optimal size
			computeChildOptimalSize(widget, childWidthHint, childHeightHint);

			// update total children size
			if (isHorizontal) {
				totalWidth += widget.getWidth();
				totalHeight = Math.max(totalHeight, widget.getHeight());
			} else {
				totalWidth = Math.max(totalWidth, widget.getWidth());
				totalHeight += widget.getHeight();
			}
		}

		// save total children size for layout
		this.totalLength = (isHorizontal ? totalWidth : totalHeight);

		// set container optimal size
		size.setSize(totalWidth, totalHeight);
	}

	@Override
	protected void layOutChildren(int contentWidth, int contentHeight) {
		int totalLength = this.totalLength;
		if (totalLength == 0) {
			return;
		}

		boolean isHorizontal = (this.orientation == LayoutOrientation.HORIZONTAL);
		int contentLength = isHorizontal ? contentWidth : contentHeight;

		int widgetPosition = 0;
		for (Widget child : getChildren()) {
			assert (child != null);

			// compute widget start and end
			int widgetLength = (isHorizontal ? child.getWidth() : child.getHeight());
			int widgetStart = widgetPosition * contentLength / totalLength;
			int widgetEnd = (widgetPosition + widgetLength) * contentLength / totalLength;

			// lay out child
			if (isHorizontal) {
				layOutChild(child, widgetStart, 0, widgetEnd - widgetStart, contentHeight);
			} else {
				layOutChild(child, 0, widgetStart, contentWidth, widgetEnd - widgetStart);
			}

			widgetPosition += widgetLength;
		}
	}
}
