/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector;

import ej.annotation.Nullable;
import ej.mwt.Container;
import ej.mwt.Widget;

/**
 * A first child selector selects by checking if a widget is the first child of its parent.
 * <p>
 * Equivalent to <code>:first-child</code> selector in CSS. Its specificity is (0,0,1,0).
 *
 * @see Widget#getParent()
 * @see SelectorHelper
 */
public class FirstChildSelector implements Selector {

	/**
	 * First child selector singleton to avoid creating several ones.
	 */
	public static final FirstChildSelector FIRST_CHILD_SELECTOR = new FirstChildSelector();

	/**
	 * Creates a first child selector.
	 */
	private FirstChildSelector() {
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		Container parent = widget.getParent();

		return parent != null && parent.getChild(0) == widget;
	}

	@Override
	public int getSpecificity() {
		return SelectorHelper.getSpecificity(0, 0, 1, 0);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		return (obj == this);
	}

	@Override
	public int hashCode() {
		return 0;
	}
}
