/*
 * Java
 *
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.image;

import java.awt.Color;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import ej.microvg.LLVGConstants;
import ej.microvg.image.pathdata.PathData;
import ej.microvg.image.pathdata.TwoArraysPathData;
import ej.microvg.paint.LinearGradient;

/**
 * The Nema engine.
 */
public class NemaImageDecoder implements ImageDecoder {

	/**
	 * The path commands.
	 */
	private static final byte CMD_CLOSE = 0x00;
	private static final byte CMD_MOVE = 0x01;
	private static final byte CMD_MOVE_REL = 0x11;
	private static final byte CMD_LINE = 0x02;
	private static final byte CMD_LINE_REL = 0x12;
	private static final byte CMD_QUAD = 0x05;
	private static final byte CMD_QUAD_REL = 0x15;
	private static final byte CMD_CUBIC = 0x06;
	private static final byte CMD_CUBIC_REL = 0x16;

	/**
	 * The fill rules.
	 */
	@SuppressWarnings("unused")
	private static final int FILL_NON_ZERO = 1;
	private static final int FILL_EVEN_ODD = 2;

	@Override
	public PathData decodePath(ByteBuffer is) {
		// skip the bounding box (useless for AWT)
		is.getFloat(); // bounding box 0
		is.getFloat(); // bounding box 1
		is.getFloat(); // bounding box 2
		is.getFloat(); // bounding box 3

		// retrieve the both arrays: commands and attributes
		int nbCommands = is.getChar();
		int nbParams = is.getChar();
		ByteBuffer params = ByteBuffer.wrap(is.array(), is.position(), nbParams * TwoArraysPathData.PARAM_SIZE)
				.order(ByteOrder.LITTLE_ENDIAN);
		ByteBuffer commands = ByteBuffer.wrap(is.array(), is.position() + nbParams * TwoArraysPathData.PARAM_SIZE,
				nbCommands * TwoArraysPathData.COMMAND_SIZE).order(ByteOrder.LITTLE_ENDIAN);

		return new TwoArraysPathData(commands, params);
	}

	@Override
	public LinearGradient decodeGradient(ByteBuffer is) {
		float x0 = is.getFloat();
		float y0 = is.getFloat();
		float x1 = is.getFloat();
		float y1 = is.getFloat();

		// read count
		int count = is.getInt();

		// read positions
		float[] positions = new float[count];
		for (int i = 0; i < count; i++) {
			float pos = is.getFloat();
			positions[i] = pos;
		}

		// read colors
		Color[] colors = new Color[count];
		for (int i = 0; i < count; i++) {
			float red = is.getFloat() / 255;
			float green = is.getFloat() / 255;
			float blue = is.getFloat() / 255;
			float alpha = is.getFloat() / 255;
			colors[i] = new Color(red, green, blue, alpha);
		}

		return new LinearGradient(colors, positions, x0, y0, x1, y1);
	}

	@Override
	public int decodeCommand(int encodedCommand) {
		switch (encodedCommand) {
		default: // unknown -> close (should not occur)
		case CMD_CLOSE:
			return LLVGConstants.COMMAND_CLOSE;
		case CMD_MOVE:
			return LLVGConstants.COMMAND_MOVE;
		case CMD_MOVE_REL:
			return LLVGConstants.COMMAND_MOVE_REL;
		case CMD_LINE:
			return LLVGConstants.COMMAND_LINE;
		case CMD_LINE_REL:
			return LLVGConstants.COMMAND_LINE_REL;
		case CMD_QUAD:
			return LLVGConstants.COMMAND_QUAD;
		case CMD_QUAD_REL:
			return LLVGConstants.COMMAND_QUAD_REL;
		case CMD_CUBIC:
			return LLVGConstants.COMMAND_CUBIC;
		case CMD_CUBIC_REL:
			return LLVGConstants.COMMAND_CUBIC_REL;
		}
	}

	@Override
	public int decodeFillRule(int rule) {
		return (FILL_EVEN_ODD == rule) ? LLVGConstants.FILLTYPE_EVEN_ODD : LLVGConstants.FILLTYPE_WINDING;
	}

	/**
	 * Encoded color is A-B-G-R
	 */
	@Override
	public int decodeColor(int color) {
		int blue = (color >> 16) & 0xff;
		int red = (color & 0xff) << 16;
		color &= 0xff00ff00;
		color |= red;
		color |= blue;
		return color;
	}

}
