/*
 * Java
 *
 * Copyright 2023-2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.event;

import java.lang.Thread.UncaughtExceptionHandler;

import ej.annotation.Nullable;
import ej.basictool.ThreadUtils;
import ej.bon.Constants;
import ej.basictool.ArrayTools;

/**
 * EventQueue is an asynchronous communication interface between the native world and the Java world based on events. It
 * allows users to send events from the native world to the Java world.
 * <p>
 * A FIFO mechanism is implemented on the native side and is system specific. The user can offer events to this FIFO.
 * <p>
 * Event notifications are handled using event listeners (Observer design pattern). The application code has to register
 * {@link EventQueueListener} to be notified when new events are coming. To do so, the user can call
 * {@link EventQueue#registerListener(EventQueueListener)}.
 * <p>
 * Then the pump automatically retrieves new events pushed in the FIFO and notifies the application via the
 * {@link EventQueueListener}.
 * <p>
 * EventQueue runs on a dedicated Java thread to forward and process events. Application event listener's calls are done
 * in the context of this thread. This thread is started when method {@link EventQueue#getInstance()} is called.
 * <p>
 * Events reading operations are done using the SNI mechanism. The thread is suspended if the events FIFO is empty. The
 * thread resume is done by the native part when a new event is sent if the events FIFO was previously empty.
 * <p>
 * <b> Event format </b>
 * <p>
 * An event is composed of a type and some data. The type identifies the listener that will handle the event. The data
 * is application specific and passed to the listener. The items stored in the FIFO buffer are integers (4 bytes). There
 * are two kinds of events that can be sent over the Event Queue:
 * <ul>
 * <li>Standard event: an event with data that fits on 24 bits. The event is stored in the FIFO as a single 4 bytes
 * item.</li>
 * <li>Extended event: an event with data that does not fit on 24 bits. The event is stored in the FIFO as multiple 4
 * bytes items.</li>
 * </ul>
 *
 * <pre>
 * +--------------+----------+---------------------------------------------------+
 * | Extended (1) | Type (7) | Data (if Extended==0), Length (if Extended==1) (24) |
 * +--------------+----------+---------------------------------------------------+
 * ...
 * +-----------------------------------------------------------------------------+
 * |                  Extended Data for extended events (32)                     | (x integers)
 * +-----------------------------------------------------------------------------+
 * </pre>
 *
 * Format explanation:
 *
 * <ul>
 * <li>Extended (1 bit): event kind flag (0 for non-extended event, 1 for extended event).</li>
 * <li>Type (7 bits): event type, which allows to find the corresponding event listener.</li>
 * <li>Length (24 bits): length of the data in bytes (for extended events only).</li>
 * <li>Data (24 bits): standard event data (for standard events only).</li>
 * <li>Extended data (`Length` bytes): extended event data (for extended events only).</li>
 * </ul>
 *
 * <p>
 *
 * For more information, please have a look at our
 * <a href= "https://docs.microej.com/en/latest/ApplicationDeveloperGuide/eventQueue.html">Event Queue
 * documentation</a>.
 *
 */
public class EventQueue { // Singleton intentionally used for global event dispatch coordination. //NOSONAR: The
							// Singleton design pattern should be used with care
	private static final String THREAD_NAME_CONSTANT = "event.thread.name";
	private static final String THREAD_PRIORITY_CONSTANT = "event.thread.priority";
	private static final String THREAD_DAEMON_CONSTANT = "event.thread.daemon";

	private static final int MAX_TYPE_ID = 127;
	private static final int TYPE_OFFSET = 24;
	private static final int TYPE_MASK = 0x7F;
	private static final int INT_24_BITS = 0xFFFFFF;
	private static final int EXTENDED_EVENT_MASK = 0x7FFFFFFF;

	@Nullable
	private static EventQueue eventQueue = null;

	private EventQueueListener[] eventListeners;
	private int nextType;

	private EventQueueListener defaultListener;
	private UncaughtExceptionHandler uncaughtExceptionHandler;

	/**
	 * Creates a pump.
	 */
	private EventQueue() {
		this.eventListeners = new EventQueueListener[0];
		this.defaultListener = new NullEventQueueListener();
	}

	/**
	 * Sets a handler for the exceptions that occur during event reading or execution.
	 *
	 * @param uncaughtExceptionHandler
	 *            the uncaught exception handler to set.
	 */
	@Nullable
	public void setUncaughtExceptionHandler(UncaughtExceptionHandler uncaughtExceptionHandler) {
		this.uncaughtExceptionHandler = uncaughtExceptionHandler;
	}

	/**
	 * Sets the default {@code listener}. It will receive all the events whose type is not handled by any registered
	 * listener.
	 * 
	 * @param listener
	 *            the listener to set.
	 * @throws IllegalArgumentException
	 *             if {@code listener} is {@code null}.
	 */
	public void setDefaultListener(EventQueueListener listener) {
		if (listener == null) {
			throw new IllegalArgumentException(String.valueOf(EventQueueException.DEFAULT_LISTENER_NULL));
		}
		this.defaultListener = listener;
	}

	/**
	 * Registers a {@code listener} that will receive the events of a {@code type}.
	 * <p>
	 * The same listener can be registered for several types.
	 *
	 * @param listener
	 *            the listener to register.
	 * @return the event type that will be listened by the listener.
	 * @throws IllegalStateException
	 *             if there is no type left for a new listener.
	 */
	public synchronized int registerListener(EventQueueListener listener) throws IllegalStateException {

		// Check the type value
		if (this.nextType > MAX_TYPE_ID) {
			throw new IllegalStateException(String.valueOf(EventQueueException.MAXIMUM_REGISTERED_EVENTS_REACHED));
		}
		int type = nextType;
		// Register the EventListener
		this.eventListeners = ArrayTools.add(this.eventListeners, listener);
		// increment type
		nextType++;
		return type;
	}

	/**
	 * Unregisters listener for the provided event type
	 * 
	 * @param type
	 *            type of listener to be unregistered
	 * @throws IllegalArgumentException
	 *             if the given {@code type} does not correspond to any registered listener.
	 */
	public synchronized void unregisterListener(int type) throws IllegalArgumentException {
		EventQueueListener[] eventQueueListeners = this.eventListeners;
		if (type < 0 || type >= eventListeners.length || eventQueueListeners[type] == null) {
			throw new IllegalArgumentException(String.valueOf(EventQueueException.NO_LISTENER_REGISTERED));
		}
		eventQueueListeners[type] = null;
	}

	/**
	 * Inserts an event to the FIFO.
	 * <p>
	 * The {@code data} must not exceed 24 bits. Otherwise, use {@link #offerExtendedEvent(int, byte[])}.
	 *
	 * @param type
	 *            the type of the event.
	 * @param data
	 *            the data of the event.
	 * @throws IllegalStateException
	 *             if the FIFO is full.
	 * @throws IllegalArgumentException
	 *             if the given {@code type} is lower than <code>0</code> or greater than <code>127</code>.
	 * @throws IllegalArgumentException
	 *             if the {@code data} exceeds 24 bits.
	 */
	public void offerEvent(int type, int data) {
		// Check the type value
		if (type < 0 || type >= MAX_TYPE_ID) {
			throw new IllegalArgumentException(String.valueOf(EventQueueException.INVALID_TYPE_MUST_BE_BETWEEN_0_AND_127));
		}
		if ((data & INT_24_BITS) != data) {
			throw new IllegalArgumentException(String.valueOf(EventQueueException.DATA_EXCEED_24_BIT));
		}
		if (!EventQueueNatives.offerEvent(type, data)) {
			throw new IllegalStateException(String.valueOf(EventQueueException.FIFO_IS_FULL));
		}
	}

	/**
	 * Inserts an extended event to the FIFO.
	 * <p>
	 * When the data exceeds 24 bits, it can be split into several integers. These integers are pushed consecutively to
	 * the FIFO. The listener will receive all these integers at once and will be able to rebuild a complex data.
	 *
	 * @param type
	 *            the type of the events.
	 * @param data
	 *            the data of the events.
	 * @throws IllegalStateException
	 *             if the FIFO is full.
	 * @throws IllegalArgumentException
	 *             if the given {@code type} is lower than <code>0</code> or greater than <code>127</code>.
	 * @throws IllegalArgumentException
	 *             if the length of {@code data} buffer exceeds 24 bits.
	 */
	public void offerExtendedEvent(int type, byte[] data) {
		// Check the type value
		if (type < 0 || type >= MAX_TYPE_ID) {
			throw new IllegalArgumentException(String.valueOf(EventQueueException.INVALID_TYPE_MUST_BE_BETWEEN_0_AND_127));
		}
		if ((data.length & INT_24_BITS) != data.length) {
			throw new IllegalArgumentException(String.valueOf(EventQueueException.DATA_EXCEED_24_BIT));
		}
		if (!EventQueueNatives.offerExtendedEvent(type, data, data.length)) {
			throw new IllegalStateException(String.valueOf(EventQueueException.FIFO_IS_FULL));
		}
	}

	/**
	 * Gets an instance of a started EventQueue.
	 *
	 * @return the instance of the queue.
	 */
	public static synchronized EventQueue getInstance() {
		EventQueue instance = EventQueue.eventQueue;
		if (instance == null) {
			instance = new EventQueue();
			EventQueue.eventQueue = instance;
			instance.start();
		}
		return instance;
	}

	/**
	 * Runs the pump on the current thread.
	 * <p>
	 * The pump loops on:
	 * <ol>
	 * <li>Read an event from the FIFO (pop),</li>
	 * <li>Extract its type and resolve the corresponding listener,</li>
	 * <li>If type is invalid call the defaultListener.</li>
	 * <li>If the event is simple, delegate its processing to {@link EventQueueListener#handleEvent(int, int)},</li>
	 * <li>Otherwise, handle the extended event directly by calling
	 * {@link #handleExtendedEvent(int, EventQueueListener, int)}.</li>
	 * </ol>
	 */
	private void runThread() {
		while (true) { // NOSONAR the pump never stops.
			try {
				int event = EventQueueNatives.waitEvent();
				// Process the data of the event. 0 (1) | Type (7) | Data (24)
				int eventType = getEventType(event);
				// Get the listener corresponding to the type. If it exists, handle the event.
				// If not use the default listener.
				EventQueueListener eventListener;
				synchronized (this) {
					EventQueueListener[] queueListeners = this.eventListeners;

					if (eventType >= 0 && eventType < queueListeners.length) {
						eventListener = queueListeners[eventType];
						if (eventListener == null) {
							eventListener = defaultListener;
						}
					} else {
						eventListener = defaultListener;
					}

					if (event >= 0) {
						eventListener.handleEvent(eventType, getEventData(event));
					} else {
						handleExtendedEvent(eventType, eventListener, event & EXTENDED_EVENT_MASK);
					}

				}
			} catch (Throwable t) { // NOSONAR Pump thread Errors should not be handled by the JVM.
				// NOSONAR Required to prevent application code from breaking the EventQueue.
				// The Throwable is dispatched to a user-defined uncaught exception handler.
				// Applications can then decide how to handle critical errors.
				handleException(t);
			}
		}
	}

	/**
	 * Handles any uncaught exception/error occurring during event processing.
	 * 
	 * @param t
	 *            the thrown {@link Throwable} to handle.
	 */
	private void handleException(Throwable t) {
		UncaughtExceptionHandler handler = this.uncaughtExceptionHandler;
		if (handler == null) {
			ThreadUtils.handleUncaughtException(t);
		} else {
			handler.uncaughtException(Thread.currentThread(), t);
		}
	}

	/**
	 * Handles an extended event. This method is called by {@link #runThread()} when an extended event is received. The
	 * event type and listener are already resolved before this call. It reads the extended data and dispatches it to *
	 * {@link EventQueueListener#handleExtendedEvent(int, EventDataReader)}.
	 * 
	 * @param eventType
	 *            the given event type
	 * @param eventListener
	 *            the eventListener of the upcoming event.
	 * @param event
	 *            the event to handle.
	 */
	private void handleExtendedEvent(int eventType, EventQueueListener eventListener, int event) {
		// Process the data of the event. 1 (1) | Type (7) | Data Length (24)
		int dataLength = getEventData(event);

		// Create the Data reader related to the event.
		EventDataReader eventQueueDataReader = new EventQueueDataReader();

		// Start to handle the extended data.
		EventQueueNatives.startReadExtendedData(dataLength);
		if (eventListener != null) {
			eventListener.handleExtendedEvent(eventType, eventQueueDataReader);
		} else {
			this.defaultListener.handleExtendedEvent(eventType, eventQueueDataReader);
		}
		// Check that the extended data has been read entirely.
		EventQueueNatives.endReadExtendedData();
	}

	/**
	 * Executes this {@link EventQueue} in a new thread.
	 */
	private void start() {
		EventQueueNatives.initialize();
		Thread t = new Thread(new Runnable() { // NOSONAR : Make this anonymous inner class a lambda

			@Override
			public void run() {
				runThread();

			}
		}, Constants.getString(THREAD_NAME_CONSTANT));
		if (Constants.getBoolean(THREAD_DAEMON_CONSTANT)) {
			t.setDaemon(Constants.getBoolean(THREAD_DAEMON_CONSTANT));
		}
		t.setPriority(Constants.getInt(THREAD_PRIORITY_CONSTANT));
		t.start();

	}

	/**
	 * Gets the type the given event.
	 *
	 * @param event
	 *            the event.
	 * @return the type of the event.
	 */
	private static int getEventType(int event) {
		return ((event >> TYPE_OFFSET) & TYPE_MASK);
	}

	/**
	 * Gets the data of an event.
	 *
	 * @param event
	 *            the event.
	 * @return the data of the event.
	 */
	private static int getEventData(int event) {
		return event & INT_24_BITS;
	}

}
