/*
 * Copyright 2022 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.regexp;

/**
 * Encapsulates char[] as CharacterIterator
 *
 * @author <a href="mailto:ales.novak@netbeans.com">Ales Novak</a>
 * @version CVS $Id: CharacterArrayCharacterIterator.java 518156 2007-03-14 14:31:26Z vgritsenko $
 */
public final class CharacterArrayCharacterIterator implements CharacterIterator {
	/** encapsulated */
	private final char[] src;
	/** offset in the char array */
	private final int off;
	/** used portion of the array */
	private final int len;

	/**
	 * @param src
	 *            - encapsulated String
	 */
	public CharacterArrayCharacterIterator(char[] src, int off, int len) {
		this.src = src;
		this.off = off;
		this.len = len;
	}

	/** @return a substring */
	@Override
	public String substring(int beginIndex, int endIndex) {
		if (endIndex > this.len) {
			throw new IndexOutOfBoundsException("endIndex=" + endIndex + "; sequence size=" + this.len);
		}
		if (beginIndex < 0 || beginIndex > endIndex) {
			throw new IndexOutOfBoundsException("beginIndex=" + beginIndex + "; endIndex=" + endIndex);
		}
		return new String(this.src, this.off + beginIndex, endIndex - beginIndex);
	}

	/** @return a substring */
	@Override
	public String substring(int beginIndex) {
		return substring(beginIndex, this.len);
	}

	/** @return a character at the specified position. */
	@Override
	public char charAt(int pos) {
		return this.src[this.off + pos];
	}

	/** @return <tt>true</tt> iff if the specified index is after the end of the character stream */
	@Override
	public boolean isEnd(int pos) {
		return (pos >= this.len);
	}
}
