/*
 * Copyright 2014-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.websocket;

import java.io.IOException;
import java.net.Socket;
import java.security.NoSuchAlgorithmException;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;

import ej.util.message.Level;

/**
 * Representation for the physical websocket connection that support SSL Context.
 *
 * {@link WebSocket}.
 *
 */
public class WebSocketSecure extends WebSocket {

	// SSL context
	private final SSLContext context;

	/**
	 * Create a new {@link WebSocket} instance. It doesn't connect to the remote server; user has to call
	 * {@link #connect()} to do so.
	 *
	 * @param uri
	 *            URI to connect to
	 * @param endpoint
	 *            the endpoint that will handle events
	 * @throws IllegalArgumentException
	 *             if 'endpoint' or 'uri' is null
	 * @throws NoSuchAlgorithmException
	 *             if the default SSLContext is not supported.
	 * @see SSLContext#getDefault()
	 */
	public WebSocketSecure(WebSocketURI uri, Endpoint endpoint)
			throws IllegalArgumentException, NoSuchAlgorithmException {
		this(uri, endpoint, null);
	}

	/**
	 * Create a new {@link WebSocket} instance with SSL support. It doesn't connect to the remote server; user has to
	 * call {@link #connect()} to do so.
	 *
	 * @param uri
	 *            URI to connect to
	 * @param endpoint
	 *            the endpoint that will handle events
	 * @param context
	 *            the SSL context to use, if <code>null</code> uses the default SSLContext.
	 * @throws IllegalArgumentException
	 *             if 'endpoint' or 'uri' is null, or if 'uri' is not secured
	 * @throws NoSuchAlgorithmException
	 *             if context is null and does not
	 * @see SSLContext#getDefault()
	 */
	public WebSocketSecure(WebSocketURI uri, Endpoint endpoint, SSLContext context)
			throws IllegalArgumentException, NoSuchAlgorithmException {
		super(uri, endpoint);

		if (context == null && uri.isSecure()) {
			context = SSLContext.getDefault();
		}
		this.context = context;
	}

	@Override
	protected void setupSocket() throws IOException {
		WebSocketURI uri = getURI();
		if (context != null && uri.isSecure()) {
			Socket socket = null;
			try {
				socket = context.getSocketFactory().createSocket(uri.getHost(), uri.getPort());
				if (socket instanceof SSLSocket) {
					((SSLSocket) socket).startHandshake();
				}
			} catch (IOException e) {
				if (socket != null) {
					try {
						socket.close();
					} catch (IOException e2) {
						Messages.LOGGER.log(Level.INFO, Messages.CATEGORY, Messages.CLOSE_TCP, e2, this);
					}
				}
				throw new IOException(e);
			}
			setSocket(socket);
		} else {
			super.setupSocket();
		}
	}


}
